/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQBATCHEVOLUTIONMODEL_H_
#define _SQBATCHEVOLUTIONMODEL_H_

#include "SQDef.h"

/** @file SQBatchEvolutionModel.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif


#include "SQErrorCodes.h"
#include "SQModel.h"

  /** @cond SQPPlus */
   
   /**
   An enum defining the different limits in a batch evolution model
   */
   typedef enum SQ_LimitEnumType
   {
      SQ_ScoreLimit,        /**< Score limit. */
      SQ_DModXLimit,        /**< DModX limit. */
      SQ_T2RangeLimit       /**< T2Range limit. */
   } SQ_LimitType;          

   /**
   *  An inquiry whether the model is a Batch Evolution Model.
   *
   *  @param[in]  pModel            The model to use.
   *  @param[out] bIsBEM            Is set to SQ_True if the model is a BEM.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsBatchEvolutionModel(SQ_Model pModel, SQ_Bool* bIsBEM);

   /**
   *  Retrieves the number of batches in the model.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[out] piNrOfBatches     The number of batches in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelNumberOfBatches(SQ_Model pModel, int *piNrOfBatches);

   /**
   *  Retrieves the name of a batch in a model.
   *
   *  @param[in]     pModel            The Batch Evolution Model to use.
   *  @param[in]     iBatchIx          The batch index we want the name for.
   *  @param[in,out] szBatchName       The name of the batch. The user is responsible to allocate and deallocate the buffer, UTF-8 encoded.
   *  @param[in]     iBufferLength     The length of the buffer.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelBatchName(SQ_Model pModel, int iBatchIx, char *szBatchName, int iBufferLength);

   /**
   *  Retrieves the name of the phase connected with the model.
   *
   *  @param[in]     pModel            The Batch Evolution Model to use.
   *  @param[in,out] szPhaseName       The name of the phase. The user is responsible to allocate and deallocate the buffer, UTF-8 encoded.
   *  @param[in]     iBufferLength     The length of the buffer.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPhaseName(SQ_Model pModel, char *szPhaseName, int iBufferLength);

   /**
   *  A request to get the batch evolution model limits.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  eLimitType        The type of the requested limit (Score, DModX or T2Range)
   *  @param[in]  iCompFrom         The "start" component, if only one component it 
   *                                should be given here.
   *  @param[in]  iCompTo           The "end" component.
   *  @param[out] pVectorData       A pointer to the Limits results.
   *                                Number of rows in matrix = 2 (first is Mean and second StdDev) 
   *                                Number of columns in matrix = The aligned Time/maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetBEMLimits(SQ_Model pModel, SQ_LimitType eLimitType, int iCompFrom, int iCompTo, SQ_VectorData* pVectorData);
   
   /**
   *  Aligned DModX contribution.
   *  Contribution is used to understand how an observation differs from the others in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  fMaturity         The aligned maturity to get the contribution for.
   *  @param[in]  iBatchIx          The batch index to get the contribution for.
   *                                <1 if the average is to be used.
   *  @param[in]  szPhaseIteration  The name of the phase iteration for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned DModX contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModX(SQ_Model pModel, float fMaturity, int iBatchIx, const char *szPhaseIteration, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);
   
   /**
   *  Aligned DModX group contribution.
   *  Contribution is used to understand how an observation differs from the others in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  pMaturity         A vector with the aligned maturity to get the contribution for.
   *  @param[in]  pBatchIx          A vector with the batch index to get the contribution for.
   *                                <1 if the average is to be used. Must be the same length as the maturity vector.
   *  @param[in]  pPhaseIterations  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned DModX contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModXGroup(SQ_Model pModel, SQ_FloatVector* pMaturity, SQ_IntVector* pBatchIx, SQ_StringVector* pPhaseIterations, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Aligned DModY contribution.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  fMaturity         The aligned maturity to get the contribution for.
   *  @param[in]  iBatchIx          The batch index to get the contribution for.
   *                                <1 if the average is to be used.
   *  @param[in]  szPhaseIteration  The name of the phase iteration for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  eWeightType       The type of weight. Must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[out] pVectorData       A pointer to the aligned DModY contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = 1.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModY(SQ_Model pModel, float fMaturity, int iBatchIx, const char *szPhaseIteration, SQ_WeightType eWeightType, int iComponent, SQ_VectorData* pVectorData);

   /**
   *  Aligned DModY group contribution.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  pMaturity         A vector with the aligned maturity to get the contribution for.
   *  @param[in]  pBatchIx          A vector with the batch index to get the contribution for.
   *                                <1 if the average is to be used. Must be the same length as the maturity vector.
   *  @param[in]  pPhaseIterations  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  eWeightType       The type of weight. Must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[out] pVectorData       A pointer to the aligned DModY contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = 1.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModYGroup(SQ_Model pModel, SQ_FloatVector* pMaturity, SQ_IntVector* pBatchIx, SQ_StringVector* pPhaseIterations, SQ_WeightType eWeightType, int iComponent, SQ_VectorData* pVectorData);
   
   /**
   *  Aligned Scores single weight contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  fMaturity1        The aligned maturity
   *                                for the reference observation (from observation).
   *  @param[in]  iBatchIx1         The batch index to get the contribution for.
   *                                for the reference observation (from observation). 
   *                                <1 if the average is to be used.
   *  @param[in]  szPhaseIteration1 The name of the first phase iteration (from observation) for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx1 <1 it is also ignored and can be NULL.
   *  @param[in]  fMaturity2        The aligned maturity
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  iBatchIx2         The batch index to get the contribution for.
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  szPhaseIteration2 The name of the second phase iteration (to observation) for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx2 <1 it is also ignored and can be NULL.
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScoreSingleWeight(SQ_Model pModel, float fMaturity1, int iBatchIx1, const char *szPhaseIteration1, float fMaturity2, int iBatchIx2, const char *szPhaseIteration2, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Aligned Scores single weight group contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  pMaturity1        A vector with the aligned maturity
   *                                for the reference observation (from observation).
   *  @param[in]  pBatchIx1         A vector with the batch index to get the contribution for.
   *                                for the reference observation (from observation). 
   *                                Must be the same length as the first maturity vector.
   *                                <1 if the average is to be used.
   *  @param[in]  pPhaseIterations1 A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  pMaturity2        A vector with the aligned maturity
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  pBatchIx2         A vector with the batch index to get the contribution for.
   *                                for which the contributions are to be calculated (to observation). 
   *                                Must be the same length as the second maturity vector.
   *  @param[in]  pPhaseIterations2 A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScoreSingleWeightGroup(SQ_Model pModel, SQ_FloatVector* pMaturity1, SQ_IntVector* pBatchIx1, SQ_StringVector* pPhaseIterations1, SQ_FloatVector* pMaturity2, SQ_IntVector* pBatchIx2, SQ_StringVector* pPhaseIterations2, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Aligned Scores single weight group contribution. In batch level projects created from score variables, 
   *  the Combined Contribution matrix can be created.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  pMaturity         A vector with the aligned maturity
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  pBatchIx          A vector with the batch index to get the contribution for.
   *                                for which the contributions are to be calculated (to observation). 
   *                                Must be the same length as the second maturity vector.
   *  @param[in]  pPhaseIterations  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the combined aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScoreSingleWeightGroupCombined(SQ_Model pModel, SQ_FloatVector* pMaturity, SQ_IntVector* pBatchIx, SQ_StringVector* pPhaseIterations, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Aligned Scores multi weight contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  fMaturity1        The aligned maturity
   *                                for the reference observation (from observation).
   *  @param[in]  iBatchIx1         The batch index to get the contribution for.
   *                                for the reference observation (from observation).
   *                                <1 if the average is to be used.
   *  @param[in]  szPhaseIteration1 The name of the second phase iteration (from observation) for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx1 <1 it is also ignored and can be NULL.
   *  @param[in]  fMaturity2        The aligned maturity
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  iBatchIx2         The batch index to get the contribution for.
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  szPhaseIteration2 The name of the second phase iteration (to observation) for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx2 <1 it is also ignored and can be NULL.
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums.
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                for weight Po the component argument is the orthogonal component number
   *                                for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score multi weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScoreMultiWeight(SQ_Model pModel, float fMaturity1, int iBatchIx1, const char *szPhaseIteration1, float fMaturity2, int iBatchIx2, const char *szPhaseIteration2, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Aligned Scores multi weight contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  pMaturity1        A vector with the aligned maturity
   *                                for the reference observation (from observation).
   *  @param[in]  pBatchIx1         A vector with the batch index to get the contribution for.
   *                                for the reference observation (from observation). 
   *                                Must be the same length as the first maturity vector.
   *                                <1 if the average is to be used.
   *  @param[in]  pPhaseIterations1 A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]  pMaturity2        A vector with the aligned maturity
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  pBatchIx2         A vector with the batch index to get the contribution for.
   *                                for which the contributions are to be calculated (to observation). 
   *                                Must be the same length as the second maturity vector.
   *  @param[in]  pPhaseIterations2 A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. Phase iteration is also
   *                                ignored for batch indexes < 1, any value can be used
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums.
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *                                For OPLS models and PRange weight the component argument is ignored, all components are used.
   *                                For weight Po the component argument is the orthogonal component number and the model must be OPLS.
   *                                For P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score multi weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScoreMultiWeightGroup(SQ_Model pModel, SQ_FloatVector* pMaturity1, SQ_IntVector* pBatchIx1, SQ_StringVector* pPhaseIterations1, SQ_FloatVector* pMaturity2, SQ_IntVector* pBatchIx2, SQ_StringVector* pPhaseIterations2, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned DModX from the model.
   *
   *  @param[in]  pModel                  The Batch Evolution Model to use.
   *  @param[in]  iComponent              The component to use.
   *  @param[in]  bNormalized             If SQ_True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If SQ_False, they will be in absolute values.
   *  @param[in]  bModelingPowerWeighted  If SQ_True, the function will weight the residuals by the modeling power of the variables.
   *  @param[in]  pBatches                A list of batches that exist in the model, 
   *                                      NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData             A pointer to the aligned DModX results.
   *  @return                             Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModX(SQ_Model pModel, int iComponent, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned DModX average batch from the model.
   *
   *  @param[in]  pModel                  The Batch Evolution Model to use.
   *  @param[in]  iComponent              The component to use.
   *  @param[in]  bNormalized             If SQ_True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If SQ_False, they will be in absolute values.
   *  @param[in]  bModelingPowerWeighted  If SQ_True, the function will weight the residuals by the modeling power of the variables.
   *  @param[out] pVectorData             A pointer to the aligned DModX average batch results.
   *  @return                             Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModXAverageBatch(SQ_Model pModel, int iComponent, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData* pVectorData);
      /**
   *  Retrieves the aligned DModX stddev batch from the model.
   *
   *  @param[in]  pModel                  The Batch Evolution Model to use.
   *  @param[in]  iComponent              The component to use.
   *  @param[in]  bNormalized             If SQ_True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If SQ_False, they will be in absolute values.
   *  @param[in]  bModelingPowerWeighted  If SQ_True, the function will weight the residuals by the modeling power of the variables.
   *  @param[out] pVectorData             A pointer to the aligned DModX StdDev results.
   *  @return                             Returns SQ_E_OK if success or an error code
   */   
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModXStdDevBatch(SQ_Model pModel, int iComponent, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned T from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  pBatches          A list of batches that exist in the model, 
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned T results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT(SQ_Model pModel, int iComponent, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);

   /**
   *  Retrieves the aligned T average batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned T average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTAverageBatch(SQ_Model pModel, int iComponent, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned T stddev batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned T StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTStdDevBatch(SQ_Model pModel, int iComponent, SQ_VectorData* pVectorData);

   /**
   *  Retrieves the aligned To from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  pBatches          A list of batches that exist in the model,
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned To results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTo(SQ_Model pModel, int iComponent, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);

   /**
   *  Retrieves the aligned To average batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned To average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedToAverageBatch(SQ_Model pModel, int iComponent, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned To stddev batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned To StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedToStdDevBatch(SQ_Model pModel, int iComponent, SQ_VectorData* pVectorData);

   /**
   *  Retrieves the aligned T2Range from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[in]  pBatches          A list of batches that exist in the model,
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned T2Range results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */   
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2Range(SQ_Model pModel, int iCompFrom, int iCompTo, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);
      /**
   *  Retrieves the aligned T2Range average batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the aligned T2Range average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2RangeAverageBatch(SQ_Model pModel, int iCompFrom, int iCompTo, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned T2Range stddev batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the aligned T2Range StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2RangeStdDevBatch(SQ_Model pModel, int iCompFrom, int iCompTo, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned XVar from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iXTerm            The X term to use.
   *  @param[in]  pBatches          A list of batches that exist in the model, 
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned XVar results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */    
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVar(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iXTerm, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned XVar average batch from the model.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iXTerm            The X term to use.
   *  @param[out] pVectorData       A pointer to the aligned XVar Average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVarAverageBatch(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iXTerm, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned XVar stddev batch from the model.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iXTerm            The X term to use.
   *  @param[out] pVectorData       A pointer to the aligned XVar StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVarStdDevBatch(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iXTerm, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YVar from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pBatches          A list of batches that exist in the model, 
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned YVar results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */    
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYVar(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YVar average batch from the model.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YVar average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYVarAverageBatch(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YVar stddev batch from the model.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YVar StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYVarStdDevBatch(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData* pVectorData); 
   /**
   *  The Time or Maturity variable(s). 
   *  Determining the end point of a Batch/phase and used as Y in the training set model. 
   *  This variable is used to align batch/phase to the average length.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[out] pVectorData       A pointer to the aligned Time/Maturity results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTimeMaturity(SQ_Model pModel, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YPred from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pBatches          A list of batches that exist in the model, 
   *                                NULL if all batches that exist in the model should be used.
   *  @param[out] pVectorData       A pointer to the aligned YPred results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */   
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPred(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector *pBatches, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YPred average batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YPred average batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */    
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPredAverageBatch(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the aligned YPred stddev batch from the model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YPred StdDev batch results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */     
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPredStdDevBatch(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData* pVectorData);
      /*****************************************************************************************
   *
   *                       Out of Control
   *
   *  The following functions are available for a Batch Evolution Model.
   *  These vectors are the Out of Control summary for all of the vectors in the Batch Control charts.
   *
   *  The Out of Control summary for a given vector, for example scores, t1, 
   *  computes for every batch the ratio of the area of the vector outside the sigma limits to the total area inside the limits.  
   *  The Out of control summary is then expressed as a percent of the area inside the limits.
   *
   *  The Out of Control summary for all the vectors, i.e., scores, DModX, T2range and Observed vs predicted, 
   *  from both the training set and the prediction set are always computed on aligned vectors.
   *
   ******************************************************************************************/
   /**
   *  Out Of Control for DModX
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  fLimit            The sigma limit.
   *  @param[out] pVectorData       A pointer to the DModX OOC results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModXOOCSum(SQ_Model pModel, int iComponent, float fLimit, SQ_VectorData* pVectorData);
   /**
   *  Out Of Control for T
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the T OOC results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTOOCSum(SQ_Model pModel, int iComponent, float fHighLimit, float fLowLimit, SQ_VectorData* pVectorData);
   /**
   *  Out Of Control for T2Range
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the T2Range OOC results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2RangeOOCSum(SQ_Model pModel, int iCompFrom, int iCompTo, SQ_VectorData* pVectorData);
   /**
   *  Out Of Control for XVar
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iColumnXIndex     The index of the X-columns.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the XVar OOC results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVarOOCSum(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iColumnXIndex, float fHighLimit, float fLowLimit, SQ_VectorData* pVectorData);
   /**
   *  Out Of Control for YPred
   *
   *  @param[in]  pModel            The Batch Evolution Model to use.
   *  @param[in]  iComponent        The component to use.
   *  @param[in]  bUnscaled         If SQ_True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_False, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_False, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_True, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_True, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_False the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_True, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the YPred OOC results.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPredOOCSum(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, float fHighLimit, float fLowLimit, SQ_VectorData* pVectorData);

  /** @endcond // end of SQPPlus */

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQBATCHEVOLUTIONMODEL_H_ */
