/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQBATCHEVOLUTIONPREDICTION_H_
#define _SQBATCHEVOLUTIONPREDICTION_H_

#include "SQDef.h"

/** @file SQBatchEvolutionPrediction.h

   This file list the functions that are available for a prediction on a BEM.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"
#include "SQPrediction.h"
#include "SQIntVector.h"
#include "SQVectorData.h"

  /** @cond SQPPlus */
   
   /**
   * Get the aligned DModX contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  fMaturity         The maturity to get the contribution for. 
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned DModXPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModXPS(SQ_Prediction pPrediction, float fMaturity, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);
   /**
   * Get the aligned DModX group contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  pMaturity         A vector with the maturity to get the contribution for.  
   *  @param[in]  pPhaseIteration   A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned DModXPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModXGroupPS(SQ_Prediction pPrediction, SQ_FloatVector* pMaturity, SQ_IntVector* pPhaseIteration, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);
   
   /**
   * Get the aligned DModY contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  fMaturity         The maturity to get the contribution for. 
   *  @param[in]  eWeightType       The type of weight. Must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[out] pVectorData       A pointer to the aligned DModYPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModYPS(SQ_Prediction pPrediction, float fMaturity, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pVectorData);
   
   /**
   * Get the aligned DModY group contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  pMaturity         A vector with the maturity to get the contribution for.  
   *  @param[in]  pPhaseIteration   A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  eWeightType       The type of weight. Must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *  @param[out] pVectorData       A pointer to the aligned DModYPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = 1.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsDModYGroupPS(SQ_Prediction pPrediction, SQ_FloatVector* pMaturity, SQ_IntVector* pPhaseIteration, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pVectorData);

   /**
   * Get the aligned score single weight contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  fMaturity1        The maturity of the reference observation (from observation). 
   *  @param[in]  bAverage          True if the reference maturity is average batch, false otherwise.
   *  @param[in]  fMaturity2        The maturity for which the contributions are to be calculated (to observation).
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQX_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScorePSSingleWeight(SQ_Prediction pPrediction, float fMaturity1, SQ_Bool bAverage, float fMaturity2, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the aligned score single weight group contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  pMaturity1        Vector with the maturity of the reference observations (from observations). 
   *  @param[in]  pPhaseIteration1  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  bAverage          True if the reference maturity is average batch, false otherwise.
   *  @param[in]  pMaturity2        Vector with the maturity for which the contributions are to be calculated (to observations).
   *  @param[in]  pPhaseIteration2  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQX_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScorePSSingleWeightGroup(SQ_Prediction pPrediction, SQ_FloatVector* pMaturity1, SQ_IntVector* pPhaseIteration1, SQ_Bool bAverage, SQ_FloatVector* pMaturity2, SQ_IntVector* pPhaseIteration2, SQ_WeightType eWeightType, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the aligned score multi weight contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  fMaturity1        The maturity of the reference observation (from observation). 
   *  @param[in]  bAverage          True if the reference maturity is average batch, false otherwise.
   *  @param[in]  fMaturity2        The maturity for which the contributions are to be calculated (to observation).
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums.
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                for weight Po the component argument is the orthogonal component number and the model must be OPLS
   *                                for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScorePSMultiWeight(SQ_Prediction pPrediction, float fMaturity1, SQ_Bool bAverage, float fMaturity2, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the aligned score multi weight group contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   * The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  pMaturity1        Vector with the maturity of the reference observations (from observations). 
   *  @param[in]  pPhaseIteration1  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]  bAverage          True if the reference maturity is average batch, false otherwise.
   *  @param[in]  pMaturity2        Vector with the maturity for which the contributions are to be calculated (to observations).
   *  @param[in]  pPhaseIteration2  A vector of phase iteration names for which you would like to get contributions.
   *                                If the project doesn't have a phase iteration ID the parameter is ignored. 
   *                                If iBatchIx <1 it is also ignored and can be NULL.
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums.
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                for weight Po the component argument is the orthogonal component number and the model must be OPLS
   *                                for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the aligned Score single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedContributionsScorePSMultiWeightGroup(SQ_Prediction pPrediction, SQ_FloatVector* pMaturity1, SQ_IntVector* pPhaseIteration1, SQ_Bool bAverage, SQ_FloatVector* pMaturity2, SQ_IntVector* pPhaseIteration2, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pVectorData);

   /**
   *  Retrieves the aligned DModX from the predicted data.
   *  Distance to the model in the X space (row residual SD), after xx components (the selected dimension),
   *  for new observations in the predictions. 
   *  If you select component 0, it is the Stdev of the observations as scaled in the workset.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction             The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent              The number of the component in the model we want the results from
   *  @param[in]  bNormalized             If SQ_True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If SQ_False, they will be in absolute values.
   *  @param[in]  bModelingPowerWeighted  If SQ_True, the function will weight the residuals by the modeling power of the variables.
   *  @param[out] pVectorData             A pointer to the aligned DModX results.
   *                                      Number of rows in matrix = 1.
   *                                      Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModXPS(SQ_Prediction pPrediction, int iComponent, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the aligned T from the predicted data.
   *  The predicted scores, (for new observations), one vector for each model dimension. 
   *  They are new variables, computed from the model. 
   *  They summarize X, to best approximate X (PC model), and both approximate X and predict Y (PLS model).
   *  The function fails if the model doesn't have any components.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned T results.
   *                                Number of rows in matrix = 1 (only one component)
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTPS(SQ_Prediction pPrediction, int iComponent, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the aligned To from the predicted data.
   *  The predicted scores, (for new observations), one vector for each model dimension.
   *  They are new variables, computed from the model.
   *  They summarize X, to best approximate X (PC model), and both approximate X and predict Y (PLS model).
   *  The function fails if the model doesn't have any components.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The component to use.
   *  @param[out] pVectorData       A pointer to the aligned To results.
   *                                Number of rows in matrix = 1 (only one component)
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedToPS(SQ_Prediction pPrediction, int iComponent, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the aligned T2Range from the predicted data.
   *  Hotelling T2RangePS is a combination of all the scores (tPS) 
   *  in the selected range of components.
   *  It is a measure of how far away an observation is from the center 
   *  of a PC or PLS model hyperplane in the selected range of components.
   *  The function fails if the model doesn't have any components.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the aligned T2Range results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2RangePS(SQ_Prediction pPrediction, int iCompFrom, int iCompTo, SQ_VectorData *pVectorData);

   /**
   *  The Time or Maturity variable. 
   *  Determining the end point of a Batch/phase from the predicted data.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[out] pVectorData       A pointer to the aligned Time/Maturity results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTimeMaturityPS(SQ_Prediction pPrediction, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the predicted aligned XVar from a model.
   *  X variable from the predictions, in original units.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  bUnscaled         If SQ_TRUE, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_FALSE, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_FALSE, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_TRUE, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_TRUE, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_FALSE the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_TRUE, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iXTerm            The X term to use.
   *  @param[out] pVectorData       A pointer to the aligned XVar results.
   *                                Number of rows in matrix = number of x-variables chosen.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVarPS(SQ_Prediction pPrediction,  SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iXTerm, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the aligned YPredPS.
   *  Aligned predicted values of the time/maturity variable in the predictions.
   *  The function fails if the model doesn't have any components.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *  @param[in]  bUnscaled         If SQ_TRUE, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_FALSE, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_FALSE, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_TRUE, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_TRUE, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_FALSE the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_TRUE, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YPredPS results.
   *                                Number of rows in matrix = 1 (only one y variable exist in a Batch Evolution Model). 
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPredPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the aligned YVarPS matrix.
   *  The aligned time/maturity variable from the predictions.
   *  The function is only valid for a Batch Evolution model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  bUnscaled         If SQ_TRUE, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_FALSE, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_FALSE, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_TRUE, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_TRUE, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_FALSE the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_TRUE, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the aligned YVarPS results.
   *                                Number of rows in matrix = 1 (only one y variable exist in an observation level project).
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */ 
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYVarPS(SQ_Prediction pPrediction, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData *pVectorData);

   /*****************************************************************************************
   *
   *                       Out of Control
   *
   *  The following functions are available for a Batch Evolution model.
   *  These vectors are the Out of Control summary for all of the vectors in the Batch Control charts.
   *
   *  The Out of Control summary for a given vector, for example scores, t1, 
   *  computes for every batch the ratio of the area of the vector outside the sigma limits to the total area inside the limits.  
   *  The Out of control summary is then expressed as a percent of the area inside the limits.
   *
   *  The Out of Control summary for all the vectors, i.e., scores, DModX, T2range and Observed vs predicted, 
   *  from both the training set and the prediction set are always computed on aligned vectors.
   *
   ******************************************************************************************/
   /**
   *  Out Of Control for DModXPS
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The component in the model to get the result from.
   *  @param[in]  fLimit            The sigma limit.
   *  @param[out] pVectorData       A pointer to the aligned DModXPS OOC results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedDModXPSOOCSum(SQ_Prediction pPrediction, int iComponent, float fLimit, SQ_VectorData *pVectorData);

   /**
   *  Out Of Control for TPS
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The component in the model to get the result from.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the aligned TPS OOC results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedTPSOOCSum(SQ_Prediction pPrediction, int iComponent, float fHighLimit, float fLowLimit, SQ_VectorData *pVectorData);

   /**
   *  Out Of Control for T2RangePS
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the aligned T2RangePS OOC results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedT2RangePSOOCSum(SQ_Prediction pPrediction, int iCompFrom, int iCompTo, SQ_VectorData *pVectorData);

   /**
   *  Out Of Control for XVarPS
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  bUnscaled         If SQ_TRUE, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If SQ_FALSE, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_FALSE, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_TRUE, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_TRUE, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_FALSE the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_TRUE, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iColumnXIndex     The index of the X-columns.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the aligned XVarPS OOC results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedXVarPSOOCSum(SQ_Prediction pPrediction, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iColumnXIndex, float fHighLimit, float fLowLimit, SQ_VectorData *pVectorData);

   /**
   *  Out Of Control for YPredPS
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetBatchEvolutionPrediction()
   *  @param[in]  iComponent        The component in the model to get the result from.
   *  @param[in]  bUnscaled         If SQ_TRUE, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If SQ_FALSE, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is SQ_FALSE, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is SQ_TRUE, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If SQ_TRUE, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If SQ_FALSE the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is SQ_TRUE, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  fHighLimit        The higher sigma limit.
   *  @param[in]  fLowLimit         The lower sigma limit.
   *  @param[out] pVectorData       A pointer to the aligned YPredPS OOC results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = size of the aligned time or maturity vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAlignedYPredPSOOCSum(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, float fHighLimit, float fLowLimit, SQ_VectorData *pVectorData);

  /** @endcond // end of SQPPlus */

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQBATCHEVOLUTIONPREDICTION_H_ */
