/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQBATCHLEVELMODEL_H_
#define _SQBATCHLEVELMODEL_H_

#include "SQDef.h"

/** @file SQBatchLevelModel.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQModel.h"
#include "SQPrepareBatchPrediction.h"

  /** @cond SQPPlus */
   
   /**
   *  An inquiry whether the model is a Batch Level Model.
   *
   *  @param[in]  pModel            The model to use.
   *  @param[out] bIsBLM            Is set to SQ_True if the model is a BLM.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsBatchLevelModel(SQ_Model pModel, SQ_Bool* bIsBLM);

   /**
   *  The importance of a variable during the batch evolution and for the quality of the completed batch. 
   *  Batch VIP is computed for every phase.
   *  For a PLS model at the batch level, it is computed for the selected Y.
   *
   *  @param[in]  pModel            The batch level model to use.
   *  @param[in]  iComponent        The component the batch VIP should be calculated for.
   *  @param[in]  iYTerm            The Y term the batch VIP should be calculated for.
   *  @param[in]  iPhase            The phase the batch VIP should be calculated for.
   *  @param[out] pVectorData       A pointer to the BatchVIP results.
   *                                Number of rows in matrix = 2
   *                                Number of columns in matrix = 
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetBatchVIP(SQ_Model pModel, int iComponent, int iYTerm, int iPhase, SQ_VectorData* pVectorData);

   /** 
   *  Retrieves the PrepareBatchPrediction object that should be used to set data for a batch prediction.
   *  The returned object must be released by SQ_ClearPrepareBatchPrediction.
   *
   *  @param[in]   pModel              The batch level model to perform a prediction on.  
   *  @param[out]  pPreparePrediction  The PreparePrediction object that should be used to set data for a prediction.   
   *                                   Must be released by SQ_ClearPrepareBatchPrediction.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPrepareBatchPrediction(SQ_Model pModel, SQ_PrepareBatchPrediction *pPreparePrediction);

   /** 
   *  Retrieves the phases that are needed to perform a prediction on this batch level model.
   *
   *  @param[in]   pModel                    The batch level model to use.  
   *  @param[out]  pPhasesForBatchPredict    The phase names.
   *                                         The vector must be cleared with SQ_ClearStringVector.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPhasesForBatchPrediction(SQ_Model pModel, SQ_StringVector *pPhasesForBatchPredict);

  /** @endcond // end of SQPPlus */

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQBATCHLEVELMODEL_H_ */
