/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQBOOLVECTOR_H_
#define _SQBOOLVECTOR_H_

#include "SQDef.h"

/** @file SQBoolVector.h

   This file list the SQ_BoolVector object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"

   /**
   An bool vector that is used to get and send data to some functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_BoolVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_BoolVector;

   /**
   *  Initialize a boolean vector with its size.
   *  @see tagSQ_BoolVector
   *
   *  @param[in]  pVector        The vector to be initialized.
   *  @param[out] iSize          The size of the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitBoolVector(SQ_BoolVector* pVector, int iSize);

   /**
   *  Get a copy of the bool vector
   *  @see tagSQ_BoolVector
   *
   *  @param[in]   pInVector        The vector to copy.
   *  @param[in, out]   pOutVector  The result vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyBoolVector(SQ_BoolVector pInVector, SQ_BoolVector* pOutVector);

   /**
   *  Removes the vector, this function must be called for every vector that is created.
   *  @see tagSQ_BoolVector
   *
   *  @param[in]  pVector        The vector to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearBoolVector(SQ_BoolVector* pVector);
   /**
   *  Sets the data at the given position.
   *  @see tagSQ_BoolVector
   *
   *  @param[in]  pVector        The vector to set the data in.
   *  @param[in]  iPos           The position to set the data in.
   *  @param[in]  bVal           The data to set the cell with.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataInBoolVector(SQ_BoolVector pVector, int iPos, SQ_Bool bVal);
   /**
   *  Gets the data from a given position.
   *  @see tagSQ_BoolVector
   *
   *  @param[in]  pVector        The vector to get the data from.
   *  @param[in]  iPos           The position to get the data from.
   *  @param[in]  pbVal          The data in the cell.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromBoolVector(const SQ_BoolVector pVector, int iPos, SQ_Bool* pbVal);
   /**
   *  Returns the size of the vector.
   *  @see tagSQ_BoolVector
   *
   *  @param[in]  pVector        The vector to get the number of columns from.
   *  @param[out] piSize         The size of the vector or a negative number if an error occurred.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetBoolVectorSize(const SQ_BoolVector pVector, int* piSize);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQBOOLVECTOR_H_ */
