/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQCOMMON_H_
#define _SQCOMMON_H_

#include "SQDef.h"

/** @file SQCommon.h
   This file includes all enum objects used in different functions and common functions about logging, license file etc.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQFloatVector.h"
#include "SQIntVector.h"

   /**
   An enum defining the different product types.
   */
   typedef enum SQ_ProductEnum
   {
      SQ_SQP=0,     /**< The product is SIMCA-QP. */
      SQ_SQPPlus,   /**< The product is SIMCA-QP+. */
      SQ_SQM,       /**< The product is SIMCA-QM. */
      SQ_SQAll     /**< The product is SIMCA-Q All (QP, QP+ and QM). */
   } SQ_Product;

   /**
   An enum defining true or false.
   */
   typedef enum SQ_BoolEnum
   {
      SQ_False = 0,
      SQ_True
   } SQ_Bool;

   /**
   An enum defining the a tristate for Unscaled.
   */
   typedef enum SQ_TriStateUnscaledState
   {
      SQ_Unscaled_False = 0,
      SQ_Unscaled_True,
      SQ_Unscaled_Default
   } SQ_UnscaledState;

   /**
   An enum defining the a tristate for Backtransformed.
   */
   typedef enum SQ_TriStateBacktransformedState
   {
      SQ_Backtransformed_False = 0,
      SQ_Backtransformed_True,
      SQ_Backtransformed_Default
   } SQ_BacktransformedState;

   /**
   An enum defining the a tristate for Reconstruct.
   */
   typedef enum SQ_TriStateReconstructState
   {
      SQ_Reconstruct_False = 0,
      SQ_Reconstruct_True,
      SQ_Reconstruct_Default
   } SQ_ReconstructState;

   /**
   An enum defining the a tristate for Standardized.
   */
   typedef enum SQ_TriStateStandardizedState
   {
      SQ_Standardized_False = 0,
      SQ_Standardized_True,
      SQ_Standardized_Default
   } SQ_StandardizedState;

   /**
   An enum defining the a tristate for ResolveHierachical.
   */
   typedef enum SQ_TriStateResolveHierachicalState
   {
      SQ_ResolveHierachical_False = 0,
      SQ_ResolveHierachical_True,
      SQ_ResolveHierachical_Default
   } SQ_ResolveHierachicalState;

   /**
   An enum defining the a tristate for Normalized.
   */
   typedef enum SQ_TriStateNormalizedState
   {
      SQ_Normalized_False = 0,
      SQ_Normalized_True,
      SQ_Normalized_Default
   } SQ_NormalizedState;

   /**
   An enum defining the a tristate for Modeling power weighted.
   */
   typedef enum SQ_TriStateModelingPowerWeightedState
   {
      SQ_ModelingPowerWeighted_False = 0,
      SQ_ModelingPowerWeighted_True,
      SQ_ModelingPowerWeighted_Default
   } SQ_ModelingPowerWeightedState;

   /**
   Different types of coefficients that is used in the GetModelCoefficientsRotated function.
   */
   typedef enum SQ_CoefficientsRotatedTypeEnum
   {
      SQ_XYUnscaledUncenterd = 0,               /**< Both X and Y are unscaled and uncentered */
      SQ_XUnscaledCenterdYUnscaledUncenterd,    /**< X is unscaled and centered, Y is unscaled and uncentered */
      SQ_XScaledCenterdYScaledUncenterd,        /**< X is scaled and centered, Y is scaled and uncentered */
      SQ_MLR                                    /**< X is scaled and centered, Y is unscaled and uncentered, the second centering and scaling of the cross terms and squares has been removed. */
   } SQ_CoefficientsRotatedType;

   /**
      An enum defining the different model types.
   */
   typedef
   enum SQ_ModelTypeEnum
   {
      SQ_UnDefined=0,/**< The model type can not be determined */
      SQ_PCA_X,      /**< The model is a PCA_X model */
      SQ_PCA_Y,      /**< The model is a PCA_Y model */
      SQ_PCA_All,    /**< The model is a PCA_All model */
      SQ_PCA_Class,  /**< The model is a PCA_Class model */
      SQ_PLS_Class,  /**< The model is a PLS_Class model */
      SQ_PLS,        /**< The model is a PLS model */
      SQ_PLS_DA,      /**< The model is a PLS_DA model */
      SQ_OPLS,        /**< The model is an OPLS model */
      SQ_OPLS_DA,     /**< The model is an OPLS_DA model */
      SQ_OPLS_Class,  /**< The model is an OPLS_Class model */
      SQ_O2PLS,       /**< The model is an O2PLS model */
      SQ_O2PLS_DA,    /**< The model is an O2PLS_DA model */
      SQ_O2PLS_Class, /**< The model is an O2PLS_Class model */
      SQ_MOCA   /**< The model is an Multiblock model */
   } SQ_ModelType;

   /**
       An enum defining the different weights that are expected as input to the Contribution functions.
   */
   typedef
   enum SQ_WeightTypeEnum
   {
      SQ_Weight_Normalized=0, /**< No weights are used, but displayed in the units of the workset. */
      SQ_Weight_Raw,          /**< No weights are used. Displays the subtraction of the values in one observation minus the other in original units. */
      SQ_Weight_P,            /**< Weight P, PCA loading */
      SQ_Weight_WStar,        /**< Weight W*, PLS weight */
      SQ_Weight_RX,           /**< Weight RX, modeled variance */
      SQ_Weight_RY,           /**< Weight RY, modeled variance */
      SQ_Weight_CoeffCS,      /**< Weight CoeffCS, regression coefficients centered and scaled. */
      SQ_Weight_CoeffCSRaw,   /**< Weight CoeffCSRaw. Contributions are normally weighted with the absolute value of the weights. But Y Predicted contributions uses raw CoeffCS weights. */
      SQ_Weight_VIP,          /**< Weight VIP */
      SQ_Weight_WStarRange,   /**< Weight W*Range */
      SQ_Weight_PRange,       /**< Weight PRange, for OPLS models the component parameter is ignored, all components are used.*/
      SQ_Weight_PO           /**< Weight Po orthogonal loadings, component parameter is the orthogonal component */
   } SQ_WeightType;

   /**
      Enum values describing what kind of type a variable is of
    */
   typedef enum SQ_VariableTypeType {
      SQ_XVariable = 0,                    /**< Variable is X. */
      SQ_YVariable,                        /**< Variable is Y. */
      SQ_UnDefVariable                     /**< Variable is Undefined, that means it's included in the model because it's needed for predictions, but neither X or Y. */
   } SQ_VariableType;

   /**
   Indicates which variable block a variable belongs to.
   */
   typedef enum SQ_BlockTypeEnum
   {
      SQ_BlockX, /**< A variable belonging to the X block. */
      SQ_BlockY  /**< A variable belonging to the Y block. */
   } SQ_VariableBlock;

   /************************************************************************/
   /* Struct for model options information                                  */
   /************************************************************************/

   /**
   *	Struct with information about options of a specific model.
   */
   typedef struct
   {
      /**
      * Model Residual options
      */
      enum SQ_ResidualsType {
         E_Raw = 0,              /**< Raw, original units */
         E_Standardized          /**< Standardized values */
      } eResiduals;              /**< Model Residual options object */

      /**
      * Model R2 options
      */
      enum SQ_R2Type {
         E_Explained = 0,        /**< Explained variation */
         E_Adjusted              /**< Adjusted variance */
      } eR2;                     /**< Model R2 options object */

      /**
      * Model Coefficient options
      */
      enum SQ_CoefficientType {
         E_Scaled = 0,           /**< Scaled and Centered */
         E_MLR,                  /**< MLR */
         E_Unscaled,             /**< Unscaled */
         E_Rotated               /**< Rotated */
      } eCoefficients;           /**< Model Coefficient options object */

      /**
      * Model Distance to Model options
      */
      enum SQ_DModType {
         E_Normalized = 0,       /**< Normal units of standard deviation */
         E_Absolute              /**< Absolute */
      } eDistanceToModel;        /**< Model Distance to Model options object */

      SQ_Bool bWeighted;             /**< True or false if the model options are weighted */
   } SQ_ModelOptions; 

   /**
	   A struct with information about a model
   */
   typedef struct tagSQ_ModelInfo
   {
	   SQ_Bool isFitted;					/**< SQ_True if fitted, otherwise SQ_False */
	   SQ_Bool isBatchLevelModel;			/**< SQ_True if batch level model, otherwise SQ_False */
	   SQ_Bool isBatchEvolutionModel;		/**< SQ_True if batch evolution model, otherwise SQ_False */
	   int numberOfPredictiveComponents;	/**< Number of predictive components */
	   int numberOfXOrthogonalComponents;	/**< Number of X orthogonal components */
	   int numberOfYOrthogonalComponents;	/**< Number of Y orthogonal components */
	   char modelName[128];					/**< The name of the model, like M1 */
	   char modelTitle[128];				/**< The title of the model */
	   char modelTypeName[128];				/**< The type of model written as text, for example PLS */
	   SQ_ModelType modelType;				/**< The type of model, for example PLS */
	   long lastModified;					/**< The time and date when the model was created or modified, representing the number of seconds elapsed since 00:00 hours, Jan 1, 1970 UTC. */
	   int numberOfXVariables;				/**< Number of X variables included in the model including expanded qualitative variables*/
	   int numberOfYVariables;				/**< Number of Y variables included in the model including expanded qualitative variables*/
	   int numberOfObservations;			/**< Number of observations included in the model */
   } SQ_ModelInfo;
   
   /*************************  Initialization functions  ***********************************/

   /* These functions should be called before the first project is opened or created. */

   /**
   *  Returns a status indicating if SIMCA-Q is logging information and errors or not.
   *
   *  @param[out] pbStatus       SQ_True if the log is on, SQ_False otherwise
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLoggingStatus(SQ_Bool *pbStatus);
   /**
   *  Sets the status indicating if SIMCA-Q should log information and errors or not.
   *
   *  @param[in]  bLogOn         SQ_True if the log should be turned on, SQ_False otherwise
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetLoggingStatus(SQ_Bool bLogOn);
   /**
   * User defined log file.
   * This function can only be called before any other initialization.
   * Make sure that the directory exists before calling this function.
   *
   *  @param[in] szPathName      The path to and name of the logfile, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetLogFile(const char *szPathName);
   /**
   * Returns the path of the log file, 
   * "" if there does not exist a log file (the log has been turned of or the log function is used).
   *
   * @param[in,out] szPathName   The path to and name of the logfile, UTF-8 encoded. The user is responsible to allocate and deallocate the buffer.
   * @param[in] iBufferLength    The length of the buffer.
   * @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLogFile(char* szPathName, int iBufferLength);
   /**
   * @see SQ_SetLogFunction
   *
   * @param[out] szMessage       The log message, UTF-8 encoded.
   * @param[out] iStatus         1 if the message is an error message, 
   *                             0 if the message is an informative message.
   */
   typedef void (SQCDECL *SQ_LogFunction)(const char *szMessage, int iStatus);
   /**
   *  User defined log function.
   *
   *  @param[in] pFunction       The function the user wants to use for logging.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetLogFunction(SQ_LogFunction pFunction);

   /**
    * Prediction logging
    */

    /**
    * User defined prediction log file.
    * This function can only be called before opening a project.
    * Make sure that the directory exists before calling this function.
    *
    *  @param[in] szPathName      The path to and name of the prediction logging, UTF-8 encoded.
    *  @return                    Returns SQ_E_OK if success or an error code
    */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPredictionLogFile(const char *szPathName);
   /**
   * Returns the path of the prediction log file,
   * "" if there does not exist a log file (the license doesn't enable logging or the log function is used).
   *
   * @param[in,out] szPathName   The path to and name of the logfile, UTF-8 encoded. The user is responsible to allocate and deallocate the buffer.
   * @param[in] iBufferLength    The length of the buffer.
   * @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPredictionLogFile(char* szPathName, int iBufferLength);
   /**
   * @see SQ_SetPredictionLogFunction
   *
   * @param[out] iNumberOfPredictions  Number of predictions made since last time the callback function was called.
   * @param[out] iStatus               Number of projects that has been opened since last time the callback function was called.
   */
   typedef void (SQCDECL *SQ_PredictionLogFunction)(int iNumberOfPredictions, int iNumberOfProjectsOpened);
   /**
   *  User defined prediction log function.
   *
   *  @param[in] pFunction       The function the user wants to use for logging.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPredictionLogFunction(SQ_PredictionLogFunction pFunction);

   /******************************************************************************
   *
   *                           General functions 
   *
   *******************************************************************************/
   /**
   *  Retrieves the number that represents missing value.
   *
   *  @param[out] pfVal    The number that represents missing value in SIMCA-Q.
   *  @return              Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMissingValue(float* pfVal);

   /**
   *  Retrieves the version number of the SIMCA-Q.dll.
   *  Four numbers will be returned that can be combined in the following format 13.0.0.0.
   *  
   *  @param[out] iMajorVersion     The major version.
   *  @param[out] iMinorVersion     The minor version.
   *  @param[out] iBuildNumber      The build number.
   *  @param[out] iRevisionNumber   The revision number.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVersionNumber(int* iMajorVersion, int* iMinorVersion, int* iBuildNumber, int* iRevisionNumber);

   /**
   *  If the SIMCA-Q is part of an OEM solution, 
   *  the password for the license file should be applied with this function.
   *  The password is case sensitive.
   *
   *  @param[in]  szOEMPassword  The OEM password that is valid for the license file, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetOEMPassword(const char* szOEMPassword);

   /**
   * Set the directory of where the license file is located.
   * If this function is not called SIMCA-Q will look for the license file 
   * in the working directory.
   *
   *  @param[in] szLicensePath   Path to where the license file is located, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetLicensePath(const char *szLicensePath);

   /**
   * Set the directory of where the plug-ins for spectral filters are located.
   * If this function is not called SIMCA-Q will look for the plug-ins 
   * in the working directory.
   *
   *  @param[in] szPluginPath    Path to where the plug-ins are located, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPluginPath(const char *szPluginPath);

#ifdef _WIN32
   /**
   * Add a the directory of where python searches for modules
   *
   *  @param[in] szPath          The new path, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddPythonPath(const char *szPath);
#endif

   /**
   * Checks if a license file is present and in that case valid.
   *
   *  @param[out] bValid      SQ_True if the file has been found and is valid,
   *                          SQ_False if not, in that case see the log file for details.
   *  @return                 Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsLicenseFileValid(SQ_Bool* bValid);

   /**
   *  Checks if a license file is present and valid and in that case for how long the license is valid.
   *
   *  @param[in, out] szExpireDate  A buffer to where the expire date should be stored. 
   *                                The string will be empty if no expire date is set or if the license file is not valid.
   *                                The user is responsible for allocation/deallocation.
   *  @param[in]   iLength          The size of the buffer.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLicenseFileExpireDate(char* szExpireDate, int iLength);
   
   /**
   *  Get the product of the license file.
   *
   *  @param[out] pSQProduct  The product of the license file.
   *  @return                 Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLicenseFileProduct(SQ_Product* pSQProduct);

   /**
   * Checks if a project is valid and can be opened by this version of SIMCA-Q.
   * Don't use this function on a opened project.
   *
   *  @param[in]  szProjectName  The full path to the project file, UTF-8 encoded.
   *  @param[in]  szPassword     The password if the project is password protected, UTF-8 encoded, if not use NULL.
   *  @param[out] bValid         SQ_True if the file is a valid project,
   *                             SQ_False if not, in that case see the log file for details.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsProjectValid(const char *szProjectName, const char *szPassword, SQ_Bool* bValid);

   /**
   *  Turn multi-threading on or off. 
   *  Multi-threading makes SIMCA-Q faster for larger projects if run on more than one processor.
   *  Multi-threading should be turned off for small projects were many predictions are performed and for
   *  hyper-threaded (old Pentium) processors.
   *
   *  @param[in] bMultiThread          SQ_True to use multi-threading, SQ_False if no multi-threading should be used.
   *  @param[in] iNumProcessors        The number of processors decides the maximum number of simultaneous threads SIMCA-Q will execute.
   *                                   A higher value than the actual number of processors will make SIMCA-Q slower.
   *                                   -1 means default number of processors.
   *                                   This parameter is ignored if bMultiThread is set to SQ_False.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_UseMultiThreading(SQ_Bool bMultiThread, int iNumProcessors);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQCOMMON_H_ */
