/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQCOMPONENTVECTOR_H_
#define _SQCOMPONENTVECTOR_H_

#include "SQDef.h"

/** @file SQComponentVector.h
   This file contains the SQ_ComponentVector object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"

   /**
   An int vector that is used to get and send data to many functions. Index base = 1.
   */
   typedef struct tagSQ_ComponentVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_ComponentVector;


   /**
   *  Initialize a component vector with its size.
   *  @see tagSQ_ComponentVector
   *
   *  @param[in]   pVector        The vector to be initialized.
   *  @param[in]   nSize          The size of the vector.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitComponentVector(SQ_ComponentVector *pVector, int nSize);
   /**
   *  Removes the vector, this function must be called for every vector that is created.
   *  @see tagSQ_ComponentVector
   *
   *  @param[in]   pVector        The vector to be removed.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearComponentVector(SQ_ComponentVector* pVector);
   /**
   *  Sets the data at the given position.
   *  @see tagSQ_ComponentVector
   *
   *  @param[in]   pVector        The vector to set the data in.
   *  @param[in]   nPos           The model or class to set number of components
   *  @param[in]   iPredComp      The prediction component. This is the only component for non O2PLS models.
   *  @param[in]   iXComp         The X side component. O2PLS models only.
   *  @param[in]   iYComp         The Y side component. O2PLS models only.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataInComponentVector(SQ_ComponentVector pVector, int nPos, int iPredComp, int iXComp, int iYComp);
   /**
   *  Gets the data from a given position.
   *  @see tagSQ_ComponentVector
   *
   *  @param[in]   pVector        The vector to get the data from.
   *  @param[in]   nPos           The model or class to set number of components.
   *  @param[out]  piPredComp     The prediction component. This is the only component for non O2PLS models.
   *  @param[out]  piXComp        The X side component. O2PLS models only.
   *  @param[out]  piYComp        The Y side component. O2PLS models only.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromComponentVector(SQ_ComponentVector pVector, int nPos, int *piPredComp, int *piXComp, int *piYComp);
   /**
   *  Returns the size of the vector.
   *  @see tagSQ_ComponentVector
   *
   *  @param[in]   pVector        The vector to get the number of columns from.
   *  @param[out]  piSize         The size of the vector.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetComponentVectorSize(SQ_ComponentVector pVector, int* piSize);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQCOMPONENTVECTOR_H_ */
