/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQCONTROLCHARTS_H_
#define _SQCONTROLCHARTS_H_

#include "SQDef.h"

/** @file SQControlCharts.h
   This file contains the different control charts that can be created with Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQProject.h"
#include "SQFloatVector.h"


   /**
   The handle used to identify a Shewhart control chart.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_ShewhartControlChart 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_ShewhartControlChart;

   /**
   The handle used to identify a EWMA control chart.
   */
   typedef struct tagSQ_EWMAControlChart 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_EWMAControlChart;

   /**
   The handle used to identify a Cusum control chart.
   */
   typedef struct tagSQ_CusumControlChart 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_CusumControlChart;

   /**
   An enum defining the different Shewhart types that can be requested from a Shewhart control chart.
   */
   typedef enum SQ_ShewhartTypeEnum
   {
      SQ_SCC_MEANRANGE=0,     /**< Make a Mean/Range Shewhart control chart */
      SQ_SCC_MEANSTDDEV       /**< Make a Mean/Std.Dev Shewhart control chart */
   } SQ_ShewhartType;
   
   /**
   An enum defining the different EWMA calculations. Before SIMCA-Q 14.1, only predictive (SQ_PREDICTIVE) was available.   
   */
   typedef enum SQ_EWMATypeEnum
   {
      SQ_FILTER = 0,     /**filter EWMA calculations */
      SQ_PREDICTIVE      /**predictive EWMA calculations (used in versions before SIMCA-Q 14.1) */
   } SQ_EWMAType;

   /**
   *  Get the Shewhart control chart.
   *  For a detailed description of the nomenclature,
   *  see "McNeese: Statistical methods for the process industries" Quality and Reliability p.28
   *  A2, A3, B3, B4, D3, D4, A2 and d2 from McNeese
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the Shewhart control chart from.
   *  @param[in]  eShewhartType  The type of the Shewhart.
   *  @param[in]  iSampleSize    Must be smaller than the number of observations (the size of pData)
   *                             and between 1 to 25.
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[out] pShewhart      A pointer to the Shewhart object.
   *  @return                    Returns SQ_E_OK if success or an error code
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetShewhartControlChart(SQ_Project pProject, SQ_FloatVector pData, SQ_ShewhartType eShewhartType, int iSampleSize, float fTarget, float fStdDev, SQ_ShewhartControlChart *pShewhart);

   /**
   *  Get the Shewhart control chart with a range variable.
   *  For a detailed description of the nomenclature,
   *  see "McNeese: Statistical methods for the process industries" Quality and Reliability p.28
   *  A2, A3, B3, B4, D3, D4, A2 and d2 from McNeese
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the Shewhart control chart from.
   *  @param[in]  pRangeData     The vector to create the grouping from. Should be of equal length as pData
   *  @param[in]  fRange         A range in the range data vector
   *  @param[in]  eShewhartType  The type of the Shewhart.
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[out] pShewhart      A pointer to the Shewhart object.
   *  @return                    Returns SQ_E_OK if success or an error code
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetShewhartRangeGroupControlChart(SQ_Project pProject, SQ_FloatVector pData, SQ_FloatVector pRangeData, float fRange, SQ_ShewhartType eShewhartType, float fTarget, float fStdDev, SQ_ShewhartControlChart *pShewhart);

   /**
   *  Removes the Shewhart Control Chart, this function must be called for every Shewhart Control Chart object that is created.
   *  @see tagSQ_ShewhartControlChart
   *
   *  @param[in]  pShewhartControlChart   The Shewhart Control Chart object to be removed.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearShewhartControlChart(SQ_ShewhartControlChart* pShewhartControlChart);

   /**
   *  Get the Estimated StdDev from the Shewhart control chart.
   *
   *  @param[in]  pShewhartControlChart   The Shewhart control chart.
   *  @param[out] pfStdDev                The process standard deviation estimated from the Workset. S(MModel number) in SIMCA.
   *  @param[out] pfStdDevProcess         If not User Entered, estimated from Avg. S(RAvg) or S(SAvg) in SIMCA.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSCCEstimatedStdDev(SQ_ShewhartControlChart pShewhartControlChart, float* pfStdDev, float* pfStdDevProcess);

   /**
   *  Get the mean from the Shewhart control chart.
   *
   *  @param[in]  pShewhartControlChart   The Shewhart control chart.
   *  @param[out] pMeanVector             The mean of samples of the selected item.
   *  @param[out] pfTarget                If not User Entered, Overall process average, estimated from average of subgroups. Target(MModel number) in SIMCA.
   *  @param[out] pfUCL                   Target + A2Rbar/A3Sbar. UCL(x) in SIMCA.
   *  @param[out] pfLCL                   Target - A2Rbar/A3Sbar. LCL(x) in SIMCA.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSCCMean(SQ_ShewhartControlChart pShewhartControlChart, SQ_FloatVector* pMeanVector, float* pfTarget, float* pfUCL, float* pfLCL);

   /**
   *  Get the range from the Shewhart control chart.
   *
   *  @param[in]  pShewhartControlChart   The Shewhart control chart.
   *  @param[out] pRangeVector            The range of samples of the selected item.
   *  @param[out] pfMean                  Average range of subgroups. (Rbar). RAvg(within, MModel number) in SIMCA.
   *  @param[out] pfUCL                   D4Rbar. UCL(r) in SIMCA.
   *  @param[out] pfLCL                   D3Rbar. LCL(r) in SIMCA.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSCCRange(SQ_ShewhartControlChart pShewhartControlChart, SQ_FloatVector* pRangeVector, float* pfMean, float* pfUCL, float* pfLCL);

   /**
   *  Get the StdDev from the Shewhart control chart.
   *
   *  @param[in]  pShewhartControlChart   The Shewhart control chart.
   *  @param[out] pStdDevVector           The standard deviation of samples of the selected item.
   *  @param[out] pfMean                  Average standard deviation of the subgroups. (Sbar) SAvg(within, MModel number) in SIMCA.
   *  @param[out] pfUCL                   B4Sbar. UCL(s) in SIMCA.
   *  @param[out] pfLCL                   B3Sbar. LCL(s) in SIMCA.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSCCStdDev(SQ_ShewhartControlChart pShewhartControlChart, SQ_FloatVector* pStdDevVector, float* pfMean, float* pfUCL, float* pfLCL);

   /**
   *  Get the EWMA(Exponentially weighted moving averages) control chart.
   *  For a detailed description of the nomenclature,
   *  see "McNeese: Statistical methods for the process industries" Quality and Reliability p.28
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the EWMA control chart from.
   *  @param[in]  iSampleSize    Must be smaller than the number of observations (the size of pData)
   *                             and between 1 to 25.
   *  @param[in]  fLambda        Set to MissingValue if lambda will be estimated.
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[in]  eEWMAType      The type of EWMA calculation filter or predictive, before SIMCA-Q 14.1 only predictive was available.
   *  @param[out] pEWMA          A pointer to the EWMA struct.
   *  @return                    Returns SQ_E_OK if success or an error code
   *
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMAControlChart(SQ_Project pProject, SQ_FloatVector pData, int iSampleSize, float fLambda, float fTarget, float fStdDev, SQ_EWMAType eEWMAType, SQ_EWMAControlChart *pEWMA);

   /**
   *  Get the EWMA(Exponentially weighted moving averages) control chart using a range variable
   *  For a detailed description of the nomenclature,
   *  see "McNeese: Statistical methods for the process industries" Quality and Reliability p.28
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the EWMA control chart from.
   *  @param[in]  pRangeData     The vector to create the grouping from. Should be of equal length as pData
   *  @param[in]  fRange         A range in the range data vector
   *  @param[in]  fLambda        Set to MissingValue if lambda will be estimated.
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[in]  eEWMAType      The type of EWMA calculation filter or predictive, before SIMCA-Q 14.1 only predictive was available.
   *  @param[out] pEWMA          A pointer to the EWMA struct.
   *  @return                    Returns SQ_E_OK if success or an error code
   *
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMARangeControlChart(SQ_Project pProject, SQ_FloatVector pData, SQ_FloatVector pRangeData, float fRange, float fLambda, float fTarget, float fStdDev, SQ_EWMAType eEWMAType, SQ_EWMAControlChart *pEWMA);

   /**
   *  Removes the EWMA Control Chart, this function must be called for every EWMA Control Chart object that is created.
   *  @see tagSQ_EWMAControlChart
   *
   *  @param[in]  pEWMA    The EWMA Control Chart object to be removed.
   *  @return              Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearEWMAControlChart(SQ_EWMAControlChart* pEWMA);
   /**
   *  Get the EWMA vector and limits from the EWMA control chart.
   *
   *  @param[in]  pEWMAControlChart       The EWMA control chart.
   *  @param[out] pEWMAVector             The EWMA of samples of the selected item.
   *  @param[out] pfTarget                If not User Entered, estimated as the grand average of the selected item over the training set. Target(MModel number) in SIMCA.
   *  @param[out] pfUCL                   Target + 3 * S(EWMA). UCL(EWMA) in SIMCA.
   *  @param[out] pfLCL                   Target  3 * S(EWMA). LCL(EWMA) in SIMCA.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMAVector(SQ_EWMAControlChart pEWMAControlChart, SQ_FloatVector* pEWMAVector, float* pfTarget, float* pfUCL, float* pfLCL);
   /**
   *  Get the EWMA lambda from the EWMA control chart.
   *  If not User Entered, estimated from the training set, which minimizes the error sum of squares.
   *
   *  @param[in]  pEWMAControlChart       The EWMA control chart.
   *  @param[out] pfLambda                The lambda.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMALambda(SQ_EWMAControlChart pEWMAControlChart, float* pfLambda);
   /**
   *  Get the EWMA StdDev individual from the EWMA control chart.
   *  Process standard deviation over all observations. S(MModel number) in SIMCA.
   *
   *  @param[in]  pEWMAControlChart       The EWMA control chart.
   *  @param[out] pfStdDev                The StdDev individual.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMAStdDevIndividual(SQ_EWMAControlChart pEWMAControlChart, float* pfStdDev);
   /**
   *  Get the EWMA StdDev from the EWMA control chart.
   *  MSSD, Mean Square Successive Difference = SAvg * (l/(2-l))1/2. S(EWMA) in SIMCA.
   *
   *  @param[in]  pEWMAControlChart       The EWMA control chart.
   *  @param[out] pfStdDev                The EWMA StdDev.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMAStdDev(SQ_EWMAControlChart pEWMAControlChart, float* pfStdDev);
   /**
   *  Get the StdDev Process from the EWMA control chart.
   *  If not User Entered, estimated from Avg. S(RAvg) or S(SAvg) in SIMCA.
   *
   *  @param[in]  pEWMAControlChart       The EWMA control chart.
   *  @param[out] pfStdDev                The StdDev Process.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEWMAStdDevProcess(SQ_EWMAControlChart pEWMAControlChart, float* pfStdDev);
   /**
   *  Get the CUSUM control chart.
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the CUSUM control chart from.
   *  @param[in]  iSampleSize    Must be smaller than or equal to half of the number of observations (the size of pData divided in two)
   *                             and between 1 to 25.
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[out] pCusum         A pointer to the CUSUM results.
   *  @return                    Returns SQ_E_OK if success or an error code
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumControlChart(SQ_Project pProject, SQ_FloatVector pData, int iSampleSize, float fTarget, float fStdDev, SQ_CusumControlChart *pCusum);

   /**
   *  Get the CUSUM control chart  using a range variable.
   *  Control Charts are available for any project, except for an observation level batch project.
   *
   *  @param[in]  pProject       The project to use
   *  @param[in]  pData          The data to get the CUSUM control chart from.
   *  @param[in]  pRangeData     The vector to create the grouping from. Should be of equal length as pData
   *  @param[in]  fRange         A range in the range data vector
   *  @param[in]  fTarget        Set to MissingValue if target will be estimated.
   *  @param[in]  fStdDev        Set to MissingValue if stddev will be estimated.
   *  @param[out] pCusum         A pointer to the CUSUM results.
   *  @return                    Returns SQ_E_OK if success or an error code
   *  @see SQ_GetMissingValue
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumRangeControlChart(SQ_Project pProject, SQ_FloatVector pData, SQ_FloatVector pRangeData, float fRange, float fTarget, float fStdDev, SQ_CusumControlChart *pCusum);

   /**
   *  Removes the Cusum Control Chart, this function must be called for every Cusum Control Chart object that is created.
   *  @see tagSQ_CusumControlChart
   *
   *  @param[in]  pCusum   The Cusum Control Chart object to be removed.
   *  @return              Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearCusumControlChart(SQ_CusumControlChart* pCusum);
   /**
   *  Get the DevFromTarget vector from the CUSUM control chart.
   *  Deviation from target = average of subgroup  target.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pDevFromTarget          The EWMA of samples of the selected item.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumDevFromTarget(SQ_CusumControlChart pCusumControlChart, SQ_FloatVector* pDevFromTarget);
   /**
   *  Get the high CUSUM vector from the CUSUM control chart.
   *  Cumulative sum on the high side difference used to detect a deviation from the target on the high side. 
   *  High CUSUM is set to zero when negative.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pHighCuSum              The high CUSUM.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumHigh(SQ_CusumControlChart pCusumControlChart, SQ_FloatVector* pHighCuSum);
   /**
   *  Get the low CUSUM vector from the CUSUM control chart.
   *  Cumulative sum on the low side difference used to detect a deviation from the target on the low side. 
   *  Low CUSUM is set to zero when negative.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pLowCuSum               The low CUSUM.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumLow(SQ_CusumControlChart pCusumControlChart, SQ_FloatVector* pLowCuSum);
   /**
   *  Get the action limit from the CUSUM control chart.
   *  4.5 standard deviation, using S(Mxx) when sample size = 1 and SAvg when sample size > 1.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pActionLimit            The action limit.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumActionLimit(SQ_CusumControlChart pCusumControlChart, float* pActionLimit);
   /**
   *  Get the dead band from the CUSUM control chart.
   *  Dead Band (K) (or allowable slack) = standard deviation/2, using S(Mxx) when sample size = 1 and SAvg when sample size > 1.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pDeadBand               The dead band.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumDeadBand(SQ_CusumControlChart pCusumControlChart, float* pDeadBand);
   /**
   *  Get the target from the CUSUM control chart.
   *  If not User Entered, estimated as the grand average of the selected item over the training set. Target(MModel number) in SIMCA.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pTarget                 The target.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumTarget(SQ_CusumControlChart pCusumControlChart, float* pTarget);
   /**
   *  Get the StdDev individual from the CUSUM control chart.
   *  Process standard deviation over all observations. S(MModel number) in SIMCA.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pStdDev                 The StdDev individual.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumStdDevIndividual(SQ_CusumControlChart pCusumControlChart, float* pStdDev);
   /**
   *  Get the StdDev process from the CUSUM control chart.
   *  If not User Entered, estimated from Avg. S(RAvg) or S(SAvg) in SIMCA.
   *
   *  @param[in]  pCusumControlChart      The CUSUM control chart.
   *  @param[out] pStdDev                 The StdDev process.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCusumStdDevProcess(SQ_CusumControlChart pCusumControlChart, float* pStdDev);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQCONTROLCHARTS_H_ */
