/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQDATASET_H_
#define _SQDATASET_H_

#include "SQDef.h"

/** @file SQDataset.h

   This file list the SQ_Dataset object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQStringVector.h"
#include "SQIntVector.h"
#include "SQVectorData.h"
#include "SQVariable.h"

   /**
   The dataset handle used to identify an opened dataset.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_Dataset 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Dataset;

   /**
   An enum defining the different dataset types.
   */
   typedef enum SQ_DatasetTypeEnum
   {
      SQ_D_Standard=0,        /**< The dataset is a standard dataset. */
      SQ_D_Spectral,          /**< The dataset is a spectral dataset. */
      SQ_D_Hierarchical,      /**< The dataset is a hierarchical dataset. */
      SQ_D_PLSDA,             /**< The dataset is a PLS-DA dataset. */
      SQ_D_BatchLevel,        /**< The dataset is a batch level dataset. */
      SQ_D_ObservationLevel,  /**< The dataset is an observation level batch dataset. */
      SQ_D_BatchCondition,    /**< The dataset is a batch condition dataset. */
	  SQ_D_LagDistance        /**< dataset for dynamic lags in a model. */
   } SQ_DatasetType;

   /**
   The handle to identify a vector of dataset types.
   */
   typedef struct tagSQ_DatasetTypeVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_DatasetTypeVector;

   /**
   *  Returns the size of the dataset types vector.
   *
   *  @param[in]  pVector        The vector to get the size of.
   *  @param[out] piSize         The size of the vector or a negative number if an error occurred.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSizeOfDatasetTypeVector(const SQ_DatasetTypeVector pVector, int* piSize);

   /**
   *  Gets the type of dataset.
   *
   *  @param[in]  pVector        The vector to get the data from.
   *  @param[in]  iPos           The position to get the data from.
   *  @param[in]  pDatasetType   The DatasetType.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromDatasetTypeVector(const SQ_DatasetTypeVector pVector, int iPos, SQ_DatasetType* pDatasetType);

   /*******************************************************************************************
   *
   *                       Functions for the datasets
   *
   *  All these functions take a SQ_Dataset as parameter. 
   *  The SQ_Dataset object is retrieved from SQ_GetDataset.
   *  Information from these functions comes from the datasets. 
   *  All functions return an error code that indicates whether the function succeeded or failed.
   *
   ********************************************************************************************/

   /**
   *  Retrieves the name of a dataset.
   *
   *  @param[in]     pDataset       The dataset to use, UTF-8 encoded.
   *  @param[in,out] szDatasetName  The name of the dataset. The user is responsible to allocate and deallocate the buffer.
   *  @param[in]     iBufferLength  The length of the buffer.
   *
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetName(SQ_Dataset pDataset, char *szDatasetName, int iBufferLength);

   /**
   *  Retrieves the types of a dataset.
   *
   *  @param[in]   pDataset         The dataset to use
   *  @param[out]  pTypeVector      The types of dataset.
   *
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetTypes(SQ_Dataset pDataset, SQ_DatasetTypeVector* pTypeVector);
   /**
   *  Retrieves the number of different variable IDs in the dataset.
   *  If there exist both a primary ID and one level of secondary variable IDs,
   *  this equals to 2 variable IDs. 
   *
   *  @param[in]   pDataset         The dataset to use
   *  @param[out]  piNrVarIDs       The number of variable IDs.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfVariableIDs(SQ_Dataset pDataset, int *piNrVarIDs);
   /**
   *  Retrieves the number of different observations IDs in the dataset.
   *  If there exist both a primary ID and one level of secondary observation IDs,
   *  this equals to 2 observation IDs. 
   *
   *  @param[in]   pDataset         The dataset to use
   *  @param[out]  piNrObsIDs       The number of observation IDs.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfObservationIDs(SQ_Dataset pDataset, int *piNrObsIDs);
   /**
   *  Retrieves the names of the variables in the given data set.
   *
   *  @param[in]  pDataset          The dataset to use
   *  @param[out] pVariableVector   The variables included in the dataset.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetVariableNames(SQ_Dataset pDataset, SQ_VariableVector *pVariableVector);
    /**
   *  Retrieves the original names of the variables in the given data set, before any processing.
   *  This function is only available for spectral datasets.
   *
   *  @param[in]  pDataset          The dataset to use
   *  @param[out] pVariableVector   The original variables included in the dataset.
   *  @return                       Returns SQ_E_OK if success or an error code
   *  @see GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetOriginalDataSetVariableNames(SQ_Dataset pDataset, SQ_VariableVector *pVariableVector);
   /**
   *  Retrieves the names of the variable IDs in the given data set.
   *
   *  @param[in]     pDataset          The dataset to use
   *  @param[in]     iNameIDIndex      The index of the Variable ID. 
   *                                   1 for the primary ID in the data set,
   *                                   2 for the first level of secondary ID, 3 for the second level of secondary ID and so on.
   *  @param[in,out] pszVariableIDName The name of the variable ID (i.e. "Primary ID", "Secondary ID" etc.), UTF-8 encoded.
   *                                   The user is responsible to allocate and deallocate the buffer.
   *  @param[in]     iBufferLength     The length of the buffer.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   *  @see SQ_GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetVariableIDName(SQ_Dataset pDataset, int iNameIDIndex, char *pszVariableIDName, int iBufferLength);
   /**
   *  Retrieves the names of the observations in the given data set.
   *
   *  @param[in]  pDataset             The dataset to use
   *  @param[in]  iNameIDIndex         The index of the Observation ID. 
   *                                   1 for the primary ID in the data set,
   *                                   2 for the first level of secondary ID, 3 for the second level of secondary ID and so on.
   *  @param[out] pObservationNames    The names of the observations.
   *                                   The vector must be cleared with SQ_ClearStringVector.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   *  @see SQ_GetNumberOfObservationIDs
   */   
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetObservationNames(SQ_Dataset pDataset, int iNameIDIndex, SQ_StringVector *pObservationNames);
   /**
   *  Retrieves the names of the observation IDs in the given data set.
   *
   *  @param[in]     pDataset                The dataset to use
   *  @param[in]     iNameIDIndex            The index of the Observation ID. 
   *                                         1 for the primary ID in the data set,
   *                                         2 for the first level of secondary ID, 3 for the second level of secondary ID and so on.
   *  @param[in,out] pszObservationIDName    The name of the observation ID, UTF-8 encoded.
   *                                         The user is responsible to allocate and deallocate the buffer.
   *  @param[in]     iBufferLength           The length of the buffer.
   *
   *  @return                                Returns SQ_E_OK if success or an error code
   *  @see SQ_GetNumberOfObservationIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetObservationIDName(SQ_Dataset pDataset, int iNameIDIndex, char *pszObservationIDName, int iBufferLength);
   /**
   *  Retrieves the data for the observations in the given data set (ObsDS).
   *  If a variable is qualitative the index of the qualitative settings will be returned in the matrix.
   *  If the project option is set to reconstruct wavelet compression the data returned will reflect that.
   *
   *  @param[in]  pDataset          The dataset to use
   *  @param[in]  pObservations     A list of observation Indices to use.
   *                                1 for observation 1 in the data set, 2 for observation 2 and so on.
   *                                NULL if all observations in the data set should be used.
   *  @param[out] pVectorData       A pointer to the ObsDS vector data.
   *                                Number of rows in vector data = number of observations chosen (length of pObservations).
   *                                Number of columns in vector data = number of variables in the dataset.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetObservations(SQ_Dataset pDataset, SQ_IntVector* pObservations, SQ_VectorData* pVectorData);
   /**
   *  Retrieves the data for the variables in the given data set (VarDS).
   *  If a variable is qualitative the index of the qualitative settings will be returned in the matrix.
   *
   *  @param[in]  pDataset          The dataset to use
   *  @param[in]  pVariables        A list of variable indices to use.
   *                                1 for variable 1 in the data set, 2 for variable 2 and so on.
   *                                NULL if all variables in the data set should be used.
   *  @param[out] pVectorData       A pointer to the VarDS vector data.
   *                                Number of rows in vector data = number of variables chosen (length of pObservations).
   *                                Number of columns in vector data = number of observations in the dataset.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataSetVariables(SQ_Dataset pDataset, SQ_IntVector* pVariables, SQ_VectorData* pVectorData);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQDATASET_H_ */
