/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQERRORCODES_H_
#define _SQERRORCODES_H_

#include "SQDef.h"

/** @file SQErrorCodes.h

   This file list error codes returned by functions in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif



   /**
   An enum defining the different error codes.
   */
   typedef enum SQ_ErrorCodeEnum
   {
      SQ_E_OK=0,                    /**< (0) Success */
      SQ_E_EXCEPTION,               /**< (1) An unknown exception occurred. Look in the log file for more information. */
      SQ_E_INVALIDPROJECTHANDLE,    /**< (2) The project handle supplied is invalid. */
      SQ_E_COULDNOTOPENPROJECT,     /**< (3) Could not open the project. */
      SQ_E_COULDNOTCLOSEPROJECT,    /**< (4) Could not close the project. */
      SQ_E_FILEDOESNOTEXIST,        /**< (5) The given file does not exist on disk. */
      SQ_E_INVALIDPASSWORD,         /**< (6) The password for the encrypted project is invalid. */
      SQ_E_INVALIDVERSION,          /**< (7) Invalid project version. */
      SQ_E_INVALIDINPARAMETER,      /**< (8) A parameter passed to the function is not valid. */
      SQ_E_INVALIDMODELHANDLE,      /**< (9) The model handle supplied is invalid. */
      SQ_E_INVALIDDATASETHANDLE,    /**< (10) The dataset handle supplied is invalid. */
      SQ_E_INVALIDCCHANDLE,         /**< (11) The control chart handle supplied is invalid. */
      SQ_E_INVALIDIDINDEX,          /**< (12) The ID index supplied is invalid. */
      SQ_E_RUSPNOTALLOWED,          /**< (13) RUSP files are not supported. */
      SQ_E_MODELNOTFOUND,           /**< (14) The model was not found, probably incorrect model number. */
      SQ_E_INVALIDHANDLE,           /**< (15) The handle supplied is invalid. */
      SQ_E_OUTOFMEMORY,             /**< (16) Could not allocated any more memory. */
      SQ_E_MODELNOTFITTED,          /**< (17) The model is not fitted and can not be used as an argument */
      SQ_E_MUSTBEPREDICTIVEMODEL,   /**< (18) The model must be a predictive model, like PLS, OPLS etc. */
      SQ_E_MUSTBEPCAMODEL,          /**< (19) The model must be a PCA model. */
      SQ_E_MUSTBECLASSORDAMODEL,    /**< (20) The model must be a class or DA model. */
      SQ_E_MUSTBEOPLSMODEL,         /**< (21) The model must be any OPLS model (OPLS, O2PLS, OPLS-Class etc.) */
      SQ_E_DATASETNOTFOUND,         /**< (22) The dataset was not found, probably incorrect dataset number. */
      SQ_E_NOTAQUALITATIVEVARIABLE, /**< (23) The variable is not a qualitative variable. */
      SQ_E_NOTASPECTRADATASET,      /**< (24) The dataset is not a spectra dataset. */
      SQ_E_COMPONENTZERONOTALLOWED, /**< (25) TComponent 0 is not valid for this vector. */
      SQ_E_INVALIDCOMPONENTNUMBER,  /**< (26) Invalid component number. */
      SQ_E_ONLYLASTCOMPONENTORZERO, /**< (27) This vector is only available for component 0 and the last component for O2PLS models. */
      SQ_E_ONLYLASTCOMPONENT,       /**< (28) This vector is only available for the last component for OPLS/O2PLS models. */
      SQ_E_INVALIDINDEX,            /**< (29) The given index in invalid. */

      SQ_E_LICENSEFILENOTVALID,     /**< (30) The license file is not valid or can not be found. */
      SQ_E_OEMPASSWORDINCORRECT,    /**< (31) The OEM password is incorrect. */
      SQ_E_ACTIVATIONCODEINCORRECT, /**< (32) Activation code is incorrect. */
      SQ_E_NOXCOMPONENTS,           /**< (33) The model must have X side components. */
      SQ_E_NOYCOMPONENTS,           /**< (34) The model must have Y side components. */
      SQ_E_INVALIDPRODUCT,          /**< (35) The product is invalid. */
      SQ_E_INVALIDBATCHPROJECTHANDLE,  /**< (36) The batch project handle is invalid. */
      SQ_E_INVALIDBATCHMODELHANDLE,    /**< (37) The batch model handle is invalid. */
      SQ_E_INVALIDBEMHANDLE,        /**< (38) The batch evolution model handle is invalid. */
      SQ_E_INVALIDBLMHANDLE,        /**< (39) The batch level model handle is invalid. */
      SQ_E_INVALIDBATCHINDEX,        /**< (40) The batch index is invalid. */
      SQ_E_INVALIDPHASEINDEX,        /**< (41) The phase index is invalid. */
      SQ_E_MUSTBEBEM,                /**< (42) The function is only available for a Batch Evolution Model. */

      SQ_E_NOVALIDCOMPONENTSFOUND,   /**< (43) The model doesn't have any valid components. */
      SQ_E_DATASETEMPTY,             /**< (44) The dataset or predictionset is empty. */
      SQ_E_INVALIDDIMENSIONS,        /**< (45) Invalid number of dimensions in the array/matrix. */
      SQ_E_INVALIDARRAYSIZE,         /**< (46) The array size is not valid. */
      SQ_E_INVALIDARRAYTYPE,         /**< (47) The array is of invalid type. */
      SQ_E_INVALIDARRAYINDEX,        /**< (48) Invalid array index. */

      SQ_E_FILEREADER_INVALIDHANDLE,      /**< (49) The filereader handle supplied is invalid. */
      SQ_E_FILEREADER_INVALIDPROCADDRESS, /**< (50) Could not obtain the address of exported functions in the UmFileReader.dll. */
      SQ_E_FILEREADER_LOADFILEERROR,      /**< (51) Could not load file. */

      SQ_E_PROJECTNOTENCRYPTED,        /**< (52) The project file is not encrypted. */
      SQ_E_PROJECTALREADYENCRYPTED,    /**< (53) The project has already been encrypted. */
      SQ_E_COULDNOTENCRYPTPROJECT,     /**< (54) Could not encrypt the project file. */

      SQ_E_INVALIDDATASETNUMBER,       /**< (55) The dataset number is not valid */
      SQ_E_NAMESNOTUNIQUE,             /**< (56) Variable names are not unique. */

      SQ_E_IMPORT_INVALIDHANDLE,       /**< (57) The import handle supplied is invalid. */
      SQ_E_PROJECT_COULDNOTSAVE,       /**< (58) Could not save the project. */
      SQ_E_DATASET_COULDNOTCREATE,     /**< (59) Could not create the dataset. */
      SQ_E_LICESERESTRICTIONS,         /**< (60) A function call has been restricted by the license. */

      SQ_E_NOTACLASSMODEL,             /**< (61) The function requires that the model is a class model. */
      SQ_E_NOCLASSESDEFINED,           /**< (62) No classes has been defined. */

      SQ_E_NOMORECOMPONENTSAVAILABLE,  /**< (63) Could not calculate any more components. */
      SQ_E_OPLSNOTSUPPORTED,           /**< (64) The function is not valid for OPLS models. */
      SQ_E_ONLYOPLSSUPPORTED,          /**< (65) The function is only valid for OPLS models. */
      SQ_E_NOCOMPONENTS,               /**< (66) There are no components. */

      SQ_E_MODELISFITTED,              /**< (67) Trying to use a function on a fitted model that requires an unfitted model. */
      SQ_E_CANNOTSETMODELTYPE,         /**< (68) The model type can not be set for this model. */

      SQ_E_INCORRECTFILTERTYPE,        /**< (69) A filter with of the given type can not be created. */

      SQ_E_NOTENOUGHXVARIABLES,        /**< (70) The filter does not have enough X variables to perform the filtering. */
      SQ_E_NOTENOUGHYVARIABLES,        /**< (71) The filter does not have enough Y variables to perform the filtering. */
      SQ_E_NOTENOUGHOBSERVATIONS,      /**< (72) The filter does not have enough observations to perform the filtering. */

      SQ_E_OLDSPECTRALPROJECT_NOTSUPPORTED, /**< (73) Spectral projects created in previous versions of SIMCA are not supported for prediction. */
      SQ_E_MUSTBEBLM,                  /**< (74) The function is only available for a Batch Level Model. */

      SQ_E_ONLY_WHEN_INITIALIZING,     /**< (75) The function can only be called before the first project has been opened. */

      SQ_E_ERROR_LAST                     /* For internal use only. Must be the last enum value */
      
   } SQ_ErrorCode;

/**
   Macro to test if a function succeeded.
*/
#define SQ_SUCCEEDED(x)          (x == SQ_E_OK)

/**
   Macro to test if a function failed.
*/
#define SQ_FAILED(x)             (x != SQ_E_OK)

/**
*  Retrieves a message string describing the latest error that occurred.
*
*  @param[in]     eCode             The code to get the error description for
*  @param[in,out] szErrorString     The buffer to receive the error message, UTF-8 encoded.
*  @param[in]     iBufferLen        The size of the buffer
*  @return                          Returns SQ_E_OK if success or an error code
*/
SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetErrorDescription(SQ_ErrorCode eCode, char* szErrorString, int iBufferLen);

/**
*  Retrieves a message string describing the latest error that occurred.
*
*  @param[in,out] szErrorString     The buffer to receive the error message, UTF-8 encoded.
*  @param[in]     iBufferLen        The size of the buffer
*  @return                          Returns SQ_E_OK if success or an error code
*/
SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLastError(char* szErrorString, int iBufferLen);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQERRORCODES_H_ */
