/************************************************************************/
/*                                                                      */
/* Copyright (C) Sartorius Stedim Data Analytics AB 2017 -              */
/************************************************************************/

#ifndef SQ_FILEREADER_H
#define SQ_FILEREADER_H

#include "SQDef.h"
#include "SIMCAQP.h"

/** @file SQFileReader.h

   This file list the SQ_FileReader object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif

#ifdef _WIN32
#pragma pack(push, 8)
#endif

   /** \addtogroup FileReader SIMCA-Q File Reader 
    *  @{
    */
   ///  This is the content of the SIMCA-Q File Reader.
   
   /**
   *  The handle used to identify a FileReader object.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_FileReaderHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_FileReader;

   /**
   *  The specification of where to find the information in the file.
   */
   typedef struct tagSQ_FileReader_Specification
   {
      int miPrimaryVariableIDRow;      /**< The number of the row where to find the Primary variable IDs. */
      int miFirstDataRow;              /**< The number of the row where to find the first row of data. */
      int miFirstDataColumn;           /**< The number of the column where to find the first column of data. */
      SQ_IntVector* mpvecExcludedRows; /**< Indexes of the rows to exclude, if all rows should be included, set this to be a vector with -1 as only entry. */
   } SQ_FileReader_Specification;

   /**
   *  The options on how to open the file.
   */
   typedef struct tagSQ_FileReader_Options
   {
      SQ_IntVector* mpSheets;             /**< Excel: The sheets in the file to import. */
      int mbMergeVertically;              /**< Excel: If several sheets are imported,
                                          set this to 1 if they should be merged vertical and
                                          0 if they should be merged horizontally. */
      const char* mszDelimiter;           /**< Text: The delimiter in the file if such is used.*/
      int mbIgnoreConsecutiveDelimiters;  /**< Text: If consecutive delimiters should be ignored,
                                          this members should be 1, 0 if they should be included.*/
      SQ_IntVector* mpDataBlocks;         /**< JCAMP: Vector with the data blocks to import from the file/s.*/
      int mbIncludeXAsAnObservation;      /**< Set to 1 if the ordinal values should be included as an observation.*/
      int mbIncludeAllObservations;       /**< Set to 1 if all observations should be included, 0 to reduce the dataset.*/
      int miNumberOfIncludedIntervals;    /**< If the dataset should be reduced, this is the number of intervals to include. */
      SQ_IntVector* mpIncludeXStartPt;    /**< If the dataset should be reduced, this is starting values of the intervals,
                                          should be the same size as iNumberOfIncludedIntervals. */
      SQ_IntVector* mpIncludeXEndPt;      /**< If the dataset should be reduced, this is end values of the intervals,
                                          should be the same size as iNumberOfIncludedIntervals */
      int miAveragePoints;                /**< If set the average of these points should be used.*/
      char* mszUserDefinedOptions;        /**< If the file to import is defined in a plug-in dll,
                                          the options should be applied here. */
      char* mszEncoding;                  /**< Encoding to use, NULL or "" for local code page. "ibm420", "iso88596", "windows1256", "iso88592", "windows1250", "gb18030",
                                             "iso2022cn", "big5", "iso88595",  "koi8r", "windows1251", "iso88597", "windows1253", "ibm424", "iso88598", "windows1255",
                                             "eucjp", "iso2022jp", "shiftjis", euckr", "iso2022kr", iso88599", "windows1254", "utf16be", "utf16le", "utf32be", "utf32le",
                                             "utf8", "cp850", "iso88591", "windows1252" */
   } SQ_FileReader_Options;

   /**
   *  Loads and reads one or more files containing data that should be used to perform a prediction on,
   *  a handle is returned that should be used to get the results.
   *  SQ_FileReader_ReleaseFileHandle should be called to remove the handle.
   *
   *  @param[in]  pInputFiles          The file/s that contains the data,
   *                                   only file formats that support multiple files can be used with more than one file name.
   *  @param[in]  szPluginPath         If the file is of a format that is implemented as a plug-in,
   *                                   this should be the path to the plug-in, UTF-8 encoded.
   *  @param[in]  pOptions             Options for the different file types, @see SQ_FileReader_Options.
   *  @param[out] pFileHandle          The handle of the FileReader.
   *  @return                          Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_OpenFileReader(SQ_StringVector pInputFiles, const char* szPluginPath, SQ_FileReader_Options* pOptions, SQ_FileReader* pFileHandle);

   /**
   *  Releases the FileReader handle returned by SQ_OpenFileReader(), this function must be called for every handle that is created.
   *
   *  @param[in]  pFileHandle          The handle to be released.
   *  @return                          Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CloseFileReader(SQ_FileReader* pFileHandle);

   /**
   *  Predicts the imported file with the given options, a handle is returned that should be used to get the results.
   *  ReleaseHandle should be called to remove the handle.
   *
   *  @param[in]  pModel               The model handle to use.
   *  @param[in]  pFileHandle          The handle of the file where the predictionset is stored.
   *  @param[in]  pSpecification       The specification of the file, @see SQ_FileReader_Specification.
   *  @param[out] pPredHandle          The prediction handle that should be used to retrieve the prediction results.   
   *                                   Must be released by SQ_ClearPrediction.
   *  @return                          Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_FileReader_PredictFromFile(SQ_Model pModel, SQ_FileReader pFileHandle, SQ_FileReader_Specification* pSpecification, SQ_Prediction* pPredHandle);

   /** @}*/ // End of SQ File Reader

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif
