/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQFLOATMATRIX_H_
#define _SQFLOATMATRIX_H_

#include "SQDef.h"

/** @file SQFloatMatrix.h

   This file list the SQFloatMatrix object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"


   /**
   A float matrix that is used to get and send data to many functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_FloatMatrix 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_FloatMatrix;

   /**
   *  Initialize a float matrix with its size.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to be initialized.
   *  @param[in]   iRow          Number of rows to give the matrix.
   *  @param[in]   iColumn       Number of columns to give the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitFloatMatrix(SQ_FloatMatrix* pMatrix, int iRow, int iColumn);

   /**
   *  Get a copy of the float matrix
   *  @see tagSQ_BoolVector
   *
   *  @param[in]   pInMatrix        The vector to copy.
   *  @param[in, out]   pOutMatrix  The result vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyFloatMatrix(SQ_FloatMatrix pInMatrix, SQ_FloatMatrix* pOutMatrix);

   /**
   *  Removes the matrix, this function must be called for every matrix
   *  that has been initialized with SQ_InitFloatMatrix.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearFloatMatrix(SQ_FloatMatrix* pMatrix);
   /**
   *  Sets the data in the given cell.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the data in.
   *  @param[in]   iRow          The row to set the data in.
   *  @param[in]   iColumn       The column to set the data in.
   *  @param[in]   fVal          The data to set the cell with.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataInFloatMatrix(SQ_FloatMatrix pMatrix, int iRow, int iColumn, float fVal);
   /**
   *  Gets the data from a given cell.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the data from.
   *  @param[in]   iRow          The row to get the data from.
   *  @param[in]   iColumn       The column to get the data from.
   *  @param[out]   pfVal        The data in the cell. A pointer to where the data should be stored
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromFloatMatrix(SQ_FloatMatrix pMatrix, int iRow, int iColumn, float *pfVal);
   
   /**
   *  Sets the data in the given row.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the data in.
   *  @param[in]   iRow          The row to set the data in.
   *  @param[in]   pfVals        An array of data to set the row with. Must be of same length as number of columns
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetRowDataInFloatMatrix(SQ_FloatMatrix pMatrix, int iRow, const float* pfVals);

   /**
   *  Sets the data in the given column.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the data in.
   *  @param[in]   iColumn       The column to set the data in.
   *  @param[in]   pfVals        An array of data to set the column with. Must be of same length as number of rows
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetColumnDataInFloatMatrix(SQ_FloatMatrix pMatrix, int iColumn, const float* pfVals);

   /**
   *  Sets the data in the matrix.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the data in.
   *  @param[in]   pfVals        A matrix of data to fill the FloatMatrix. Must be of same size as the FloatMatrix. The
   *                             format is Row 1, Row 2, etc.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetFloatMatrix(SQ_FloatMatrix pMatrix, const float* pfVals);
   /**
   *  Gets the data from a given row.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the data from.
   *  @param[in]   iRow          The row to get the data from.
   *  @param[out]  pfVals        The data in given row. Must be of same length as number of columns
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRowDataFromFloatMatrix(SQ_FloatMatrix pMatrix, int iRow, float *pfVals);

   /**
   *  Gets the data from a given column.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the data from.
   *  @param[in]   iColumn       The column to get the data from.
   *  @param[in]   pfVals        The data in given column. Must be of same length as number of rows
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnDataFromFloatMatrix(SQ_FloatMatrix pMatrix, int iColumn, float *pfVals);

   /**
   *  Gets the data from the matrix.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the data from.
   *  @param[out]  pfVals        A pointer to a matrix to be filled with data from the FloatMatrix. Must be of same size as the FloatMatrix The
   *                             format is Row 1, Row 2, etc.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetFloatMatrix(SQ_FloatMatrix pMatrix, float* pfVals);

   /**
   *  Returns the number of rows in the matrix.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the number of rows from.
   *  @param[out]   piNumRows    The number of rows in the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumRowsInFloatMatrix(SQ_FloatMatrix pMatrix, int* piNumRows);
   /**
   *  Returns the number of columns in the matrix.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the number of columns from.
   *  @param[out]   piNumCols    The number of columns in the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumColumnsInFloatMatrix(SQ_FloatMatrix pMatrix, int* piNumCols);
   /**
   *  Transpose the matrix. This can only be done if the float matrix does not come from a VectorData object.
   *  If it is from a VectorData object, it must be extracted with the GetDataMatrixCopy function.
   *  @see tagSQ_FloatMatrix
   *
   *  @param[in]   pMatrix       The matrix to transpose.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_TransposeFloatMatrix(SQ_FloatMatrix pMatrix);




#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQFLOATMATRIX_H_ */
