/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQFLOATVECTOR_H_
#define _SQFLOATVECTOR_H_

#include "SQDef.h"

/** @file SQFloatVector.h

   This file list the SQ_FloatVector object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"

   /**
   An int vector that is used to get and send data to many functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_FloatVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_FloatVector;

      /**
   *  Initialize an integer vector with its size.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to be initialized.
   *  @param[in]   iSize          The size of the vector.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitFloatVector(SQ_FloatVector* pVector, int iSize);

   /**
   *  Get a copy of the float vector
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pInVector        The vector to copy.
   *  @param[in, out] pOutVector    The result vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyFloatVector(SQ_FloatVector pInVector, SQ_FloatVector* pOutVector);

   /**
   *  Removes the vector, this function must be called for every vector that is created.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to be removed.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearFloatVector(SQ_FloatVector* pVector);
   /**
   *  Sets the data at the given position.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to set the data in.
   *  @param[in]   iPos           The position to set the data in.
   *  @param[in]   fVal           The data to set the cell with.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataInFloatVector(SQ_FloatVector pVector, int iPos, float fVal);
   /**
   *  Gets the data from a given position.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to get the data from.
   *  @param[in]   iPos           The position to get the data from.
   *  @param[out]  pfVal          The data in the cell.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromFloatVector(const SQ_FloatVector pVector, int iPos, float* pfVal);

   /**
   *  Sets the data in the vector.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to set the data in.
   *  @param[in]   pfVals         An array of data to fill the vector with. Must be at least of same size as the vector.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetFloatVector(SQ_FloatVector pVector, const float* pfVals);
   /**
   *  Gets the data from the vector.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to get the data from.
   *  @param[out]  pfVals         An array to be filled with the data from the vector. Must be at least of same size as the vector.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetFloatVector(const SQ_FloatVector pVector, float* pfVals);
   /**
   *  Returns the size of the vector.
   *  @see tagSQ_FloatVector
   *
   *  @param[in]   pVector        The vector to get the number of columns from.
   *  @param[out]  piSize         The size of the vector or a negative number if an error occurred.
   *  @return                     Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetFloatVectorSize(const SQ_FloatVector pVector, int* piSize);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQFLOATVECTOR_H_ */
