/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQINTVECTOR_H_
#define _SQINTVECTOR_H_

#include "SQDef.h"

/** @file SQIntVector.h

   This file list the SQ_IntVector object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"

   /**
   An int vector that is used to get and send data to many functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_IntVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_IntVector;

   /**
   *  Initialize an integer vector with its size.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to be initialized.
   *  @param[out] iSize          The size of the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitIntVector(SQ_IntVector* pVector, int iSize);

   /**
   *  Get a copy of the int vector
   *  @see tagSQ_IntVector
   *
   *  @param[in]   pInVector        The vector to copy.
   *  @param[in, out] pOutVector    The result vector.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyIntVector(SQ_IntVector pInVector, SQ_IntVector* pOutVector);
   /**
   *  Removes the vector, this function must be called for every vector that is created.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearIntVector(SQ_IntVector* pVector);
   /**
   *  Sets the data at the given position.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to set the data in.
   *  @param[in]  iPos           The position to set the data in.
   *  @param[in]  iVal           The data to set the cell with.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataInIntVector(SQ_IntVector pVector, int iPos, int iVal);
   /**
   *  Gets the data from a given position.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to get the data from.
   *  @param[in]  iPos           The position to get the data from.
   *  @param[in]  piVal          The data in the cell.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataFromIntVector(const SQ_IntVector pVector, int iPos, int* piVal);

   /**
   *  Sets the data in the vector.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to set the data in.
   *  @param[in]  piVals         An array of data to fill the vector with. Must be at least of same size as the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetIntVector(SQ_IntVector pVector, const int* piVals);
   /**
   *  Gets the data from the vector.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to get the data from.
   *  @param[in]  piVals         An array to be filled with the data from the vector. Must be at least of same size as the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetIntVector(const SQ_IntVector pVector, int* piVals);
   /**
   *  Returns the size of the vector.
   *  @see tagSQ_IntVector
   *
   *  @param[in]  pVector        The vector to get the size of.
   *  @param[out] piSize         The size of the vector or a negative number if an error occurred.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetIntVectorSize(const SQ_IntVector pVector, int* piSize);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQINTVECTOR_H_ */
