/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMDEFINES_H_
#define _SQMDEFINES_H_

#include "SQDef.h"

/** @file SQMDefines.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"

   /// @cond SQM 
   
   /************************************************************************/
   /* Struct for project options information                                  */
   /************************************************************************/

   /** 
   * Scale enum used in Project Options.
   */
   typedef enum SQ_POScaleTypesEnum
   {
      SQ_POScaleNone,           /**< No scaling */
      SQ_POScaleUV,             /**< Centered and scaled to Unit Variance. */
      SQ_POScaleUVN,            /**< Scaled to Unit Variance (not centered). */
      SQ_POScalePar,            /**< Centered and scaled to Pareto Variance. */
      SQ_POScaleParN,           /**< Scaled to Pareto Variance (not centered). */
      SQ_POScaleCtr             /**< Centered but not scaled. */
   } SQ_POScaleTypes;

   /**   
   * A struct that is used to set and get project options.
   */
   typedef struct tagSQ_ProjectOptions
   {
      int   iCrossValidationRounds;    /**< 0 means no cross validation */
      int   iMaxIterations;            /**< Maximum number of iterations */
      float fMissingValueToleranceObs; /**< As a fraction. 0.5 means 50% */
      float fMissingValueToleranceVar; /**< As a fraction. 0.5 means 50% */
      float fConfLevelForParameters;   /**< Confidence level for parameters. */
      float fSignificanceLevel;        /**< Significance level. 0.05 means 95% probability */
      int   iMinNumOfNonMedianPoints;  /**< Minimum number of non median points */
      SQ_POScaleTypes eDefaultXScale; /**< Default scale to be used for the x-variables */
      SQ_POScaleTypes eDefaultYScale; /**< Default scale to be used for the y-variables */
   } SQ_ProjectOptions;

   /** 
   * Base scaling type. 
   * For further information see the SIMCA manual and SQ_SetScaling.
   */
   typedef enum SQ_WSBaseScalingTypeEnum
   {
      SQ_WSScaleNone,           /**< No scaling */
      SQ_WSScaleUnitVariance,   /**< Scaled by the inverse of the standard deviation. */
      SQ_WSScalePareto,         /**< Scaled by the inverse of the square root of the standard deviation. */
      SQ_WSScaleFrozen,         /**< The scaling weight and offset is set to a fixed value. */
      SQ_WSScaleLag,            /**< Default scaling for lagged variables. Can only be used for lagged variables. */
      SQ_WSScaleExpanded,       /**< Scaling type for expanded variables. Expanded variables will always
                                 *   have this type and no other type of variable can be set to use
                                 *   this type of scaling. 
                                 */
      SQ_WSScalePercentOfCenter, /**< The inverse of a percentage of the mean. */
      SQ_WSScaleInverseStandardDeviation, /**< The scaling weight and center is set to a fixed value. */
      SQ_WSScaleStandardDeviation         /**< The scaling weight and center can be set to a fixed value. */
   } SQ_WSBaseScalingType;

   /**
   * Block scaling type.
   * For further information see the SIMCA manual and SQ_SetScaling.
   */
   typedef enum SQ_WSBlockScalingTypeEnum
   {
      SQ_WSBlockScaleNone,         /**< The variable is not block scaled. */
      SQ_WSBlockScaleSquareRoot,   /**< Scale by the inverse of the square root of the number of variables in the block */
      SQ_WSBlockScaleFourthRoot    /**< Scale by the inverse of the fourth root of the number of variables in the block */
   } SQ_WSBlockScalingType;

   /**
   * Centering types.
   * For further information see the SIMCA manual and SQ_SetScaling.
   */
   typedef enum SQ_WSCenteringEnum
   {
      SQ_WSCenterNone,    /**< Do not center the variable. */
      SQ_WSCenterAuto,    /**< Center by subtracting the arithmetic mean of the variable. */
      SQ_WSCenterFixed,   /**< Center by subtracting a fixed value. */
      SQ_WSCenterFrozen   /**< The center is a fixed value. */
   } SQ_WSCentering;

   /** 
   * Transform types.
   * For further information see the SIMCA manual and SQ_SetScaling.
   */
   typedef enum SQ_WSTransformTypeEnum
   {
      SQ_WSTransformNone,    /**< The variable is not transformed. */
      SQ_WSTransformLinear,  /**< Linear transform. a*X+b */
      SQ_WSTransformLog,     /**< Log10 transform. log(a*X+b) */
      SQ_WSTransformNegLog,  /**< Negative log10 transform. -log(a*X+b) */
      SQ_WSTransformLogIt,   /**< log10((X-a)/(b-X)) */
      SQ_WSTransformExp,     /**< Natural exponent. e^(a*X+b) */
      SQ_WSTransformPower    /**< (a*X+b)^C Note that */
   } SQ_WSTransformType;

   /**
   * Struct used to get and set the scaling of a variable.
   */
   typedef struct SQ_WSScaleInfoStruct
   {
      SQ_WSBaseScalingType     eBaseScalingType; /**< The base scaling type. If this parameter is 
                                                *   SQMWS_ScaleFrozen, SQMWS_ScaleStandardDeviation
                                                *   or SQMWS_ScaleInverseStandardDeviation 
                                                *   the base scaling weight is fScalingWeight
                                                *   and fCenter is the center
                                                *   and all other members are ignored. 
                                                *   If it is SQMWS_ScaleLag or SQMWS_ScaleExpanded
                                                *   all other members are ignored except 
                                                *   fScaleModifier, eBlockScalingType
                                                *   and iBlockScalingBlock.
                                                *   If the parameter is SQMWS_ScalePercentOfCenter
                                                *   the percent of center is the fScaleModifier
                                                *   and fCenter is the center
                                                *   and all other members are ignored. 
                                                *   Note that SQMWS_ScaleExpanded can not be used
                                                *   when setting the scaling weight of a variable.
                                                *   Expanded variables always have scaling type
                                                *   SQMWS_ScaleExpanded which can not be changed.
                                                *   Note also that SQMWS_ScaleLag can only be set
                                                *   for lagged non expanded variables.
                                                */
      float                   fScalingWeight;   /**< The base scaling weight if eBaseScalingType is 
                                                *   equal to SQMWS_ScaleFrozen, SQMWS_ScaleStandardDeviation
                                                *   or SQMWS_ScaleInverseStandardDeviation. 
                                                *   Otherwise this member is ignored.
                                                */
      SQ_WSCentering           eCentering;       /**< The centering type. If this parameter is 
                                                *   SQMWS_CenterFixed fCenter will be used 
                                                *   as the center value, otherwise it will be
                                                *   calculated from the data.
                                                */
      float                   fCenter;          /**< The center value if eCentering is equal to
                                                *   SQMWS_CenterFixed or if eBaseScalingType is
                                                *   equal to SQMWS_ScaleFrozen, SQMWS_ScalePercentOfCenter, 
                                                *   SQMWS_ScaleStandardDeviation or
                                                *   SQMWS_ScaleInverseStandardDeviation.  
                                                *   Otherwise this member is ignored.
                                                */
      float                   fScaleModifier;   /**< An additional modifier of the base scaling weight.
                                                *   This parameter is ignored if eBaseScalingType equals 
                                                *   SQMWS_ScaleFrozen. If eBaseScalingType is equal to
                                                *   SQMWS_ScalePercentOfCenter this is the percentage used.
                                                */
      SQ_WSBlockScalingType    eBlockScalingType;  /**< The block scaling type. If this parameter is not
                                                  *   equal to SQMWS_BlockScaleNone, iBlockScalingBlock 
                                                  *   is the block scaling block used for this variable.
                                                  */
      int                     iBlockScalingBlock; /**< The block scaling block if eBlockScalingType
                                                  *   is not equal to SQMWS_BlockScaleNone, otherwise
                                                  *   this member is ignored.
                                                  */
   } SQ_WSScaleInfo;

   /**
   * Allowed values for the exponent in power transform
   */
   typedef enum SQ_WSPowerExponentEnum
   {
      SQ_WSExpInvSquare,     /**< -2 */
      SQ_WSExpInverse,       /**< -1 */
      SQ_WSExpInvSquareRoot, /**< -0.5 */
      SQ_WSExpInvFourthRoot, /**< -0.25 */
      SQ_WSExpFourthRoot,    /**< 0.25 */
      SQ_WSExpSquareRoot,    /**< 0.5 */
      SQ_WSExpIdentity,      /**< 1 */
      SQ_WSExpSquare         /**< 2 */
   } SQ_WSPowerExponent;

   /**
   * Struct for setting and getting variable transform
   */
   typedef struct SQ_WSTransformInfoStruct
   {
      SQ_WSTransformType   eTransformType;   /**< The transform type */
      float                fConstant_a;      /**< The first constant used. See SQMWS_TransformPower. */
      float                fConstant_b;      /**< The second constant used. See SQMWS_TransformPower. */
      SQ_WSPowerExponent   eExponent;        /**< The exponent used in power transforms. 
                                             *   This member is ignored unless eTransformType equals
                                             *   SQMWS_TransformPower. */
   } SQ_WSTransformInfo;

   /**
   * Indicates how to divide observations into different groups.
   */
   typedef enum SQ_CVGroupByEnum
   {
      SQ_CVSimilar,   /**< Group similar observations in the same group. */
      SQ_CVDisSimilar,/**< Group dissimilar observations in the same group. */
      SQ_CVGroupNum  /**< Group observations with the same value in the same group. */
   } SQ_CVGroupBy;

   /**
   * Indicates what variables should be added to the workset from the hierarchical base model.
   */
   typedef enum SQ_HierarchicalEnum
   {
      SQ_HierarchicalScores,       /**< The scores will be added as variables. */
      SQ_HierarchicalResiduals,    /**< The residuals will be added as variables. */
      SQ_HierarchicalBoth          /**< Both scores and residuals will be added as variables. */
   } SQ_Hierarchical;

   /**
   * Types of spectral filters that can be used.
   */
   typedef enum SQ_FilterTypeEnum
   {
      // Spectral filters
      SQ_FilterMSC,      /**< A MSC filter. */
      SQ_FilterOSC,      /**< A OSC filter. */
      SQ_FilterSNV,      /**< A SNV filter. */
      SQ_FilterWCS,      /**< A WCS filter. */
      SQ_FilterDerivate, /**< A Derivate filter. */
      SQ_FilterSavitzky, /**< A Savitzky-Golay filter. */
      SQ_FilterEWMA,     /**< A EWMA filter. */
      SQ_FilterRowCenter, /**< A Row Center filter. */
   } SQ_FilterType;

   /**
   * Types of time series filters that can be used.
   */
   typedef enum SQ_TSFilterTypeEnum
   {
      // Time Series Filters
      SQ_TSFilterWDS,      /**< A WDS filter. */
      SQ_TSFilterWDTS,     /**< A WDTS filter. */
   } SQ_TSFilterType;

   /** 
   * Filter scaling type. 
   * For further information see the SIMCA manual and SQMF_SetScaling.
   */
   typedef enum SQ_FilterScalingTypeEnum
   {
      SQ_FilterScaleNone=1,         /**< No scaling */
      SQ_FilterScaleCenter,         /**< Scaled by the center */
      SQ_FilterScalePareto,         /**< Scaled by the inverse of the square root of the standard deviation. */
      SQ_FilterScaleUnitVariance    /**< Scaled by the inverse of the standard deviation. */
   } SQ_FilterScalingType;

   /**
   * Filter polynomial order.
   */
   typedef enum SQ_FilterPolynomialOrderEnum
   {
      SQ_FilterQuadratic=2, /**< Quadratic polynomial order */
      SQ_FilterCubic=3      /**< Cubic polynomial order */
   } SQ_FilterPolynomialOrder;
   
   /**
   * Derivate filter derivation order.
   */
   typedef enum SQ_FilterDerivateOrderEnum
   {
      SQ_FilterFirstOrder=1,  /**< First derivate */
      SQ_FilterSecondOrder=2, /**< Second derivate */
      SQ_FilterThirdOrder=3   /**< Third derivate */
   } SQ_FilterDerivateOrder;

   /**
   * Wavelet filter detrend mode.
   */
   typedef enum SQ_FilterDetrendModeEnum
   {
      SQ_FilterDetrendNone=0, /**< No detrend */
      SQ_FilterDetrendMean,   /**< Detrend by mean value */
      SQ_FilterDetrendLinear  /**< Detrend linear */
   } SQ_FilterDetrendMode;

   /**
   * Wavelet filter functions.
   */
   typedef enum SQ_FilterWaveletFunctionEnum
   {
      SQ_FilterFunBeylkin=0,      /**< Wavelet function Beylkin */
      SQ_FilterFunCoiflet,        /**< Wavelet function Coiflet */
      SQ_FilterFunDaubechies,     /**< Wavelet function Daubechies */
      SQ_FilterFunSymmlet,        /**< Wavelet function Symmlet */
      SQ_FilterFunBiorthogonal1,  /**< Wavelet function Biorthogonal1 */
      SQ_FilterFunBiorthogonal2,  /**< Wavelet function Biorthogonal2 */
      SQ_FilterFunBiorthogonal3,  /**< Wavelet function Biorthogonal3 */
      SQ_FilterFunBiorthogonal4,  /**< Wavelet function Biorthogonal4 */
      SQ_FilterFunBiorthogonal5,  /**< Wavelet function Biorthogonal5 */  
      SQ_FilterFunBiorthogonal6   /**< Wavelet function Biorthogonal6 */
   } SQ_FilterWaveletFunction;

   /**
   * Wavelet compression method.
   */
   typedef enum SQ_FilterCompressionMethodEnum
   {
      SQ_FilterCompDWT=0,         /**< Compression by DWT */
      SQ_FilterCompBestBasis      /**< Compression by Best Basis */
   } SQ_FilterCompressionMethod;

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQMDEFINES_H_ */
