/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMFILTER_H_
#define _SQMFILTER_H_

#include "SQDef.h"

/** @file SQMFilter.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQProject.h"
#include "SQMDefines.h"
#include "SQControlCharts.h"
   /// @cond SQM 
   
   /** 
   The import handle used to identify an ongoing import.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_FilterHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Filter;

   /**
   * Creates a spectral filter that filters data in the first dataset.
   * If WCS/WOSC/OSCW filter is created the filtered data is appended last in the new dataset, 
   * otherwise the variables are replaced by the new data. If OSC/WOSC an unfitted model will always be created in
   * the new project.
   * Note that a chained filter is always created.
   *
   * @param[in] hProject               Handle to the project.
   * @param[in] iDatasetNumber         The number of the dataset that should be filtered
   * @param[out] pFilterHandle          The handle to the filter.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CreateSpectralFilter(SQ_Project hProject, int iDatasetNumber, SQ_Filter* pFilterHandle);

   /**
   * Creates a time series filter that filters data in the first dataset.
   *
   * @param[in] hProject               Handle to the project.
   * @param[in] eFilter                A time series filter to create
   * @param[in] iDatasetNumber         The number of the dataset that should be filtered
   * @param[out] pFilterHandle          The handle to the filter.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CreateTimeSeriesFilter(SQ_Project hProject, SQ_TSFilterType eFilter, int iDatasetNumber, SQ_Filter* pFilterHandle);

   /**
   * Sets x,y and observation data that should be used when filtering.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] oXVarNums              The vector of x variables in dataset to include for filtering.
   *                                    The indices must be in ascending order.
   * @param[in] oYVarNums              The vector of y variables in dataset to include for filtering.
   *                                    The indices must be in ascending order.
   * @param[in] oObsNums               The vector of observations in dataset to include for filtering.
   *                                    The observations must be in ascending order.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetData(SQ_Filter hFilter, SQ_IntVector* oXVarNums, SQ_IntVector* oYVarNums, SQ_IntVector* oObsNums);

   /**
   * Sets transformation for the variables set with SetData.
   * Default transformation is None.
   * Must be called before StartFiltering.
   * Valid if OSC, WCS, WOSC or OSCW filter.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] oVarNums               The vector of variables in dataset to transform according to the TransformInfo.
   * @param[in] iVarNums               The number of variables in the vector. Note: The vector size may be larger.
   *                                    Transformation will be done for the iVarNums first variables in the vector.
   * @param[in] oTransInfo             The TransformInfo that describes the transformation to apply.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetTransformation(SQ_Filter hFilter, SQ_IntVector oVarNums, int iVarNums, SQ_WSTransformInfo oTransInfo);

   /**
   * Sets scale for the variables set with SetData.
   * Must be called before StartFiltering.
   * Default for X is Center.
   * Default for Y is Unit Variance.
   * Valid if OSC, WOSC or OSCW filter.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] oVarNums               The vector of variables in dataset to transform according to the TransformInfo.
   * @param[in] iVarNums               The number of variables in the vector. Note: The vector size may be larger.
   *                                    Scaling will be done for the iVarNums first variables in the vector.
   * @param[in] eScale                 The ScalingType that describes the scaling to apply.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetScale(SQ_Filter hFilter, SQ_IntVector oVarNums, int iVarNums, SQ_FilterScalingType eScale);

   /**
   * Sets the lambda value
   * Only valid when EWMAfilter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the derivate filter.
   * @param[in] fLambda                The lambda value, between 0 and 1.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetLambdaValue(SQ_Filter hFilter, float fLambda);
   
   /**
   * Sets the type of EWMA calculation
   * Only valid when EWMAfilter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the derivate filter.
   * @param[in] eEWMAType              The type of EWMA calculation filter or predictive, before SIMCA-Q 14.1 only predictive was available.
   * @return                           Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetEWMAType(SQ_Filter hFilter, SQ_EWMAType eEWMAType);

   /**
   * Sets the polynomial order.
   * Only valid when derivate or Salvitzky-Golay filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the derivate filter.
   * @param[in] eOrder                 The polynomial order.
   * @param[in] bDerivate              True if it's a derivate filter, otherwise it's a Savitzky-Golay filter
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPolynomialOrder(SQ_Filter hFilter, SQ_FilterPolynomialOrder eOrder, SQ_Bool bDerivate);

   /**
   * Sets the derivate order.
   * Only valid when derivate filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the derivate filter.
   * @param[in] eDerivateOrder         The derivate order.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDerivateOrder(SQ_Filter hFilter, SQ_FilterDerivateOrder eDerivateOrder);

   /**
   * Sets the number of points in each sub-model.
   * Only valid when derivate or Salvitzky-Golay filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the derivate filter.
   * @param[in] iSubModelPoints        The number of sub-model points.
   *                                   Must be odd and at least 5, but not larger than or equal to the number of variables selected.
   * @param[in] bDerivate              True if it's a derivate filter, otherwise it's a Savitzky-Golay filter
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetSubModelPoints(SQ_Filter hFilter, int iSubModelPoints, SQ_Bool bDerivate);

   /**
   * Sets the distance between each point.
   * Only valid when derivate filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] fDeltaX                The distance between the points.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDeltaX(SQ_Filter hFilter, float fDeltaX);


   /**
   * Gets angle in degrees of the last component calculated.
   * Only valid when OSC filter.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the OSC filter.
   * @param[out] pfAngle               The angle in degrees.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetAngle(SQ_Filter hFilter, float* pfAngle);

   /**
   * Gets the remaining SS in percent of the last component calculated.
   * Only valid when OSC filter.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the OSC filter.
   * @param[out] pfRemainingSS         The returned remaining SS in percent.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRemainingSS(SQ_Filter hFilter, float* pfRemainingSS);

   /**
   * Gets the eigenvalue of the last component calculated.
   * Only valid when OSC filter.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the OSC filter.
   * @param[out] pfEigenvalue          The eigenvalue.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEigenvalue(SQ_Filter hFilter, float* pfEigenvalue);

   /**
   * Calculates the next component.
   * Only valid when OSC filter, which by default starts with one component.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the OSC filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_NextFilterComponent(SQ_Filter hFilter);

   /**
   * Removes the last component.
   * Only valid when OSC filter, which by default starts with one component.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the OSC filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_RemoveFilterComponent(SQ_Filter hFilter);


   /**
   * Sets the detrend mode for the wavelet filter.
   * Only valid when some kind of WAV filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter.
   * @param[in] eMode                  The detrend mode.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDetrendMode(SQ_Filter hFilter, SQ_FilterDetrendMode eMode);

   /**
   * Sets the wavelet function and corresponding wavelet order to be used.
   * Only valid when some kind of WAV filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter.
   * @param[in] eFunction              The wavelet function.
   * @param[in] iOrder                 The wavelet order. If Beylkin or Biorthogonal{4,5,6} is used as function this argument is ignored.
   *                                   See the documentation of valid orders for each of the wavelet functions.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetWaveletFunction(SQ_Filter hFilter, SQ_FilterWaveletFunction eFunction, int iOrder);

   /**
   * Sets the filter to use energy retained by variance.
   * Only valid when some kind of WAV filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter.
   * @param[in] eMethod                The compression method to use.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetEnergyRetainedByVariance(SQ_Filter hFilter, SQ_FilterCompressionMethod eMethod);

   /**
   * Sets option to use padding on a WCS filter when retained energy by variance and
   * compression method is DWT.
   * Only valid when WCS filter, retained energy by variance and compression is DWT.
   * When compression is best basis, padding is always used.
   * Must be called before StartFiltering.
   * Padding means "pad to the power of 2", see the documentation for more information.
   *
   * @param[in] hFilter                Handle to the WAV type filter.
   * @param[in] bPadding               True if padding should be used, false if not.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetVarianceUsePadding(SQ_Filter hFilter, SQ_Bool bPadding);

   /**
   * Sets the filter to use energy retained by detail level.
   * Only valid when some kind of WAV filter.
   * Must be called before StartFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetEnergyRetainedByDetailLevel(SQ_Filter hFilter);

   /**
   * Gets the number of detail level indexes that can be used to select the detail levels.
   * The lowest index corresponds to the highest frequencies.
   * Only valid when some kind of WAV filter and retained energy by details is set.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[out] pnIndexes             The number of detail level indexes that can be used in
   *                                   SetDetailLevelIndexes. Indexes that may be used are 1..pnIndexes.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDetailLevelNumOfIndexes(SQ_Filter hFilter, int* pnIndexes);

   /**
   * Sets the detail level indexes to be used.
   * Only valid when some kind of WAV filter and retained energy by details is set.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[in] nVecDetailLevels       The vector of indexes to be used.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDetailLevelIndexes(SQ_Filter hFilter, SQ_IntVector nVecDetailLevels);

   /**
   * Gets the retained energy (in %) and number of coefficients for a given detail level index.
   * Only valid when some kind of WAV filter and retained energy by details is set.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[in] iDetailLevelIndex      The detail level index to get data from.
   * @param[out] pnCoeffs              The number of coefficients for the given index.
   * @param[out] pfRetained            The retained energy (in %) for the given index.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDetailLevelData(SQ_Filter hFilter, int iDetailLevelIndex, int* pnCoeffs, float* pfRetained);

   /**
   * Gets the total retained energy (in %) with the previous set detail levels.
   * Only valid when some kind of WAV filter and retained energy by details is set.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[out] pfRetained            The total energy retained.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDetailLevelTotalRetained(SQ_Filter hFilter, float* pfRetained);

   /**
   * Sets the decimation number, which will exclude some observations.
   * Only valid when a WDTS filter and when energy retained by details is chosen.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[in] iDecNum                The decimation number.
   *                                   1 means include all observations.
   *                                   2 means include every second.
   *                                   4 means include every fourth and so on.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDetailLevelDecimation(SQ_Filter hFilter, int iDecNum);

   /**
   * Sets the target variable for a WDTS filter.
   * Only valid when a WDTS filter.
   *
   * @param[in] hFilter                Handle to the WAV filter type.
   * @param[in] iTargetVariable        The target variable index. This index must also be set in SetData.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetTargetVariable(SQ_Filter hFilter, int iTargetVariable);

   /**
   * Sets the dataset to use for a WDTS filter.
   * Only valid when a WDTS filter.
   * Must be called before StartFiltering and SetData.
   *
   * @param[in] hFilter                Handle to the WDTS filter.
   * @param[in] iDatasetNumber         The number of the dataset to use.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataset(SQ_Filter hFilter, int iDatasetNumber);

   /**
   * Sets the number of coefficients to use.
   * Only valid when some kind of WAV filter and retained energy by variance is set.
   * Must be called between StartFiltering and FinishFiltering.
   *
   * @param[in] hFilter                Handle to the WAV type filter.
   * @param[in] iCoeffs                The number of coefficients.
   * @param[out] pfRetained            The total retained energy when using iCoeffs coefficients.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetVarianceCoeffs(SQ_Filter hFilter, int iCoeffs, float* pfRetained);

   /**
   * Starts the filtering process.
   * If OSC, a first component is automatically calculated.
   *
   * @param[in] hFilter                Handle to the filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_StartFiltering(SQ_Filter hFilter);

   /**
   * Sets the filtering order.
   * Must be called directly after CreateSpectralFilter.
   * A chained filter can be a combination of the following filter types:
   * 1. SQ_FilterDerivate 
   * 2. SQ_FilterSNV or SQ_FilterMSC
   * 3. SQ_FilterOSC or SQ_FilterWCS
   * 4. SQ_FilterOSC or SQ_FilterWCS (the one not selected in step 3)
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] eFilter                The filter type.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetChainOrder(SQ_Filter hFilter, SQ_FilterType eFilter);

   /**
   * Sets the filtering order.
   * Must be called directly after CreateSpectralFilter.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] szPluginFilter         The name of the plugin filter, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetChainOrderPlugin(SQ_Filter hFilter, const char* szPluginFilter);

   /**
   * Returns an vector of names of the plugin filters.
   *
   * @param[out] pvecNames              A vector of name of the plugin filters.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPluginFilterNames(SQ_StringVector* pvecNames);

   /**
   * Returns the options for a plugin filter.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] szPluginName           The name of the plugin filter to receive options from.
   * @param[in,out] pszOptions         An XML formatted string with the options for the specified plugin.
   * @param[in] iBufferLen             The lenght of the buffer to recieve the options.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPluginFilterOptions(SQ_Filter hFilter, const char* szPluginName, char* pszOptions, int iBufferLen);

   /**
   * Sets the options for a plugin filter.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] szPluginName           The name of the plugin filter to receive options from.
   * @param[in] szOptions              An XML formatted string with the options for the specified plugin.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPluginFilterOptions(SQ_Filter hFilter, const char* szPluginName, const char* szOptions);

   /**
   * Set the name of the filtered dataset.
   *
   * @param[in] hFilter                Handle to the filter.
   * @param[in] strName                The name of the dataset to create.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDatasetName(SQ_Filter hFilter, const char* strName);

   /**
   * Finish the filtering process.
   *
   * @param[in] hFilter                Handle to the filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_FinishFiltering(SQ_Filter hFilter);

   /**
   * Releases the filter handle.
   *
   * @param[in] hFilter                Handle to the filter.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ReleaseFilterHandle(SQ_Filter hFilter);

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQMFILTER_H_ */
