/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef SQMIMPORT_H
#define SQMIMPORT_H

#include "SQDef.h"

/** @file SQMImport.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQProject.h"
#include "SQStringMatrix.h"

   /// @cond SQM 
   
   /** 
   The import handle used to identify an ongoing import.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_ImportHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Import;

   /**
   *  Prepare import of data to a new project.
   *  It is the users responsibility to close the import when finished (SQ_FinishImport).
   *
   *  @param[in]  szProjectPath     The path, including the project name and for the project, UTF-8 encoded.
   *  @param[in]  szTempDir         The directory where temporary files should be saved and where the project should recover from, UTF-8 encoded.
   *                                Recovering is disabled if szTempDir = NULL.
   *  @param[out] pImport           The handle to this import.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitImport(const char* szProjectPath, const char* szTempDir, SQ_Import* pImport);
   /**
   *  Prepare import of data to a new project, the new project will not be saved on disk but kept in memory.
   *  To save the project to disk when it has been created call SQ_SaveAs.
   *  It is the users responsibility to close the import when finished (SQ_FinishImport).
   *
   *  @param[out] pImport           The handle to this import.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitImportNoUsp(SQ_Import* pImport);

   /**
   *  Prepare import of a new dataset to an existing project.
   *  It is the users responsibility to close the project when finished (SQ_CloseImport).
   *
   *  @param[in]  pProject          The project to import the new dataset to.
   *  @param[in]  szDatasetName     The name of the new dataset, UTF-8 encoded.
   *  @param[out] pImport           The handle to this import.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitImportDataset(SQ_Project pProject, const char* szDatasetName, SQ_Import* pImport);

   /**
   *  Reserve space for the import, this will make the import faster if you call
   *  AddQuantitativeVariables or AddQualitativeVariables more than once.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  iNumObs           The total number of observations.
   *  @param[in]  iNumQuantitative  The total number of quantitative variables.
   *  @param[in]  iNumQualitative   The total number of qualitative variables.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_Reserve(SQ_Import pImport, int iNumObs, int iNumQuantitative, int iNumQualitative);

   /**
   *  Add information on local centering.
   *  With local centering one models the variation of the variable, 
   *  or part of variable with for instance classes, around the specified center.
   *  If the szCenterIDsetting or variable name don't match the one imported this call will be ignored.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  iCenterID         The observation ID index specifying the observations to center.
   *  @param[in]  pVariableNames    The name of the variables to center
   *  @param[in]  fValue            The value to subtract from the selected observation/variable combination. 
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddLocalCenteringData(SQ_Import pImport, int iCenterID, SQ_StringVector* pVariableNames, float fValue);

   /**
   *  Set the observation names in the new dataset.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  pObservationIDs   Matrix containing the observation names, 
   *                                each row is an observation. 
   *                                Put secondary observation IDs in columns.
   *  @param[in]  iPrimaryID        The index of the ID to use as primary observation ID.
   *                                Every entry in the primary observation ID vector must be unique.
   *                                if 0, primary ids will be auto generated.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetObservationNames(SQ_Import pImport, SQ_StringMatrix* pObservationIDs, int iPrimaryID);

   /**
   *  Set a value that the import will treat as a missing value for quantitative variables.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  fMissing          The missing value representation.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetMissingValueRepresentation(SQ_Import pImport, float fMissing);

   /**
   *  Set a value that the import will treat as a missing value for qualitative variables.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  szMisVal          The missing value representation, UTF-8 encoded.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetMissingValueStringRepresentation(SQ_Import pImport, const char* szMisVal);

   /**
   *  Add quantitative variables to the new dataset.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  pData             Matrix containing the quantitative variable values each row is an observation, and each column is a variable.
   *  @param[in]  pVariableIDs      Matrix containing the quantitative variable names names, each column is an variable, put secondary variable IDs in rows.
   *  @param[in]  iPrimaryID        The index of the ID to use as primary variable ID.
   *                                Every entry in the primary variable ID vector must be unique.
   *                                If 0, primary ids will be auto generated.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddQuantitativeVariables(SQ_Import pImport, SQ_FloatMatrix* pData, SQ_StringMatrix* pVariableIDs, int iPrimaryID);

   /**
   *  Add qualitative variables to the new dataset.
   *  Note: If this import is a merge or a secondary dataset with existing variables, 
   *  the values for each variable can not be a value that did not already exist. I.e., you can not import new settings for a qualitative variable.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  pData             Matrix containing the qualitative variable values each row is an observation, and each column is a variable.
   *  @param[in]  pVariableIDs      Matrix containing the qualitative variable names names, each column is an variable, put secondary observation IDs in rows
   *  @param[in]  iPrimaryID        The index of the ID to use as primary variable ID.
   *                                Every entry in the primary variable ID vector must be unique.
   *                                If 0, primary ids will be auto generated.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddQualitativeVariables(SQ_Import pImport, SQ_StringMatrix* pData, SQ_StringMatrix* pVariableIDs, int iPrimaryID);

   /**
   *  Name the secondary IDs, the primary ID is always called PrimaryID and can't be renamed
   *  call this function only after Variables are added.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  szNames           The names of the variable id series.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetVariableIDSeriesNames(SQ_Import pImport, SQ_StringVector* szNames);

   /**
   *  Name the secondary IDs, the primary ID is always called PrimaryID and can't be renamed
   *  call this function only after SetObservationNames.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  szNames           The names of the Observation id series.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetObservationIDSeriesNames(SQ_Import pImport, SQ_StringVector* szNames);

   /**
   *	Set the index of the Y columns.
   *  Call this only after adding every time column
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  pYColumns         The indexes of the Y columns in the data.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetYColumns(SQ_Import pImport, SQ_IntVector* pYColumns);

   /**
   *  Specify the column in the matrix that is date/time
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  pDateTimeColumns  The indices of the date/time columns.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDateTimeColumns(SQ_Import pImport, SQ_IntVector* pDateTimeColumns);

   /**
   *  Set the name of the new dataset.
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[in]  szName            The name of the new dataset, UTF-8 encoded.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetDataSetName(SQ_Import pImport, const char* szName);

   /**
   *  Create a dataset of the variables and observations added.
   *  This function automatically releases the import handle created from SQ_InitImport().
   *
   *  @param[in]  pImport           Handle for the import of the new project.
   *  @param[out] pProject          The handle to the new project.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_FinishImport(SQ_Import* pImport, SQ_Project* pProject);

   /**
   *  Create the new dataset of the variables and observations added.
   *  The import handle must come from SQ_InitImportDataset(), otherwise it will fail.
   *  This function automatically releases the import handle created from SQ_InitImportDataset().
   *
   *  @param[in]  pImport           Handle for the import of the new dataset.
   *  @param[out] iDatasetNumber    The number of the new dataset.
   *  @return                       Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_FinishImportDataset(SQ_Import* pImport, int* iDatasetNumber);

   /**
   *  Merge the imported data into an existing dataset.
   *  No new variables may exist in the import that does not exist in the existing dataset.
   *  All created models for the specified dataset will be deleted.
   *  The import handle must come from SQ_InitImportDataset(), otherwise it will fail.
   *  This function automatically releases the import handle created from SQ_InitImportDataset().
   *
   *  @param[in]  pImport              Handle for the import of the new dataset.
   *  @param[in]  iDatasetNumber       The dataset number to merge to.
   *  @return                          Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_FinishImportMerge(SQ_Import* pImport, int iDatasetNumber);

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* SQMIMPORT_H */
