/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMMODEL_H_
#define _SQMMODEL_H_

#include "SQDef.h"

/** @file SQMModel.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQModel.h"
#include "SQComponentVector.h"
#include "SQMWorkset.h"

   /// @cond SQM 
   
   /**
   * Create a new workset as a copy of the workset used in an existing model.
   * The workset can be manipulated with the functions in SQMWorkset.h
   * Note that no new model is created in the project until SQ_CreateModel or SQ_UpdateModel is called.
   * The returned workset handle must be released with SQ_ReleaseWorkset
   *
   * @see SQMWorksetCInterface.h
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[out]  pWorksetHandle      The handle to the new workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWorksetCopy(SQ_Model pModel, SQ_Workset* pWorksetHandle);


   /**
   *  Autofit model.
   *
   *  @param[in]   pModel              The model handle to use
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AutofitModel(SQ_Model pModel);

   /**
   *  Autofit class models.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   hClasses            Array of classes to use
   *  @param[in]   hComponents         Array of number of components for each class. If number
   *                                   of components for a class is set to -1 autofit will be made.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AutofitClassModels(SQ_Model pModel, SQ_IntVector hClasses, SQ_ComponentVector hComponents);

   /**
   *  Sets the center value of T used when calculating T2Range. 
   *  Set it to -99 to use the arithmetic mean.
   *
   *  @param[in]  pModel               The model handle to use
   *  @param[in]  fCenterOfScores      The center value to use.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetCenterOfScores(SQ_Model pModel, float fCenterOfScores);
   /**
   *  Gets the center value of T used when calculating T2Range.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[out]   pfCenterOfScores   The currently used center value.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCenterOfScores(SQ_Model pModel, float *pfCenterOfScores);

   /**
   *  Set the model title.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   szModelTitle        The new model title, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_SetModelTitle(SQ_Model pModel, const char* szModelTitle);

   /**
   *  Set the default model confidence level.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   fLevel              The confidence level. 0.95 means 95% confidence level. 
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_SetModelConfidenceLevel(SQ_Model pModel, float fLevel);

   /**
   *  Set the default model significance level for DModX and Hotelling's T2.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   fLevel              The significance level. 0.05 means 95% probability level.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_SetModelSignificanceLevel(SQ_Model pModel, float fLevel);

   /**
   *  Delete a model.
   *
   *  @param[in]   pModel              The model handle to use
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_DeleteModel(SQ_Model pModel);

   /**
   *  Calculate the next component. This can not be called for OPLS/O2PLS models.
   *
   *  @param[in]   pModel              The model handle to use
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CalculateNextComponent(SQ_Model pModel);

   /**
   *  Calculate component. This should be used for OPLS/O2PLS models.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   oComponents        A component vector of at least size 1 with the number of
   *                                   components for each type
   * oComponents[]={ Predictive components,
	*			         X-Side orthogonal components,
	*				      Y-Side orthogonal components,
	*				      X-Side PCA components,            for extended O2PLS models
	*				      Y-Side PCA components},           for extended O2PLS models
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CalculateOPLSComponents(SQ_Model pModel, SQ_ComponentVector oComponents);

   /**
   *  Remove a component. This can not be called for OPLS/O2PLS models.
   *
   *  @param[in]   pModel             The model handle to use
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_RemoveComponent(SQ_Model pModel);

   /**
   *  Change the cross validation groups. 
   *	Select every Nth observation where N is the number of cross validation groups. 
   *  The function fails if the model is fitted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   iCVGroupsNumber     The number of cross validation groups.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetCVGroupAsEveryNthObservation(SQ_Model pModel, int iCVGroupsNumber);

   /**
   *  Change the cross validation groups. 
   *  Assign observation based on the scores from an existing model.
   *  The function fails if the model is fitted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   iCVGroupsNumber     The number of cross validation groups.
   *  @param[in]   iScoresFromModel    The model to use the scores from. This model must be fitted.
   *  @param[in]   eCVGroupBy          The strategy for dividing the observations into groups. @see SQ_CVGroupBy
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetCVGroupObservationBasedOnScores(SQ_Model pModel, int iCVGroupsNumber, int iScoresFromModel, SQ_CVGroupBy eCVGroupBy);

   /**
   *  Change the cross validation groups by selecting a variable from a dataset.
   *  The function fails if the model is fitted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   iCVGroupsNumber     The number of cross validation groups.
   *  @param[in]   iDSIndex            The index of the dataset.
   *  @param[in]   iVarIndex           The index of the variable in the dataset.
   *  @param[in]   eCVGroupBy          The strategy for dividing the observations into groups. @see SQ_CVGroupBy
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetCVGroupObservationBasedOnVariable(SQ_Model pModel, int iCVGroupsNumber, int iDSIndex, int iVarIndex, SQ_CVGroupBy eCVGroupBy);

   /**
   *  Change the cross validation groups by selecting a secondary variable ID.
   *  The function fails if the model is fitted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   iCVGroupsNumber     The number of cross validation groups.
   *  @param[in]   iVarID              The index of the Variable ID that will be used to identify the variables in the project. 
   *                                   The variable names in the selected variable ID must be unique or the initialization will fail.
   *  @param[in]   eCVGroupBy          The strategy for dividing the observations into groups. @see SQ_CVGroupBy
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetCVGroupObservationBasedOnObservationID(SQ_Model pModel, int iCVGroupsNumber, int iVarID, SQ_CVGroupBy eCVGroupBy);

   /**
   *  Sets a fitted model as a hierarchical base model. 
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   eIncludeVar         Defines what variables should be added to the new worksets.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetHierarchicalBaseModel(SQ_Model pModel, SQ_Hierarchical eIncludeVar);
   /**
   *  Remove the hierarchical base model flag on an hierarchical base model. 
   *  Note: If there exists top level models with variables from the given model, these models will be deleted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetNonHierarchicalBaseModel(SQ_Model pModel);

   
   /**
   *  Change model type. 
   *  The function fails if the model is fitted.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in]   eType               The new model type.
   *  @param[in]   iClassNo            The class number to set, for PCA_Class and PLS_Class.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ChangeModelType(SQ_Model pModel, SQ_ModelType eType, int iClassNo);

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQMMODEL_H_ */
