/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMPROJECT_H_
#define _SQMPROJECT_H_

#include "SQDef.h"

/** @file SQMProject.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"
#include "SQProject.h"
#include "SQMWorkset.h"

   /// @cond SQM 
   
   /**
   *  Initialize the predictor with a new project and use a temporary directory for intermediate storing project data.
   *  Using a temporary directory is faster when editing the projects from an interactive application.
   *  The predictor can handle multiple project that are identified with a project id.
   *  It is the users responsibility to remove the project when finished.
   *  @see RemoveProject
   *
   *  @param[in]   szProjectName       The full path to the SIMCA project file, UTF-8 encoded.
   *  @param[in]   bRecover            If Recover= SQ_True, SQM tries to recover from the szTempDir,else SQM does not.
   *  @param[in]   szTempDir           The directory where temporary files should be saved, and where the project should be recovered from, UTF-8 encoded.
   *  @param[in]   szPassword          If the usp is encrypted, supply the password that was used to encrypt it, UTF-8 encoded, else use NULL.
   *  @param[out]  pProject            The project handle
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_OpenProjectTempDir(const char *szProjectName, SQ_Bool bRecover, const char *szTempDir, const char *szPassword, SQ_Project *pProject);

   /**
   *  Checks if it is possible to reconstruct the usp from the temporary directory.
   *
   *  @param[in]   szProjectName       The full path to the SIMCA project file, UTF-8 encoded.
   *  @param[in]   szTempDir           The directory where temporary files should be saved, and where the project should recover from, UTF-8 encoded.
   *  @param[in]   bCanRecover         Tell if the project can be recovered or not.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CanRecover(const char *szProjectName, const char *szTempDir, SQ_Bool* bCanRecover);

   /**
   *  Decrypts an encrypted project, use encrypt to encrypt it again.
   *
   *  @param[in]   pProject            The handle to the project to decrypt.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_Decrypt(SQ_Project pProject);
   /**
   *  Encrypt a project.
   *  Note: An encrypted project can only be opened with the password that was supplied when the encryption was performed.
   *
   *  @param[in]   pProject            The handle to the project to encrypt.
   *  @param[in]   szPassword          The password that the project should be encrypted with, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_Encrypt(SQ_Project pProject, const char *szPassword);
   /**
   * Save the project with a new name. 
   * If the new project already exists it will be overwritten.
   *
   * @param[in]   pProject             The handle to the project to be saved.
   * @param[in]   szNewProjectPath     The full path including the new name of the project,
   *                                   for example C:\\SIMCA-QM\\NewProject.usp, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SaveAs(SQ_Project pProject, const char *szNewProjectPath);

   /**
   *  Gets the options for a project.
   *  @see tagSQMProjectOptions
   *
   *  @param[in]   pProject            The handle to the project
   *  @param[in]   pSQMProj            The struct to get the project options from.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetProjectOptions(SQ_Project pProject, SQ_ProjectOptions *pSQMProj);
   /**
   *  Sets the options for a project.
   *  @see tagSQMProjectOptions
   *
   *  @param[in]   pProject            The handle to the project
   *  @param[in]   pSQMProj            The struct to set the project options in.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetProjectOptions(SQ_Project pProject, SQ_ProjectOptions *pSQMProj);

   /**
   * Create a new empty workset. Use @see SQ_CreateDefaultWorkset or @see SQ_AddObservations to add observations from the datasets.
   * The workset can be manipulated with the functions in SQMWorksetCInterface.h
   * Note that no new model is created in the project until SQ_CreateModel is called (@see SQMWorksetCInterface.h)
   * The returned workset handle must be released with SQ_ReleaseWorkset
   *
   * @param[in]   pProject             The handle to the project for which a new workset is to be created.
   * @param[in]   vecDatasets          Dataset numbers to build the workset from
   * @param[out]  pWorkset             The handle to the new workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNewWorkset(SQ_Project pProject, SQ_IntVector vecDatasets, SQ_Workset *pWorkset);

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQMPROJECT_H_ */
