/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMWORKSET_H_
#define _SQMWORKSET_H_

#include "SQDef.h"

/** @file SQMWorkset.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"
#include "SQMDefines.h"

   /// @cond SQM 
   
   /** 
   The Workset handle used to identify a Workset.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_WorksetHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Workset;


   /**
   * Releases the workset (frees resources used by the workset).
   * This function has to be called for each SQ_WorksetHandle obtained from SQ_GetNewWorkset().
   * The handle should not be used after this function has been called.
   *
   * @param[in]   hWorkset    The workset that is to be released.
   *  @return                 Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ReleaseWorkset(SQ_Workset hWorkset);

   /**
   * Adds a new dataset last in the workset.
   *
   * @param[in] hWorkset               Handle to the workset.
   * @param[in] iDatasetNumber         The number of the dataset to add from, if that dataset is not already in the model it will be added last.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddDataset(SQ_Workset hWorkset, int iDatasetNumber);

   /**
   * Adds observations to the workset.
   *
   * @param[in] hWorkset               Handle to the workset.
   * @param[in] iDatasetNumber         The number of the dataset to add from, if that dataset is not already in the model it will be added last.
   * @param[in] vecObservationIndices  A vector of indices of the observation to add from the dataset.
   * @param[out] piWorksetIndices      The indices of the added observations in the workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddObservations(SQ_Workset hWorkset, int iDatasetNumber, SQ_IntVector vecObservationIndices, SQ_IntVector *piWorksetIndices);

   /**
   * Adds a variable to the workset. The added variable will be un-scaled (SQ_ScaleNone).
   * @see SQ_SetScaling.
   *
   * @param[in] hWorkset               Handle to the workset.
   * @param[in] iDatasetNumber         The number of the dataset to add from.
   * @param[in] hDSVariableIndices     A vector containing variable indices in the dataset.
   *                                   Ex. If the dataset contains variables A, B and C
   *                                     (in that order), and pDSVariableIndices contains 1, 1, 3, 
   *                                     the added variable will be A*A*C
   *                                     The vector must have at least one element and a maximum of 
   *                                     three.
   * @param[in] pLags                  Lags for the variables indexed by pDS1VariableIndices.
   *                                   This parameter must have the same length as
   *                                   pDS1VariableIndices or be of zero length (if none of the sub-variables
   *                                   are to be lagged). if pLagTimeVariable and pLagSpeedVariable is not supplied or <1 the pLag is assumed to be integer steps.
   * @param[in] pLagTimeVariable	   The index of the time variable used to calculate the dynamic lag
   *								   This parameter must have the same length as pLags be null or empty
   *                                   if both pLagTimeVariable and pLagSpeedVariable is empty or a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   * @param[in] pLagSpeedVariable	   The index of the speed variable used to calculate the dynamic lag
   *								   This parameter must have the same length as pLags be null or empty
   *                                   if it both pLagTimeVariable and pLagSpeedVariable a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   * @param[in] eBlock                 The block where the new variable is to be added.
   *                                     Interaction terms can not be added to the Y-block.
   * @param[out] hWorksetIndex         Will contain the index in the workset of the new variable
   *                                     on return.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AddVariable(SQ_Workset hWorkset, int iDatasetNumber, SQ_IntVector hDSVariableIndices, SQ_FloatVector* pLags, SQ_IntVector* pLagTimeVariable, SQ_IntVector* pLagSpeedVariable, SQ_VariableBlock eBlock, int *hWorksetIndex);

   /**
   * Creates a default workset.
   * The default workset is a PCA model containing all variables in the main
   * dataset unless something else has been defined in the project options.
   * The function clears the workset before creating the new workset.
   *
   * @param[in]   hWorkset             A handle to the Workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CreateDefaultWorkset(SQ_Workset hWorkset);

   /**
   * Sets the description of the workset.
   * The description is usually a longer string that the name.
   * The default is 'Untitled'.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] szDescription          The description of the workset, UTF-8 encoded, or NULL to use the default description
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_SetWorksetDescription(SQ_Workset hWorkset, const char *szDescription);

   /**
   * Returns the description of the workset.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in,out] szDescription      Holds the description of the workset on exit, UTF-8 encoded.
   * @param[in] iBufferLen             The length of szDescription buffer
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL  SQ_GetWorksetDescription(SQ_Workset hWorkset, char *szDescription, int iBufferLen);

   /**
   * Returns the number of variables in a block.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in] eBlock                 The block (SQ_BlockX or SQ_BlockY).
   * @param[out] pSize                 Will contain the number of variables in the block on exit.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetVariableSize(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int *pSize);

   /**
   * Gets the missing value tolerance for observations.
   * Observations with more missing values than the limit will be excluded from
   * the model when the model is created.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[out] pfFractionTolerated   The tolerance as a fraction (between 0 and 1).
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_GetMissingValueToleranceObs(SQ_Workset hWorkset, float *pfFractionTolerated);
   /**
   * Gets the missing value tolerance for variables.
   * Variables with more missing values than the limit will be excluded from
   * the model when the model is created.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[out] pfFractionTolerated   The tolerance as a fraction (between 0 and 1).
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_GetMissingValueToleranceVar(SQ_Workset hWorkset, float *pfFractionTolerated);

   /**
   * Sets the missing value tolerance for observations.
   * Observations with more missing values than the limit will be excluded from
   * the model when the model is created.
   * The default value is the default for the project.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] fFractionTolerated     The tolerance as a fraction (must be between 0 and 1).
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_SetMissingValueToleranceObs(SQ_Workset hWorkset, float fFractionTolerated);
   /**
   * Sets the missing value tolerance for variables.
   * Variables with more missing values than the limit will be excluded from
   * the model when the model is created.
   * The default value is the default for the project.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] fFractionTolerated     The tolerance as a fraction (must be between 0 and 1).
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_SetMissingValueToleranceVar(SQ_Workset hWorkset, float fFractionTolerated);

   /**
   * Decide if we should exclude variables with low variance.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] bKeepVariables         SQ_True to keep low variance variables, SQ_False to exclude them
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode  SQCDECL SQ_SetKeepLowVarianceVariables(SQ_Workset hWorkset, SQ_Bool bKeepVariables);

   /**
   * Returns a string containing a complete description of a variable scaling.
   * The string returned can be up to 130 characters long.
   *
   * @param[in] pScaleInfo             Describes the scaling.
   * @param[in,out] szName             Will contain the name, UTF-8 encoded.
   * @param[in] iBufferLen             Length of the in buffer to get the name.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLongScalingName(SQ_WSScaleInfo *pScaleInfo, char *szName, int iBufferLen);

   /**
   * Creates one or several unfitted models from the workset.
   * If class model are to be created as many models as there are classes will be created.
   *
   * @param[in]   hWorkset             A handle to the workset.
   * @param[in]   bCreateClassModels   Set to SQ_True if class models are to be created.
   * @param[in]   eModelType           The type of model to create.
   * @param[out]   pModelNumbers       Vector with the new model numbers.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CreateModel(SQ_Workset hWorkset, SQ_Bool bCreateClassModels, SQ_ModelType eModelType, SQ_IntVector* pModelNumbers);

   /**
   * Updates the model, the model becomes unfitted. You can only use a handle retrieved from
   * SQ_GetWorksetCopy, (not SQ_GetNewWorkset).
   *
   * @param[in]   hWorkset             A handle to the workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_UpdateModel(SQ_Workset hWorkset);


   /**
   * Gets the number of the class that an observation belongs to.
   *
   * @param[in]   hWorkset                   A handle to the workset.
   * @param[in]   iWorksetObservationIndex   The index of the observation in the workset.
   * @param[out]   piClassNo                 The number of the class that the observation belongs to.
   *                                         If the observation belongs to all classes,
   *                                         this parameter will be 0 (zero) on exit.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetObservationClass(SQ_Workset hWorkset, int iWorksetObservationIndex, int *piClassNo);
   /**
   * Gets the classes that a variable belongs to.
   *
   * @param[in]   hWorkset                   A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetVariableIndex        The index in the workset of the variable.
   * @param[out] piClasses                   A vector containing the numbers of the
   *                                         classes that the variable belongs to.
   *                                         The first element in the vector has index 0.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariableClass(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetVariableIndex, SQ_IntVector *piClasses);
   /**
   * Gets all classes in the workset which at least one observation belongs to.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[out] piClasses                   A vector containing the classes that at least
   *                                         one variable belongs to.
   *                                         The first element in the vector has index 0.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWorksetClasses(SQ_Workset hWorkset, SQ_IntVector *piClasses);


   /** 
   * Gets the scaling for a variable.
   * @see SQ_SetScaling() for further details.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetIndex                The index in the workset of the variable.
   * @param[out] *pScaleInfo                 Will contain the scaling info on exit.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetScaling(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex, SQ_WSScaleInfo *pScaleInfo);

   /**
   * Gets a string with a complete description of a transform.
   * The string returned is never more than 45 characters long.
   *
   * @param[in] pTransformInfo Describes the transform.
   * @param[in,out] szName                   Will contain the name on return, UTF-8 encoded.
   * @param[in] iBufferLen                   Length of the in buffer to get the name.
   *  @return                                Returns SQ_E_OK if success or an error code
   **/
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLongTransformName(SQ_WSTransformInfo *pTransformInfo, char* szName, int iBufferLen);

   /**
   * Gets the name of an observation in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] iWorksetIndex                The index of the observation in the workset.
   * @param[in] iID                          The variable ID to use. Variables always has a primary ID
   *                                         (with index 1) but can have one or more secondary IDs also.
   *                                         you can use GetNrVarObsIDs to obtain the number of IDs.
   * @param[in,out] szName                   The name of the observation.
   * @param[in] iBufferLen                   Length of the in buffer to get the name.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWorksetObservationName(SQ_Workset hWorkset, int iWorksetIndex, int iID, char* szName, int iBufferLen);
   /**
   * Returns the number of observations in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[out] piSize                      The number of observations in the workset.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWorksetObservationSize(SQ_Workset hWorkset, int *piSize);
   /**
   * Returns a short string describing the scaling for a variable.
   * The string returned is never more than 10 characters long.
   *
   * @param[in] pScaleInfo                   Describes the scaling.
   * @param[in,out] szName                   Will contain the name on return, UTF-8 encoded.
   * @param[in] iBufferLen                   Length of the in buffer to get the name.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetShortScalingName(SQ_WSScaleInfo *pScaleInfo, char* szName, int iBufferLen);
   /**
   * Gets a short name for a transform.
   * The string returned is never more than 15 characters long.
   *
   * @param[in] pTransformInfo               Describes the transform.
   * @param[out] szName                      Will contain the name on return, UTF-8 encoded.
   * @param[in] iBufferLen                   Length of the in buffer to get the name.
   *  @return                                Returns SQ_E_OK if success or an error code
   **/
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetShortTransformName(SQ_WSTransformInfo *pTransformInfo, char* szName, int iBufferLen);
   /** 
   * Gets the transformation of a variable.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iIndex                       The index in the workset of the variable.
   * @param[out] pTransformInfo              Will contain the transformation info on exit.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTransform(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iIndex, SQ_WSTransformInfo *pTransformInfo);

   /**
   * Gets class name in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] iClassNo                     The class number to get the name, UTF-8 encoded.
   * @param[in,out] szName                   Will contain the name on exit.
   * @param[in] iBufferLen                   Length of the in buffer to get the name.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWorksetClassName(SQ_Workset hWorkset, int iClassNo, char* szName, int iBufferLen);

   /**
   * Sets class name in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] iClassNo                     The class number to set the name.
   * @param[in] szName                       The name of the class, UTF-8 encoded.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetWorksetClassName(SQ_Workset hWorkset, int iClassNo, const char *szName);
   /**
   * Autogenerate classes from a qualitative variable in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetIndex                The index of the variable in the workset.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AutogenerateClasses(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex);

   /** 
   * Sets the scaling for a variable.
   * The default scaling (Unit variance, centered around the mean and no block scaling)
   * is equivalent to:
   * SQ_SetScaling(hWorkset, iVariableIndex, SQ_ScaleUnitVariance, 0, SQ_CenterAuto, 0, 1, SQ_BlockScaleNone, 0)
   * note that the fScalingWeight, fCenter and iBlockScalingBlock parameters are ignored in this case.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetIndex                The index in the workset of the variable for which
   *                                         scaling is to be set.
   * @param[in] pScaleInfo                   The scaling to use for this variable.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetScaling(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex, SQ_WSScaleInfo *pScaleInfo);
   /**
   * Sets the transform for a variable.
   * Note that transforms can not be set for lagged and expanded variables.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetIndex                The index in the workset of the variable for which
   *                                         transformation is to be set.
   * @param[in] pTransformInfo               The transform to use for this variable.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetTransform(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex, SQ_WSTransformInfo *pTransformInfo);

   /**
   * Checks all variables and sets transformations for them as necessary.
   * The function will remove any previously set transformations.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[out] bAutoTransformAllIfOneNeddsTransform    If equal to 0 only variables needing transformation is transformed
   *                                                        otherwise all variables are transformed if at least one needs transformation.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetAutoTransformAll(SQ_Workset hWorkset, SQ_Bool bAutoTransformAllIfOneNeddsTransform);

   /**
   * Sets the class that an observation belongs to.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] iWorksetObservationIndex     The index of the observation in the workset.
   * @param[in] iClassNo                     The class that the observation should belong to.
   *                                         The class number must be larger than or equal to 0
   *                                         with 0 meaning that the observation belongs to
   *                                         all classes.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetObservationClass(SQ_Workset hWorkset, int iWorksetObservationIndex, int iClassNo);
   /**
   * Sets the class membership to the observations in the workset.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] pClasses                     A vector containing the class number for each 
   *                                         observation. The vector needs to be the same size
   *                                         as the number of observations in the dataset or use NULL to remove the classes.
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetObservationClasses(SQ_Workset hWorkset, SQ_IntVector* pClasses);

   /**
   * Adds a class or clears the classes that a variable belongs to.
   *
   * @param[in] hWorkset                     A handle to the workset.
   * @param[in] eBlock                       The block that the variable belongs to.
   * @param[in] iWorksetIndex                The index of the variable in the workset.
   * @param[in] iClassNo                     If this parameter is larger than 0, that class will be added
   *                                         to the set of classes that the variable belongs to.
   *                                         If it is 0 the set of classes that the variable belongs to
   *                                         will be cleared (it will belong to all classes).
   *  @return                                Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetVariableClass(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex, int iClassNo);

   /**
   * Removes an observation from the workset.
   *
   * @param[in] hWorkset                  A handle to the workset
   * @param[in] iWorksetIndex             The index of the observation in the workset.
   *                                      If iIndex = -1 all observations in the workset will be removed.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_RemoveObservation(SQ_Workset hWorkset, int iWorksetIndex);

   /**
   * Removes a variable from the workset.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in] eBlock                 The block that the variable belongs to.
   * @param[in] iWorksetIndex          The index of the variable in the workset. If iIndex = -1 then 
                                       all variables belonging to the given variable block will be removed.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_RemoveVariable(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iWorksetIndex);

   /** 
   * Returns the index of a observation in the workset.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] iDSIndex               The index in the dataset of the observation.
   * @param[out] hWorksetIndex         Will contain the index in the workset of the observation
   *                                   on return or 0 if it is not in the workset.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetObservationWorksetIndex(SQ_Workset hWorkset, int iDSIndex, int *hWorksetIndex);

   /** 
   * Returns the index of a observation in the dataset.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in] iWorksetIndex          The index in the workset of the observation
   * @param[out] pDatasetIndex         Will contain the index in the dataset of the observation
   *                                   on return.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetObservationDatasetIndex(SQ_Workset hWorkset, int iWorksetIndex, int *pDatasetIndex);

   /**
   * Returns the index of a variable in the workset.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in] iDatasetNumber         The number of the dataset to add from.
   * @param[in] pDSVariableIndices     A vector containing variable indices in the dataset.
   *                                   Ex. If the dataset contains variables A, B and C
   *                                   (in that order), and pDS1VariableIndices contains 1, 1, 3,
   *                                   the variable is A*A*C
   *                                   The vector must have at least one element and a maximum of
   *                                   three.
   * @param[in] pLags                  Lags for the variables indexed by pDS1VariableIndices.
   *                                   This parameter must have the same length as
   *                                   pDS1VariableIndices or be of zero length (if none of the sub-variables
   *                                   are to be lagged). if pLagDistanceVariable is not supplied or <1 the pLag is assumed to be integer steps.
   * @param[in] pLagTimeVariable	   The index of the time variable used to calculate the dynamic lag
   *								   This parameter must have the same length as pLags be null or empty
   *                                   if both pLagTimeVariable and pLagSpeedVariable is empty or a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   * @param[in] pLagSpeedVariable	   The index of the speed variable used to calculate the dynamic lag
   *								   This parameter must have the same length as pLags be null or empty
   *                                   if it both pLagTimeVariable and pLagSpeedVariable a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   * @param[in] eSQBlock                 The block that the variable belongs to.
   * @param[out] pWorksetIndex         Will contain the index in the workset of the variable
   *                                   on return. If the variable does not exist in the workset the index is set to -1.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetVariableWorksetIndex(SQ_Workset hWorkset, int iDatasetNumber, SQ_IntVector pDSVariableIndices, SQ_FloatVector* pLags, SQ_IntVector* pLagTimeVariable, SQ_IntVector* pLagSpeedVariable, SQ_VariableBlock eSQBlock, int *pWorksetIndex);

   /**
   * Returns the name of a variable.
   *
   * @param[in] hWorkset               A handle to the workset
   * @param[in] eBlock                 The block (SQ_BlockX or SQ_BlockY).
   * @param[in] iIndex                 The index of the variable. The first index is 1.
   * @param[in] iID                    The variable ID to use. Variables always has a primary ID
   *                                   (with index 1) but can have one or more secondary IDs also.
   *                                   You can use GetNrVarObsIDs to obtain the number of IDs.
   * @param[in,out] szName             Will contain the variable name on return, UTF-8 encoded.
   * @param[in] iBufferLen             Length of the in buffer to get the name.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetVariableWorksetName(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iIndex, int iID, char* szName, int iBufferLen);


   /** 
   * Gets a variable. @see SQ_AddVariable.
   *
   * @param[in] hWorkset               A handle to the workset.
   * @param[in] eBlock                 The block that the variable belongs to.
   * @param[in] iIndex                 The index of the variable in it's block.
   * @param[out] iDSNumber             The dataset number where .
   * @param[out] pDSVariableIndices    A vector that contains dataset indices
   *                                   of the variables comprising this term.
   *                                   The first element has index 1.
   * @param[out] pLags                 A vector that contains the lags of the variables.
   *                                   The first element has index 1.
   * @param[in] pLagTimeVariable	   A vector that contains the time variable for dynamic lags
   *                                   if both pLagTimeVariable and pLagSpeedVariable is empty or a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   * @param[in] pLagSpeedVariable	   A vector that contains the speed variable for dynamic lags
   *                                   if it both pLagTimeVariable and pLagSpeedVariable a value <1 means that the variable is lagged by step and pLags is rounded to the closest integer value else it is dynamically lagged by this variable.
   *                                   The first element has index 1.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariable(SQ_Workset hWorkset, SQ_VariableBlock eBlock, int iIndex, int* iDSNumber, SQ_IntVector* pDSVariableIndices, SQ_FloatVector* pLags, SQ_IntVector* pLagTimeVariable, SQ_IntVector* pLagSpeedVariable);

   /// @endcond // end of SQM

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQMWORKSET_H_ */
