/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMODEL_H_
#define _SQMODEL_H_

#include "SQDef.h"

/** @file SQModel.h

*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

   /** 
   The object used to identify an opened model.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_ModelHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Model;

#include "SQErrorCodes.h"
#include "SQProject.h"
#include "SQCommon.h"
#include "SQVectorData.h"
#include "SQIntVector.h"
#include "SQPreparePrediction.h"
#include "SQBoolVector.h"
#include "SQModelStatistics.h"

/*********************************************************************************************************************
 * GENERAL MODEL FUNCTIONS
 *********************************************************************************************************************/

   /** 
   *  Retrieves the PreparePrediction object that should be used to set data for a prediction.
   *  The returned object must be released by SQ_ClearPreparePrediction.
   *
   *  @param[in]   pModel                The model to perform a prediction on.  
   *  @param[out]  pPreparePrediction    The PreparePrediction object that should be used to set data for a prediction.   
   *                                     Must be released by SQ_ClearPreparePrediction or is removed when the model is closed.
   *  @return                            Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPreparePrediction(SQ_Model pModel, SQ_PreparePrediction *pPreparePrediction);

/*********************************************************************************************************************
 * GENERAL MODEL INFO
 *********************************************************************************************************************/

   /**
   *  Retrieves the dataset numbers that the model was created from.
   *
   *  @param[in]    pModel            The model handle to use
   *  @param[out]   pDatasetNumbers   A vector with the dataset numbers that the model was created from.
   *                                  The vector must be cleared with SQ_ClearIntVector.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelDatasets(SQ_Model pModel, SQ_IntVector *pDatasetNumbers);

   /**
   *  Retrieves the fitted status of a model.
   *  Data can only be requested from a fitted model.
   *
   *  @param[in]    pModel            The model handle to use
   *  @param[out]   bIsFitted         SQ_True if the model is fitted, SQ_False otherwise.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsModelFitted(SQ_Model pModel, SQ_Bool* bIsFitted);

   /**
   *  Retrieves the name of a model.
   *
   *  @param[in]     pModel            The model handle to use
   *  @param[in,out] szModelName       The name of the model, UTF-8 encoded. The user is responsible to allocate and deallocate the buffer.
   *  @param[in]     iBufferLength     The length of the buffer.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelName(SQ_Model pModel, char *szModelName, int iBufferLength);
   /**
   *  Retrieves the number of the model.
   *
   *  @param[in]    pModel          The model handle to use
   *  @param[out]   iModelNumber    The model number.
   *
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelNumber(SQ_Model pModel, int *iModelNumber);

   /**
   *  Retrieves the title of a model.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in,out]   szModelTitle    The title of the model, UTF-8 encoded. The user is responsible to allocate and deallocate the buffer.
   *  @param[in]   iBufferLength       The length of the buffer.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelTitle(SQ_Model pModel, char *szModelTitle, int iBufferLength);
   
   /**
   *  Retrieves the time of a model when it was last modified.
   *
   *  @param[in]     pModel            The model handle to use
   *  @param[out]    lModifiedTime     The time and date when the model was created or modified, representing the number of seconds elapsed since 00:00 hours, Jan 1, 1970 UTC.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelLastModified(SQ_Model pModel, long *lModifiedTime);

   /**
   *  Retrieves the type of a model as a string.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[in,out]   szModelType     The type of the model, UTF-8 encoded. The user is responsible to allocate and deallocate the buffer.
   *  @param[in]   iBufferLength       The length of the buffer.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelTypeString(SQ_Model pModel, char *szModelType, int iBufferLength);
   /**
   *  Retrieves the type of a model.
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[out]   eModelType        The type of the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see ModelType
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelType(SQ_Model pModel, SQ_ModelType *eModelType);
   /**
   *  Returns 1 if the model is any type of PCA model (PCA-X, PCA-Y, PCA-Class).
   *
   *  @param[in]   pModel              The model handle to use
   *  @param[out]  bIsPCA              SQ_True if the model is a PCA model, otherwise SQ_False. 
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsModelPCA(SQ_Model pModel, SQ_Bool *bIsPCA);
   /**
   *  Returns 1 if the model is any type of PLS model (PLS, PLS-Class, PLS-DA, OPLS, OPLS-DA, OPLS-Class, O2PLS, O2PLS-DA, O2PLS-Class).
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  bIsPLS             SQ_True  if the model is a PLS model, otherwise SQ_False. 
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsModelPLS(SQ_Model pModel, SQ_Bool *bIsPLS);
   /**
   *  Returns 1 if the model is any type of class model (PCA-Class, PLS-Class, OPLS-Class, O2PLS-Class, PLS-DA and O2PLS-DA).
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  bIsClass           SQ_True if the model is a class model, otherwise SQ_False. 
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsModelClass(SQ_Model pModel, SQ_Bool *bIsClass);
   /**
   *  Retrieves the class of a PCA_Class or PLS_Class model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  piClass           the class of this model, if the model is not a class model piClass is set to -1.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelClass(SQ_Model pModel, int *piClass);
   /**
   *  Returns 1 if the model is cross-validated.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  bIsCV              SQ_True if the model is cross-validated, otherwise SQ_False. 
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsModelCrossValidated(SQ_Model pModel, SQ_Bool *bIsCV);
   /**
   *  Retrieves the number of cross-validation rounds in a model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   iCVRounds         The number of cross-validation rounds in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCrossValidationRounds(SQ_Model pModel, int *iCVRounds);
   /**
   *  Retrieves the number of components in a model. It will return number of predictive components despite model type.
   *  Note: for unfitted models the function returns false and sets piNumComp to -1.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piNumComp         The number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfComponents(SQ_Model pModel, int *piNumComp);
   /**
   *  Retrieves the number of predictive components in an OPLS model. Note: for unfitted models the function returns
   *  false and sets piNumComp to -1.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piNumComp         The number of predictive components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfPredictiveComponents(SQ_Model pModel, int *piNumComp);
   /**
   *  Retrieves the number of X orthogonal components in an OPLS/O2PLS model. Note: for unfitted models the function returns
   *  false and sets piNumComp to -1.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piNumComp         The number of X orthogonal components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfXOrthogonalComponents(SQ_Model pModel, int *piNumComp);
   /**
   *  Retrieves the number of Y orthogonal components in an O2PLS model. Note: for unfitted models the function returns
   *  false and sets piNumComp to -1.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piNumComp         The number of Y orthogonal components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfYOrthogonalComponents(SQ_Model pModel, int *piNumComp);

   /**
   *  Retrieves the Iteration matrix from a model.
   *  Number of iterations of the algorithm till convergence 
   *  
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pIterations             A pointer to the Iterations matrix.
   *                                        Number of rows in matrix = number of components in the model.
   *                                        Number of columns in matrix = 1
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetIterations(SQ_Model pModel, SQ_VectorData* pIterations);

   /**
   *  Displays descriptive statistics on the selected variables.
   *
   *  The statistics are computed on the included observations as specified in the workset of the active model, 
   *  and on the raw or transformed or trimmed (if specified in the workset) variables.
   *  If the model is a DA model (PLS-DA, OPLS-DA, O2PLS-DA) the Statistics will be calculated for all classes,
   *  if the model is a regular class model (PCA-Class, PLS-Class etc) the Statistics will be calculated only for the class included in the model.
   *
   *  @param[in] pModel           The model handle to use
   *  @param[in] pColumnXIndices  A list of X column indices to use.
   *                              NULL if all x columns in the model should be used
   *  @param[in] pColumnYIndices  A list of Y column indices to use.
   *                              NULL if all y columns in the model should be used
   *  @param[in,out] pModelStatistics A pointer to the model statistics
   *  @return                         Returns SQ_E_OK if success or an error code
   *
   *  @see SQ_GetColumnXIndexByName
   *  @see SQ_GetColumnYIndexByName
   *  @see SQ_ModelStatistics
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStatistics(SQ_Model pModel, SQ_IntVector* pColumnXIndices, SQ_IntVector* pColumnYIndices, SQ_ModelStatistics *pModelStatistics);

   /**
   *  Retrieves different options from how a model is created
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   poModelOptions    A pointer to a SQ_ModelOptions struct
   *                                  containing information about the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelOptions(SQ_Model pModel, SQ_ModelOptions* poModelOptions);

   /**
   *  CV groups.
   *  The cross validation groups that the observations in the model belongs to.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pCVGroups         A pointer to the CV groups vector.
   *                                  Number of entries in the vector = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCVGroups(SQ_Model pModel, SQ_VectorData* pCVGroups);
   
   /**
   *  Retrieves the Eigen values matrix from a model.
   *  Eigen values of the X matrix.
   *  
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pEigenValues      A pointer to the Iterations matrix.
   *                                  Number of rows in matrix = number of components in the model.
   *                                  Number of columns in matrix = 1
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetEigenValues(SQ_Model pModel, SQ_VectorData *pEigenValues);

   /**
   *  Returns 1 if the specified model is a Finite Impulse Response model (FIR).
   *
   *  FIR models have a name that starts with 'FIR'.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  bIsFIR             SQ_True if the model is a FIR model, SQ_False otherwise.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsImpulseResponseModel(SQ_Model pModel, SQ_Bool* bIsFIR);

   /**
   *  Retrieves the Finite Impulse Response from a model.
   *
   *  The model has to be a Finite Impulse Response model (FIR). 
   *  FIR models have a name that starts with 'FIR'.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent               The component to use
   *                                        1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iColumnXIndex            The index of X column to use
   *  @param[in]   iIntegrationStartLag     The start lag for the integration.
   *  @param[in]   iIntegrationEndLag       The end lag for the integration.
   *  @param[out]   pRespons                A pointer to the Finite Impulse Response matrix.
   *                                        Number of rows in matrix = number of lagged variables 
   *                                        between the interval iIntegrationStartLag and iIntegrationEndLag.
   *                                        Number of columns in matrix = number of y-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetImpulseResponse(SQ_Model pModel, int iComponent, int iColumnXIndex, int iIntegrationStartLag, int iIntegrationEndLag, SQ_VectorData *pRespons);
   


/*********************************************************************************************************************
 * COLUMN INDEX FUNCTIONS
 *********************************************************************************************************************/
   /**
   *  Retrieves the name of a X column given an index in a specific model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iColumnXIndex      The index of the column to get the name of.
   *  @param[in]   iVarID             The index of the Variable ID that will be used to identify the variables in the project. 
   *                                  The variable names in the selected variable ID must be unique or the initialization will fail.
   *  @param[out]  szColumnXName      The name of the given column, UTF-8 encoded.
   *                                  This parameter could be overwritten at the next call to a SIMCA-Q function.
   *
   *  @param[in]   iBufferLength      The length of the buffer.
   *  @return                         Returns SQ_E_OK if success or an error code
   *
   *  @see GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnXNameByIndex(SQ_Model pModel, int iColumnXIndex, int iVarID, char *szColumnXName, int iBufferLength);
   /**
   *  Retrieves the index of a X column given a name in a specific model
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   szColumnXName      The name of the given column to get the index from, UTF-8 encoded.
   *  @param[in]   iVarID             The index of the Variable ID that will be used to identify the variables in the project. 
   *                                  The variable names in the selected variable ID must be unique or the initialization will fail.
   *  @param[out]   piColumnXIndex    The index of the named column.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnXIndexByName(SQ_Model pModel, const char *szColumnXName, int iVarID, int *piColumnXIndex);
   /**
   *  Retrieves the number of X columns in a specific model
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  piColumnXSize      The number of X columns in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnXSize(SQ_Model pModel, int *piColumnXSize);

   /**
   *  Retrieves the name of a Y column given an index in a specific model
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iColumnYIndex      The index of the column to get the name of.
   *  @param[in]   iVarID             The index of the Variable ID that will be used to identify the variables in the project. 
   *                                  The variable names in the selected variable ID must be unique or the initialization will fail.
   *  @param[out]  szColumnYName      The name of the given column, UTF-8 encoded.
   *                                  This parameter could be overwritten at the next call to a SIMCA-Q function.
   *
   *  @param[in]   iBufferLength      The length of the buffer.
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnYNameByIndex(SQ_Model pModel, int iColumnYIndex, int iVarID, char *szColumnYName, int iBufferLength);
   /**
   *  Retrieves the index of a Y column given a name in a specific model
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   szColumnYName      The name of the given column to get the index from, UTF-8 encoded.
   *  @param[in]   iVarID             The index of the Variable ID that will be used to identify the variables in the project. 
   *                                  The variable names in the selected variable ID must be unique or the initialization will fail.
   *  @param[out]   piColumnYIndex    The index of the named column.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see GetNumberOfVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnYIndexByName(SQ_Model pModel, const char *szColumnYName, int iVarID, int *piColumnYIndex);
   /**
   *  Retrieves the number of Y columns in a specific model
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piColumnYSize     The number of Y columns in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnYSize(SQ_Model pModel, int *piColumnYSize);

/*********************************************************************************************************************
 * VARIABLE AND OBSERVATION INFO
 *********************************************************************************************************************/
  
   /**
   *  Retrieves the number of observations in a model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   piNumObs          The number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfObservations(SQ_Model pModel, int *piNumObs);
   /**
   *  Gets the class numbers for each observation in a model
   *
   *  @param[in]   pModel           The model handle to use
   *  @param[out]  piClasses        A vector with the class numbers for 
   *                                each observation in the workset. NOTE: 
   *                                this vector is of ZERO length if the model
   *                                has no classes.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetObservationClasses(SQ_Model pModel, SQ_IntVector* piClasses);
   /**
   *  Retrieves the name of an observation in a model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObsIx             The index of the observation to get the name of.
   *  @param[in]   iObsID             The observation ID to get.
   *  @param[out]  szObsName          The name of the observation, UTF-8 encoded.
   *                                  This parameter could be overwritten at the next call to a SIMCA-Q function.
   *  @param[in]   iBufferLength      The length of the buffer.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetObservationName(SQ_Model pModel, int iObsIx, int iObsID, char *szObsName, int iBufferLength);
   /**
   *  Retrieves the names of the observations in a model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObsID             The index of the Observation ID. 
   *                                        1 for the primary ID in the data set,
   *                                        2 for the first level of secondary ID, 3 for the second level of secondary ID and so on.
   *  @param[out]   pObservationNames The names of the observations.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetObservationNames(SQ_Model pModel, int iObsID, SQ_StringVector *pObservationNames);

  
/*********************************************************************************************************************
 * LEVELS
 *********************************************************************************************************************/

   /**
   *  Get the default probability level from the model.
   *  .95 means 95% probability
   *  In SIMCA the probability level is the critical limit for PModXPS, 
   *  PModXPS values outside the limit will be colored in red in the predictionset.
   *
   *  @param[in]  pModel         The model to use
   *  @param[out] pfPLevel       Default probability level.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDefaultProbabilityLevel(SQ_Model pModel, float *pfPLevel);
   
   /**
   *  Retrieves the confidence level.
   *  The confidence level is used when computing confidence intervals on the parameters.
   *
   *  @param[in]  pModel               The model to use
   *  @param[out] pfConfidenceLevel    A pointer to the confidence level.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetConfidenceLevel(SQ_Model pModel, float *pfConfidenceLevel);
   
   /**
   *  Retrieves the significance level.
   *  The significance level is used when computing the critical limits for DModX and the Hotelling's T2 ellipse
   *
   *  @param[in]  pModel               The model to use
   *  @param[out] pfSignificanceLevel  A pointer to the significance level. 0.05 means 95% probability.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSignificanceLevel(SQ_Model pModel, float *pfSignificanceLevel);

   
/*********************************************************************************************************************
 * C VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the C matrix from a model.
   *  In every dimension, the c's are the weights used to combine the Y's (linearly) to form the scores u. 
   *  The c's express the correlation between the Y's and the t's (X scores).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pC                A pointer to the C matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetC(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pC);
   /**
   *  Retrieves the Y-loadings(C) correlation scaled matrix from a model.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pCCorr            A pointer to the CCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCCorrelation(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pCCorr);
   /**
   *  The c weights for a selected model dimension, computed from all cross validation rounds in the model
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pCcv              A pointer to the Ccv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds in the model.
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCcv(SQ_Model pModel, int iComponent, SQ_VectorData *pCcv);
   /**
   *  The jack-knife standard error of the weights c computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pCcvSE            A pointer to the CcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetCcvSE(SQ_Model pModel, int iComponent, SQ_VectorData *pCcvSE);
   /**
   *  The degrees of freedom for CcvSE.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfCcvStdDevDF     A pointer to the CcvStdDevDF result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetCcvStdDevDF(SQ_Model pModel, int iComponent, float *pfCcvStdDevDF);
   
   /**
   *  Retrieves the Co matrix from a model.
   *  Weights that combine the Y variables (first dimension) or the Y residuals 
   *  (subsequent dimensions) to form the scores Uo. These weights are selected so as 
   *  to minimize the correlation between Uo and T, thereby indirectly between Uo and X.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of Y side orthogonal component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pCo               A pointer to the Co matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pCo);

   /**
   *  Retrieves the Cocv matrix from a model.
   *  Orthogonal weights co from the Y-part of the model, for a selected model dimension,
   *  computed from the selected cross validation round.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pCocv             A pointer to the Cocv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCocv(SQ_Model pModel, int iComponent, SQ_VectorData* pCocv);

   /**
   *  Retrieves the CocvSE matrix from a model.
   *  Standard error of the cross validated loadings co.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pCocvSE           A pointer to the CocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCocvSE(SQ_Model pModel, int iComponent, SQ_VectorData* pCocvSE);

   /**
   *  CV-ANOVA, ANalysis Of VAriance testing of Cross-Validated predictive residuals, 
   *  is a diagnostic tool for assessing the reliability of PLS, OPLS and O2PLS models.
   *
   *  SS - Total corr (Total corrected), SS of the Y of the workset corrected for the mean. 
   *       Regression, Fraction of Total Corrected SS accounted for by the model, estimated via the cross validation principle.
   *       Residual, Difference between Total Corrected and Regression SS, i.e., the fraction of 
   *       Total Corrected unaccounted for by the model.
   *  DF - Total corr, Regression, residual, The number of degrees of freedom (DF). This is an 
   *       approximate number based on the experience that PLS needs half the components to reach the same explanation of Y as principal components regression. 
   *  MS - Total corr, Regression, residual, By dividing each SS by the respective DF, the corresponding mean squares (MS), or variances, are obtained. 
   *  F  - The F-test, based on the ratio MS Regression/MS Residual, then formally assesses the significance of the model. 
   *  p  - The p-value indicates the probability level where a model with this F-value may be the result of just chance. The common practice is to interpret a p-value lower than 0.05 as pointing to a significant model.
   *  SD - Standard deviation, Square root of MS.
   *
   *  An example: Returns the following matrix 
   *  Model       SS          DF          MS          F           p            SD
   *  Mean        3,75        15          0,25		
   *  OPLS(A=1)   0,954741379 13          0,073441645 19,03047405 0,000137421
   *  diff        2,795258621 2           1,39762931		
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  poCVAnovaTable     A pointer to a SQ_VectorData containing 
   *                                  the anova table for the the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCVAnovaTable(SQ_Model pModel, SQ_VectorData* poCVAnovaTable);

/*********************************************************************************************************************
 * Correlation
 *********************************************************************************************************************/

   /**
   *  Retrieves the Correlation matrix from a model.
   *  The correlation matrix shows the pair-wise correlation between all variables (X and Y) in the current workset, scaled and transformed as the workset. 
   *  Each variable is displayed on one row and one column in the correlation matrix, and the correlation between two variables is shown in the cell where the two variables intersect.
   *  The function fails if the model is not a class model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  pCorrMatrix        A pointer to the Correlation matrix.
   *                                  Number of rows in matrix = number of variables in the model.
   *                                  Number of columns in matrix = number of variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCorrelationMatrix(SQ_Model pModel, SQ_VectorData* pCorrMatrix);

/*********************************************************************************************************************
 * Coefficient VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the Coefficients from a model.
   *  PLS Regression coefficients corresponding to the unscaled and uncentered X and Y.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeff            A pointer to the pCoeff matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficients(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeff);
   /**
   *  Retrieves the centered scaled coefficients from a model.
   *  PLS Regression coefficients corresponding to the centered and scaled X and the scaled (but uncentered) Y.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[in]   bResolveHierarchical  If the hierarchical coefficiants should be resolved or not
   *
   *  @param[out]   pCoeffCS          A pointer to the pCoeffCS matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCS(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_ResolveHierachicalState bResolveHierarchical, SQ_VectorData *pCoeffCS);
   /**
   *  Confidence Interval for CoefficientsCS.
   *  The function fails if the model is not a PLS model and if the model is not cross-validated
   *  or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffCSCI        A pointer to the pCoeffCSCI matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCSCI(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffCSCI);
   /**
   *  PLS Regression coefficients corresponding to the unscaled but centered X and unscaled Y.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffCenterd     A pointer to the pCoeffCenterd matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCenterd(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffCenterd);
   /**
   *  PLS Regression coefficients corresponding to the scaled and centered X but the unscaled and uncentered Y.
   *  (the double scaling of the extended terms has been removed).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffMLR        A pointer to the pCoeffMLR matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsMLR(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffMLR);
   /**
   *  Retrieves the lagged centered scaled coefficients from a model.
   *  Coefficients (for Scaled and centered data), of a lagged X variable, for a selected Y, as a functions of Lags.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   iColumnXIndex      The index of the X column to use.
   *  @param[in]   iColumnYIndex      The index of the Y column to use.
   *  @param[out]   pCoeffCSLag       A pointer to the pCoeffCSLag matrix.
   *                                  Number of rows in matrix = 1 (only one Y-variable).
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCSLag(SQ_Model pModel, int iComponent, int iColumnXIndex, int iColumnYIndex, SQ_VectorData *pCoeffCSLag);
   /**
   *  PLS Regression coefficients corresponding to the scaled and centered X 
   *  and the scaled (but uncentered) Y computed from all the cross validation rounds in the model.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   iColumnYIndex      The index of the Y column to use.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffCScv        A pointer to the pCoeffCScv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds in the model.
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCScv(SQ_Model pModel, int iComponent, int iColumnYIndex, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffCScv);
   /**
   *  The jack-knife standard error of the coefficients CoeffCS computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   iColumnYIndex      The index of the Y column to use.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffCScvSE      A pointer to the pCoeffCScvSE matrix.
   *                                  Number of rows in matrix = 1 (only one Y-variable).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCScvSE(SQ_Model pModel, int iComponent, int iColumnYIndex, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffCScvSE);
   /**
   *  The degrees of freedom for CoeffCScvSE
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pfCoeffCScvSEDF   A floating point value.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCScvSEDF(SQ_Model pModel, int iComponent, float *pfCoeffCScvSEDF);
   /**
   *  The jack-knife standard error of the coefficients CoeffCS computed from the cross validations,
   *  of a lagged variable x, for a selected Y as a function of lags.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   iColumnXIndex      The index of the X column to use.
   *  @param[in]   iColumnYIndex      The index of the Y column to use.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pCoeffCScvSELag   A pointer to the pCoeffCScvSELag matrix.
   *                                  Number of rows in matrix = 1 (only one Y-variable).
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsCScvSELag(SQ_Model pModel, int iComponent, int iColumnXIndex, int iColumnYIndex, SQ_ReconstructState bReconstruct, SQ_VectorData *pCoeffCScvSELag);
 
   /**
   *  PLS rotated regression coefficients.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[in]   eCoeffRotatedType  The type of CoefficientRotated. Can be one of following:
   *                                  SQ_XYUnscaledUncenterd 
   *                                  SQ_XUnscaledCenterdYUnscaledUncenterd 
   *                                  SQ_XScaledCenterdYScaledUncenterd
   *                                  @ see SQ_CoefficientsRotatedType
   *  @param[out]   pCoeffRotated     A pointer to the pCoeffRotated matrix.
   *                                  Number of rows in matrix = number of Y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of X-variables + 1 (Constant).
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetCoefficientsRotated(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_ReconstructState bReconstruct, SQ_CoefficientsRotatedType eCoeffRotatedType, SQ_VectorData *pCoeffRotated);

/*********************************************************************************************************************
 * Contribution VECTORS
 *********************************************************************************************************************/
   /**
   *  Scores single weight contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObs1Ix            Index in the observation matrix (the training set) 
   *                                  for the reference observation (from observation).
   *                                  0 if the average is to be used.
   *  @param[in]   iObs2Ix            Index in the observation matrix (the training set) of the observation 
   *                                  for which the contributions are to be calculated (to observation).
   *  @param[in]   eWeightType        The type of weight.
   *                                  If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                  If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]   iComponent         The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *                                  For an OPLS model with weight CoeffCS or VIP the component must be the last predictive.
   *  @param[in]   iYVar              The index of the Y variable to use if eWeightType is CoeffCS or CoeffCSRaw. 
   *                                  If eWeightType is something else, this parameter is ignored. 
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrSSW         A pointer to the Scores single weight contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScoresSingleWeight(SQ_Model pModel, int iObs1Ix, int iObs2Ix, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrSSW);
   /**
   *  Scores single weight group contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pObs1Ix            A list of indices in the observation matrix (the training set) 
   *                                  for the reference observations (from observation).
   *                                  NULL if the average is to be used.
   *  @param[in]   pObs2Ix            A list of indices in the observation matrix (the training set) of the observations 
   *                                  for which the contributions are to be calculated (to observation).
   *  @param[in]   eWeightType        The type of weight.
   *                                  If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                  If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]   iComponent         The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *                                  For an OPLS model with weight CoeffCS or VIP the component must be the last predictive.
   *  @param[in]   iYVar              The index of the Y variable to use if eWeightType is CoeffCS or CoeffCSRaw. 
   *                                  If eWeightType is something else, this parameter is ignored. 
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrSSW         A pointer to the Scores single weight group contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScoresSingleWeightGroup(SQ_Model pModel, SQ_IntVector* pObs1Ix, SQ_IntVector* pObs2Ix, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrSSW);
   /**
   *  Scores multi weight contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObs1Ix            Index in the observation matrix (the training set) 
   *                                  for the reference observation (from observation).
   *                                  0 if the average is to be used.
   *  @param[in]   iObs2Ix            Index in the observation matrix (the training set) of the observation 
   *                                  for which the contributions are to be calculated (to observation).
   *  @param[in]   pWeightType        An int vector containing SQ_WeightType enums
   *                                  If the model is a PCA model this parameter must be P.
   *                                  If the model is a PLS model this parameter must be P or WStar.
   *                                  If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                  for weight Po the component argument is the orthogonal component number
   *                                  for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrSMW         A pointer to the Scores multi weight contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScoresMultiWeight(SQ_Model pModel, int iObs1Ix, int iObs2Ix, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrSMW);
   /**
   *  Scores multi weight group contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pObs1Ix            A list of indices in the observation matrix (the training set) 
   *                                  for the reference observations (from observation).
   *                                  NULL if the average is to be used.
   *  @param[in]   pObs2Ix            A list of indices in the observation matrix (the training set) of the observations 
   *                                  for which the contributions are to be calculated (to observation).
   *  @param[in]   pWeightType        An int vector containing SQ_WeightType enums.
   *                                  If the model is a PCA model this parameter must be P.
   *                                  If the model is a PLS model this parameter must be P or WStar.
   *                                  If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                  for weight Po the component argument is the orthogonal component number
   *                                  for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrSMW         A pointer to the Scores multi weight group contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScoresMultiWeightGroup(SQ_Model pModel, SQ_IntVector* pObs1Ix, SQ_IntVector* pObs2Ix, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrSMW);
   /**
   *  DModX contribution.
   *  Contribution is used to understand how an observation differs from the others in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObsIx             Index in the observation matrix (the training set) .
   *  @param[in]   eWeightType        The type of weight.
   *                                  If the model is a PCA model this parameter must be Normalized, RX.
   *                                  If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]   iComponent         The component of the weight. 
   *                                  For an OPLS model the component must be the last predictive.
   *  @param[in]   iYVar              The index of the Y variable to use if eWeightType is CoeffCS. 
   *                                  If eWeightType is something else, this parameter is ignored. 
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrDModX       A pointer to the DModX contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModX(SQ_Model pModel, int iObsIx, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrDModX);
   /**
   *  DModX group contribution.
   *  Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pObsIx             A list of indices in the observation matrix (the training set) 
   *  @param[in]   eWeightType        The type of weight.
   *                                  If the model is a PCA model this parameter must be Normalized, RX.
   *                                  If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]   iComponent         The component of the weight.
   *                                  For an OPLS model the component must be the last predictive.
   *  @param[in]   iYVar              The index of the Y variable to use if eWeightType is CoeffCS. 
   *                                  If eWeightType is something else, this parameter is ignored. 
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pContrDModX       A pointer to the Scores single weight group contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModXGroup(SQ_Model pModel, SQ_IntVector* pObsIx, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pContrDModX);
   /**
   *  DModY contribution.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components
   *  or if the model isn't a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iObsIx             Index in the observation matrix (the training set) .
   *  @param[in]   eWeightType        The type of weight.
   *                                  This parameter must be Normalized or RY.
   *  @param[in]   iComponent         The component of the weight. 
   *                                  For an OPLS model the component must be the last predictive.
   *  @param[out]  pContrDModY        A pointer to the DModY contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModY(SQ_Model pModel, int iObsIx, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pContrDModY);
   /**
   *  DModY group contribution.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.doc" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components
   *  or if the model isn't a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pObsIx             A list of indices in the observation matrix (the training set) 
   *  @param[in]   eWeightType        The type of weight.
   *                                  This parameter must be Normalized or RY.
   *  @param[in]   iComponent         The component of the weight. 
   *                                  For an OPLS model the component must be the last predictive.
   *  @param[out]  pContrDModY        A pointer to the DModY contribution matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of Y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModYGroup(SQ_Model pModel, SQ_IntVector* pObsIx, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pContrDModY);

/*********************************************************************************************************************
 * DMod VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the DModX matrix from a model.
   *  
   *  Distance to the model in X space (row residual SD), after A components (the selected dimension), 
   *  for the observations used to fit the model. 
   *  If you select component 0, it is the standard deviation of the observations with scaling and centering 
   *  as specified in the workset (no row means subtracted), i.e., 
   *  it is the distance to the origin of the scaled coordinate system.
   *
   *  @param[in]   pModel                   The model handle to use
   *  @param[in]   pComponents              A list of component indices to use.
   *                                        1 for component 1 in the model, 2 for component 2 and so on.
   *                                        NULL if all components in the model should be used.
   *                                        If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                        the first row will contain the data after 0 components.
   *                                        For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bNormalized              If True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                        If False, they will be in absolute values.
   *  @param[in]   bModelingPowerWeighted   If True, the function will weight the residuals by the modeling power of the variables.
   *  @param[out]  pDModX                   A pointer to the DModX matrix.
   *                                        Number of rows in matrix = number of components chosen (length of pComponents).
   *                                        Number of columns in matrix = number of observations in the model.
   *  @return                               Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetDModX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData *pDModX);

   /**
   *  Retrieves the DModXCrit matrix from a model.
   *  The critical limit for DModX for a certain component and probability level.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bNormalized        If SQ_True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                  If SQ_False, they will be in absolute values.
   *  @param[in]   fLevel             The probability level, .95 means 95% probability. 
   *                                  The value should be between 0.00001 and 0.99999.
   *                                  If -1, the default level from the project is used.
   *  @param[out]  pfDModXCrit        A pointer to the DModXCrit result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetDModXCrit(SQ_Model pModel, int iComponent, SQ_NormalizedState bNormalized, float fLevel, float *pfDModXCrit);

   /**
   *  Retrieves the DModY matrix from a model.
   *  
   *  Distance to the model in the Y space, after A components (the selected dimension), 
   *  for the observations used to fit the model. 
   *  If you select, component 0, it is the Standard deviation (no row means subtracted) 
   *  of the observations with scaling (or none) as specified in the workset, i.e., 
   *  it is the distance to the origin of the coordinate system.
   *
   *  The function fails if the model isn't a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                  the first row will contain the data after 0 components.
   *                                  For an OPLS model, the last predictive component and zero are the only valid ones.
   *  @param[in]   bNormalized        If true, the results will be in units of standard deviation of the pooled RSD of the model
   *                                  If false, they will be in absolute values.
   *  @param[out]  pDModY             A pointer to the DModY matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetDModY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_NormalizedState bNormalized, SQ_VectorData *pDModY);

/*********************************************************************************************************************
 * M VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the MPowX matrix from a model.
   *  The modeling power of variable X is the fraction of its standard deviation explained by the model after the specified component.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]  pMPowX             A pointer to the MPowX matrix.
   *                                  Number of rows in matrix = number of components entered.
   *                                  Number of columns in matrix = number of x variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetMPowX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pMPowX);

/*********************************************************************************************************************
 * O VECTORS
 *********************************************************************************************************************/
   /**
   *  Returns the R2/Q2 overview part of the model information matrix for an OPLS model.
   *  The returned matrix is defined as follows:
   *  A matrix containing six rows, the first contains R2X values, the second R2XCum values,
   *  the third Q2 values, the fourth Q2Cum values, the fifth R2Y values and the sixth R2YCum values.
   *  There will be as many columns as the number of components.
   *  The first columns contains values for the predictive
   *  components. If there are orthogonal components, they will follow the predictive components.
   *
   *  @param[in]  pModel              The model handle to use
   *  @param[out] pModelR2Q2Overview  The overview results.
   *  @return                         Returns SQ_E_OK if success or an error code
   *      
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetOPLSR2Q2Overview(SQ_Model pModel, SQ_VectorData *pModelR2Q2Overview);

   /**
   *  Retrieves the OLevX matrix from a model.
   *  The leverage is a measure of the influence of a point (observation) on the PC model or the PLS model in the X space.
   *  The observations leverages are computed as the diagonal elements of the matrix H0 after A dimensions.
   *  H0 = T[T'T]-1T'.
   *  The function fails if the model doesn't have any components. 
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pOLevX            A pointer to the OLevX matrix.
   *                                  Number of rows in matrix = number of components entered.
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetOLevX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pOLevX);
   /**
   *  Retrieves the OLevY matrix from a model.
   *  The leverage is a measure of the influence of a point (observation) on the PLS model in the Y space.
   *  The observations leverages are computed as the diagonal elements of the matrix Hy after A dimensions.
   *  Hy = U[U'U]-1U'.
   *  The function fails if the model isn't a PLS model
   *  or if the model doesn't have any components.  
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pOLevY            A pointer to the OLevY matrix.
   *                                  Number of rows in matrix = number of components entered.
   *                                  Number of columns in matrix = 1
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetOLevY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pOLevY);
   /**
   *  Retrieves the ORisk matrix from a model.
   *  This is a measure of the sensitivity of the residual, of a selected Y, 
   *  for an observation in the training set. 
   *  It is computed from the difference between the residual standard deviation of the selected Y, 
   *  when the observation is and is not in the training set model.
   *
   *  The function fails if the model isn't a PLS model
   *  or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iColumnYIndex      The index of Y column to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pORisk            A pointer to the ORisk matrix.
   *                                  Number of rows in matrix = number of components entered.
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetORisk(SQ_Model pModel, int iColumnYIndex, SQ_IntVector* pComponents, SQ_VectorData *pORisk);
   /**
   *  Retrieves the ORiskPooled matrix from a model.
   *  This is a measure of the sensitivity of the pooled Y residuals, 
   *  of an observation in the training set. 
   *  It is computed from the difference between the pooled Y residual standard deviation, 
   *  when the observation is and is not in the training set model.
   *
   *  The function fails if the model isn't a PLS model
   *  or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pORiskPooled      A pointer to the ORiskPooled matrix.
   *                                  Number of rows in matrix = number of components entered.
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetORiskPooled(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pORiskPooled);

/*********************************************************************************************************************
 * P VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the P matrix from a model.
   *  The Loading weights of the X part of the model. 
   *  With a PC model, the p's represent the importance of the variables in that dimension. 
   *  With a PLS model, the p's express the importance of the variables in approximating X.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pP                A pointer to the P matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetP(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pP);

   /**
   *  Retrieves the Pc matrix from a model.
   *  X loading p and Y loading weight c combined to one vector.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pPc               A pointer to the Pc matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPc(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pPc);

   /**
   *  Retrieves the PcCorr matrix from a model.
   *  X loading p and Y loading weight c scaled as correlation coefficients between X and t (p) and Y and u (c), 
   *  and combined to one vector.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pPcCorr           A pointer to the PcCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPcCorr(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pPcCorr);

   /**
   *  Retrieves the loadings(P) correlation scaled matrix from a model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pPCorr            A pointer to the PCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPCorrelation(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pPCorr);
   /**
   *  Retrieves the Pcv matrix from a model.
   *  The p weights for a selected model dimension, 
   *  computed from the cross validation round CV=the number set in the project.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pPcv              A pointer to the Pcv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds set the project.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPcv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pPcv);
   /**
   *  Retrieves the PcvSE matrix from a model.
   *  The jack knife standard error of the loadings p computed from the cross validations.
   *  The function fails if the model doesn't have any components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pPcvSE            A pointer to the PcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData *pPcvSE);

   /**
   *  Calculates the percentile for the cvSE vectors which can be used to calculate confidence intervalsl, for example PcvSE.
   *  to get confidence intervals, multiply the cvSE values with the percentile value. 
   *  see GetCcvSE, GetCocvSE, GetCoefficientsCScvSE,GetPcvSE,GetPccvSE,GetPocvSE,GetPqcvSE,GetQcvSE
   *      GetQocvSE,GetVIPcvSE,GetWcvSE,GetWStarCcvSE,GetWStarcvSE and GetWocvSE
   *
   *  
   *  @param[in]   pModel             The model handle to use
   *                                  to the original domain.
   *  @param[in]   fSignificance      the significance level. 0.05 means 95% probability.
   *  @param[out]  pfStudentsT        the t value needed to get the confidence intervals
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetcvSEPercentile(SQ_Model pModel, float fSignificance, float* pfStudentsT);


   /**
   *  The degrees of freedom for PcvSE.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfPcvSEDF         The degrees of freedom for PcvSE.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPcvSEDF(SQ_Model pModel, int iComponent, float *pfPcvSEDF);

   /**
   *  Retrieves the PccvSE matrix from a model.
   *  Jack Knife standard error of the combined X loading P and Y loading weight c computed from all rounds of cross-validation.
   *  The function fails if the model is not a OPLS model and if the model doesn't have any Y side orthogonal components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iCompontent        Components to use.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pPccvSE            A pointer to the PccvSE matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of y-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPccvSE(SQ_Model pModel, int iCompontent, SQ_ReconstructState bReconstruct, SQ_VectorData* pPccvSE);

   /**
   *  Retrieves the permutation test matrix from a model. Only valid for PLS and PLS-DA models.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iYvariable         The Y variable to calculate the permutation test for.
   *  @param[in]   iNumOfPermutations Number of permutations to do.
   *  @param[out]  pPermutationTest   A float matrix with three columns, Correlation, R2 and Q2. 
   *                                  Number of rows equals number of permutations.
   *
   *  @return                         Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPermutationTest(SQ_Model pModel, int iYvariable, int iNumOfPermutations, SQ_FloatMatrix *pPermutationTest);

   /**
   *  Retrieves the PLag matrix from a model.
   *  Loadings p of a lagged variable as a function of the Lags.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]   pPLag             A pointer to the PLag matrix.
   *                                  Number of rows in matrix = 1 (only one x-variable).
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPLag(SQ_Model pModel, int iComponent, int iColumnXIndex, SQ_VectorData *pPLag);
   /**
   *  Retrieves the PModX matrix from a model.
   *  Probability of belonging to the model in the X space, for observations used to fit the model. 
   *  Component 0 corresponds to a point model, i.e., the center of the coordinate system.
   *  Observations with probability of belonging of less than 5% are considered to be non-members, 
   *  i.e., they are different from the normal observations used to build the model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                  the first row will contain the data after 0 components.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pPModX            A pointer to the PModX matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPModX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pPModX);
   /**
   *  Retrieves the PModY matrix from a model.
   *  Probability of belonging to the model in the Y space, 
   *  for observations used to fit the model. 
   *  Component 0 corresponds to a point model, i.e., the center of the coordinate system.
   *  Observations with probability of belonging of less than 5% are considered to be non-members, 
   *  i.e., they are different from the normal observations used to build the model.
   *
   *  The function fails if the model isn't a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                  the first row will contain the data after 0 components.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pPModY            A pointer to the PModY matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetPModY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pPModY);

   /**
   *  Retrieves the Po matrix from a model.
   *  Orthogonal loadings of the X-part of the model. Po expresses the importance of the variables in 
   *  approximating X variation orthogonal to Y, in the selected component.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pPo                A pointer to the Po matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pPo);

   /**
   *  Retrieves the Pocv matrix from a model.
   *  Orthogonal loadings Po from the X-part of the model, for a selected model
   *  dimension, computed from the selected cross validation round.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pPocv             A pointer to the Pocv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPocv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pPocv);

   /**
   *  Retrieves the PocvSE matrix from a model.
   *  Standard error of the cross validated loadings po.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pPocvSE           A pointer to the PocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPocvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pPocvSE);

   /**
   *  Retrieves the PoCorr matrix from a model.
   *  Orthogonal loadings Po, scaled as the correlation coefficient
   *  between X and To, in the selected component.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pPoCorr           A pointer to the PoCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPoCorrelation(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pPoCorr);

   

   /**
   *  Retrieves the PoSo matrix from a model.
   *  Loading of the Y-part of the model (po) and the projection of To on Y (so) concatenated to one vector.
   *  The function fails if the model is not a OPLS model and if the model doesn't have any Y side orthogonal components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pCompontentsList   A list of X side components to use.
   *                                  NULL if all components model should be used
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pPoso              A pointer to the pPoso matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of y-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPoSo(SQ_Model pModel, SQ_IntVector* pCompontentsList, SQ_ReconstructState bReconstruct, SQ_VectorData* pPoso);

   /**
   *  Retrieves the PoSoCorr matrix from a model.
   *  Loading of the Y-part of the model (po) and the projection of To on Y (so) scaled as correlation coefficients concatenated to one vector.
   *  The function fails if the model is not a OPLS model and if the model doesn't have any Y side orthogonal components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pCompontentsList   A list of components to use.
   *                                  NULL if all components in the model should be used
   *  @param[out]   pPosoCorr         A pointer to the PosoCorr matrix.
   *                                  Number of rows in matrix = x side orthogonal components.
   *                                  Number of columns in matrix =  the x and Y terms of the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPoSoCorr(SQ_Model pModel, SQ_IntVector* pCompontentsList, SQ_VectorData* pPosoCorr);

   /**
   *  Retrieves the Pq matrix from a model.
   *  X loading weight p and Y loading weight q combined to one vector.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all predictive components in the model will be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pPq                A pointer to the Pq matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPq(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pPq);

   /**
   *  Retrieves the PqCorr matrix from a model.
   *  X loading weight p and Y loading weight q scaled as correlation coefficients, combined to one vector.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all predictive components in the model will be used.
   *  @param[out]  pPqCorr            A pointer to the PqCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPqCorr(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pPqCorr);

   /**
   *  Retrieves the PqcvSE matrix from a model.
   *  The jack knife standard error of the X loading weight p and Y loading weight q
   *  computed from the cross validations, combined to one vector.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The predictive component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pPqcvSE            A pointer to the PqcvSE matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPqcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pPqcvSE);


/*********************************************************************************************************************
 * Q VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the Q matrix from a model.
   *  Loadings of the Y-part of the model. Q expresses the importance 
   *  of the variables in approximating Y variation correlated to X, in the selected 
   *  component. Y variables with large Q (positive or negative) are highly correlated with T (and X).
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQ                A pointer to the Q matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pQ);

   /**
   *  Retrieves the QCorr matrix from a model.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQCorr            A pointer to the QCorr matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQCorrelation(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pQCorr);

   /**
   *  Retrieves the Qcv matrix from a model.
   *  The q loadings for a selected model dimension, computed from the selected cross validation round.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pQcv              A pointer to the Qcv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQcv(SQ_Model pModel, int iComponent, SQ_VectorData* pQcv);

   /**
   *  Retrieves the QcvSE matrix from a model.
   *  Standard error of the cross validated loadings q.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pQcvSE            A pointer to the QcvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQcvSE(SQ_Model pModel, int iComponent, SQ_VectorData* pQcvSE);

   /**
   *  Retrieves the Qo matrix from a model.
   *  Orthogonal loadings of the Y-part of the model. Qo expresses the importance 
   *  of the variables in approximating Y variation orthogonal to X, in the selected component.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pQo               A pointer to the Qo matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pQo);

   /**
   *  Retrieves the Qocv matrix from a model.
   *  Orthogonal loadings Qo from the Y-part of the model, for a selected model
   *  dimension, computed from the selected cross validation round.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pQocv             A pointer to the Qocv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQocv(SQ_Model pModel, int iComponent, SQ_VectorData* pQocv);

   /**
   *  Retrieves the QocvSE matrix from a model.
   *  Standard error of the cross validated loadings qo.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pQocvSE           A pointer to the QocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQocvSE(SQ_Model pModel, int iComponent, SQ_VectorData* pQocvSE);

   /**
   *  Retrieves the Q2 matrix from a model.
   *  The fraction of the total variation of the X's (PC) or Y's (PLS) that can be predicted by each component.
   
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pQ2               A pointer to the Q2 matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2(SQ_Model pModel, SQ_VectorData *pQ2);
   /**
   *  Retrieves the Q2Cum matrix from a model.
   *  The cumulative Q2 for each component.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pQ2Cum            A pointer to the Q2Cum matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2Cum(SQ_Model pModel, SQ_VectorData *pQ2Cum);
   /**
   *  Retrieves the Q2VX matrix from a model.
   *  The predicted fraction (according to cross-validation) of the variation of the X variable
   *  for the selected components. For a PC model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQ2VX             A pointer to the Q2VX matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2VX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pQ2VX);
   /**
   *  Retrieves the Q2VXCum matrix from a model.
   *  The cumulative predicted fraction (cross-validation) of the variation 
   *  of the X variables for a PC model for the selected components.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQ2VXCum          A pointer to the Q2VXCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2VXCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pQ2VXCum);
   /**
   *  Retrieves the Q2VY matrix from a model.
   *  The predicted fraction (according to cross-validation) of the variation
   *  of the Y variables for the selected components.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQ2VY             A pointer to the Q2VY matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2VY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pQ2VY);
   /**
   *  Retrieves the Q2VYCum matrix from a model.
   *  The cumulative Q2, for the selected components, for the Y variables.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pQ2VYCum          A pointer to the Q2VYCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2VYCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pQ2VYCum);

   /**
   *  Retrieves the Q2CumProgression matrix from a model.
   *  Cumulative Q2 for the extracted components, showing the progression of cumulative values for each added orthogonal component in the OPLS model, e.g. 1+0, 1+1, 1+2.
   *  The function fails if the model is not a 1+x+0 OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  pQ2CumProgression  A pointer to the Q2CumProgression matrix.
   *                                  Number of rows in matrix = 1 + XSideOrthogonalComponents + XSidePCAComponents.
   *                                  Number of columns in matrix = 1.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQ2CumProgression(SQ_Model pModel, SQ_VectorData* pQ2CumProgression);

/*********************************************************************************************************************
 * R VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the R matrix from a model.
   *  R is the projection of Uo on X.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR                A pointer to the R matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pR);

   /**
   *  Retrieves the R2VX matrix from a model.
   *  The fraction of the variation of the X variables explained by the selected components.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VX             A pointer to the R2VX matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VX);
   /**
   *  Retrieves the R2VXAdj matrix from a model.
   *  The fraction of the variation of the X variables explained by the selected components, 
   *  adjusted for degrees of freedom. (Variance explained by that component).
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents       A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VXAdj          A pointer to the R2VXAdj matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VXAdj(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VXAdj);
   /**
   *  Retrieves the R2VXAdjCum matrix from a model.
   *  The cumulative fraction of the variation of the X variables explained after the selected component, 
   *  adjusted for degrees of freedom. (Cumulative Variance explained after the selected component).
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VXAdjCum       A pointer to the R2VXAdjCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VXAdjCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VXAdjCum);
   /**
   *  Retrieves the R2VXCum matrix from a model.
   *  The cumulative fraction of the variation of the X variables explained after the selected component.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VXCum          A pointer to the R2VXCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VXCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VXCum);
   /**
   *  Retrieves the R2VY matrix from a model.
   *  The fraction of the variation of the Y variables explained by the selected component.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VY             A pointer to the R2VY matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VY);
   /**
   *  Retrieves the R2VYAdj matrix from a model.
   *  The fraction of the variation of the Y variables explained by the selected component, 
   *  adjusted for degrees of freedom. (Variance explained by that component).
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VYAdj          A pointer to the R2VYAdj matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents)
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VYAdj(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VYAdj);
   /**
   *  Retrieves the R2VYAdjCum matrix from a model.
   *  The cumulative fraction of the variation of the Y variables explained after the selected component, 
   *  adjusted for degrees of freedom. (Cumulative Variance explained after the selected component).
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VYAdjCum       A pointer to the R2VYAdjCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VYAdjCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VYAdjCum);
   /**
   *  Retrieves the R2VYCum matrix from a model.
   *  The cumulative fraction of the variation of the Y variables explained after the selected component.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pR2VYCum          A pointer to the R2VYCum matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2VYCum(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pR2VYCum);
   /**
   *  Retrieves the R2X matrix from a model.
   *  Fraction of Sum of Squares (SS) of all the X's explained by each component.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2X              A pointer to the R2X matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2X(SQ_Model pModel, SQ_VectorData *pR2X);
   /**
   *  Retrieves the R2XAdj matrix from a model.
   *  Variance of all the X's explained by each component.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2XAdj           A pointer to the R2XAdj matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2XAdj(SQ_Model pModel, SQ_VectorData *pR2XAdj);
   /**
   *  Retrieves the R2XAdjCum matrix from a model.
   *  Cumulative variance of all the X's explained after each extracted component.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  pR2XAdjCum         A pointer to the R2XAdjCum matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2XAdjCum(SQ_Model pModel, SQ_VectorData *pR2XAdjCum);
   /**
   *  Retrieves the R2XCum matrix from a model.
   *  Cumulative fraction of Sum of Squares (SS) of all the X's explained after each extracted component.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2XCum           A pointer to the R2XCum matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2XCum(SQ_Model pModel, SQ_VectorData *pR2XCum);
   /**
   *  Retrieves the R2Y matrix from a model.
   *  Fraction of Sum of Squares (SS) of all the Y's explained by each component.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2Y              A pointer to the R2Y matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2Y(SQ_Model pModel, SQ_VectorData *pR2Y);
   /**
   *  Retrieves the R2YCum matrix from a model.
   *  Cumulative fraction of Sum of Squares (SS) of all the Y's explained after each extracted component.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2YCum           A pointer to the R2YCum matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2YCum(SQ_Model pModel, SQ_VectorData *pR2YCum);
   /**
   *  Retrieves the R2YAdj matrix from a model.
   *  Variance of all the Y's explained by each component.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2YAdj           A pointer to the R2YAdj matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2YAdj(SQ_Model pModel, SQ_VectorData *pR2YAdj);
   /**
   *  Retrieves the R2YAdjCum matrix from a model.
   *  Cumulative variance of all the Y's explained after each extracted component.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pR2YAdjCum        A pointer to the R2YAdjCum matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2YAdjCum(SQ_Model pModel, SQ_VectorData *pR2YAdjCum);

   /**
   *  Retrieves the R2CumProgression matrix from a model.
   *  Cumulative R2 for the extracted components, showing the progression of cumulative values for each added orthogonal component in the OPLS model, e.g. 1+0, 1+1, 1+2.
   *  The function fails if the model is not a 1+x+0 OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]  pR2CumProgression  A pointer to the R2CumProgression matrix.
   *                                  Number of rows in matrix = 1 + XSideOrthogonalComponents + XSidePCAComponents.
   *                                  Number of columns in matrix = 1.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetR2CumProgression(SQ_Model pModel, SQ_VectorData* pR2CumProgression);

   /**
   *  Retrieves the RMSEE matrix from a model.
   *  Root mean square error of the fit for observations in the workset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]  pRMSEE             A pointer to the RMSEE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRMSEE(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pRMSEE);
   /**
   *  Retrieves the RMSEcv matrix from a model.
   *  Cross validated root mean square error for observations in the workset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pRMSEcv           A pointer to the RMSEcv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRMSEcv(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pRMSEcv);

   /**
   *  Retrieves the RMSEcvProgression matrix from a model.
   *  Root mean square error showing the progression of RMSEcv for each added orthogonal component in the OPLS model, e.g. 1+0, 1+1, 1+2.
   *  The function fails if the model is not a 1+x+0 OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in] pColumnYIndices      A list of Y column indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]  pRMSEcvProgression A pointer to the RMSEcvProgression matrix.
   *                                  Number of rows in matrix = 1 + XSideOrthogonalComponents + XSidePCAComponents.
   *                                  Number of columns in matrix = 1.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRMSEcvProgression(SQ_Model pModel, SQ_IntVector* pColumnYIndices, SQ_VectorData* pRMSEcvProgression);

////////////////////////////////////

   /**
   *  Retrieves the MBEE matrix from a model.
   *  Root mean square error of the fit for observations in the workset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]  pMBEE              A pointer to the MBEE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMBEE(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pMBEE);

   /**
   *  Retrieves the MBEcv matrix from a model.
   *  Cross validated root mean square error for observations in the workset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component Indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[out]   pMBEcv            A pointer to the MBEcv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMBEcv(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pMBEcv);

/*********************************************************************************************************************
 * S VECTORS
 *********************************************************************************************************************/
   
   /**
   *  Retrieves the S matrix from a model.
   *  S is the projection of To on Y. S contains non-zero entries when the score 
   *  matrix To is not completely orthogonal to Y. The norm of this matrix is usually 
   *  very small but is used to enhance the predictions of Y.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pS                A pointer to the S matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetS(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pS);

   /**
   *  Retrieves the S2X matrix from a model.
   *  The variance of the X matrix. For component number xx,it is the residual variance of X after component xx.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pS2X              A pointer to the S2X matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetS2X(SQ_Model pModel, SQ_VectorData *pS2X);
   /**
   *  Retrieves the S2Y matrix from a model.
   *  The variance of the Y matrix. 
   *  For component number xx, it is the residual variance of Y after component xx.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pS2Y              A pointer to the S2Y matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetS2Y(SQ_Model pModel, SQ_VectorData *pS2Y);
   /**
   *  Retrieves the S2VX matrix from a model.
   *  Residual variance of the X variables, after the selected component, scaled as in the workset. 
   *  Component 0 is the variance of the X variables scaled as in the workset.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents       A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                  the first row will contain the data after 0 components.
   *  @param[out]   pS2VX             A pointer to the S2VX matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetS2VX(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pS2VX);
   /**
   *  Retrieves the S2VY matrix from a model.
   *  Residual variance of the Y variables, after the selected component, scaled as in the workset. 
   *  Component 0 is the variance of the Y variables scaled as in the workset.
   *  The function fails if the model is not a PLS model,
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents       A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                  the first row will contain the data after 0 components.
   *  @param[out]   pS2VY             A pointer to the S2VY matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetS2VY(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData *pS2VY);
   /**
   *  Retrieves the SDT matrix from a model.
   *  Standard deviation of the X scores (t).
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pSDT              A pointer to the SDT matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSDT(SQ_Model pModel, SQ_VectorData *pSDT);
   /**
   *  Retrieves the SDU matrix from a model.
   *  Standard deviation of the Y scores (u).
   *  The function fails if the model is not a PLS model
   *  or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pSDU              A pointer to the SDU matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSDU(SQ_Model pModel, SQ_VectorData *pSDU);
   /**
   *  Retrieves the SSX matrix from a model.
   *  Sum of squares of the X matrix. For component number xx, 
   *  it is the X residual Sum of Squares after component xx.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pSSX              A pointer to the SSX matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSSX(SQ_Model pModel, SQ_VectorData *pSSX);
   /**
   *  Retrieves the SSY matrix from a model.
   *  Sum of squares of the Y matrix. 
   *  For component number xx, it is the Y residual Sum of Squares after component xx.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pSSY              A pointer to the SSY matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of components in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSSY(SQ_Model pModel, SQ_VectorData *pSSY);
   
   /**
   *  Retrieves the SerrL matrix from a model.
   *  Lower limit of the Standard error of the predicted response Y for an observation used to fit the model.
   *  SerrL is always in original units, i.e. back transformed when the response Y was back transformed.
   *  The function fails if the model is not a PLS model, if the model doesn't have any components
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pSerrL            A pointer to the SerrL matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @see SQ_GetColumnYIndexByName
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSerrL(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_VectorData *pSerrL);
   /**
   *  Retrieves the SerrU matrix from a model.
   *  Upper limit of the Standard error of the predicted response Y for an observation used to fit the model.
   *  SerrU is always in original units, i.e. back transformed when the response Y was back transformed.
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pSerrU            A pointer to the SerrU matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSerrU(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_VectorData *pSerrU);

/*********************************************************************************************************************
 * T VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the T matrix from a model.
   *  The scores t, one vector for each model dimension, are new variables, 
   *  computed as linear combination of the X's. 
   *  They summarize X, to best approximate X (PC model), and both approximate X and predict Y (PLS model).
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]  pT                 A pointer to the T vector data.
   *                                  Number of rows in vector data = number of components chosen (length of pComponents).
   *                                  Number of columns in vector data = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetT(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pT);
   /**
   *  Retrieves the scores(T) correlation scaled from a model.
   *  T(corr) is used to create the Loadings Bi-plot in SIMCA.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pModel                        The model handle to use
   *  @param[in]  pComponents                   A list of component indices to use.
   *                                            1 for component 1 in the model, 2 for component 2 and so on.
   *                                            NULL if all predictive components in the model will be used.
   *  @param[in]  pComponentIsPredictiveVector  A bool vector with the same size as the pComponents vector. 
   *                                            Set to true for each component that represents a predictive component
   *                                            and to false for the components that represents an orthogonal component.
   *                                            If NULL, the components are assumed to be predictive.
   *  @param[out] pTCorr                        A pointer to the T(corr) vector.
   *                                            Number of entries = number of observations in the model.
   *  @return                                   Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTCorrelation(SQ_Model pModel, SQ_IntVector* pComponents, SQ_BoolVector* pComponentIsPredictiveVector, SQ_VectorData *pTCorr);
   /**
   *  Retrieves the Tcv matrix from a model.
   *  X score t for the selected model dimension, computed from the selected cross validation round.
   *  For PLS/OPLS/O2PLS tcv contains one value per observation and for PCA one value per observation and cross validation round.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pTcv              A pointer to the Tcv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds set in the project.
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTcv(SQ_Model pModel, int iComponent, SQ_VectorData *pTcv);
   /**
   *  Retrieves the TcvSE matrix from a model.
   *  Jack knife standard error of the scores t computed from the cross validations.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pTcvSE            A pointer to the TcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTcvSE(SQ_Model pModel, int iComponent, SQ_VectorData *pTcvSE);
   /**
   *  The degrees of freedom of TcvSE from a model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfTcvSEDF         The DoF TcvSE.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTcvSEDF(SQ_Model pModel, int iComponent, float *pfTcvSEDF);
   /**
   *  Retrieves the TCrit from a model.
   *  The critical limit for T for a certain component and probability level.
   *  The axis length on the T2 ellips.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   fLevel             The probability level, .95 means 95% probability. 
   *                                  If -1, the default level from the project is used.
   *  @param[out]   pfTCrit           A pointer to the TCrit result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTCrit(SQ_Model pModel, int iComponent, float fLevel, float *pfTCrit);
   
   /**
   *  Retrieves the ToCrit from a model.
   *  The critical limit for To for a certain orthogonal component and probability level.
   *  The function fails for non OPLS models.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         the orthogonal component to use
   *  @param[in]   fLevel             The probability level, .95 means 95% probability.
   *                                  If -1, the default level from the project is used.
   *  @param[out]   pfTCrit           A pointer to the TCrit result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetToCrit(SQ_Model pModel, int iComponent, float fLevel, float *pfTCrit);
   /**
   *  Retrieves the TMean from a model.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]  pfTMean            A pointer to the TMean result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTMean(SQ_Model pModel, int iComponent, float *pfTMean);
   /**
   *  Retrieves the TStandardDeviation and degrees of freedom from a model.
   *  Standard deviation of the X scores (t) for each component.
   *  The function fails if the model doesn't have any components.
   *  
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfDF              The degrees of freedom
   *  @param[out]   pfTStdev          A pointer to the T standard deviation result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTStandardDeviation(SQ_Model pModel, int iComponent, float *pfDF, float *pfTStdev);
   /**
   *  Retrieves the T2Crit from a model.
   *  The critical limit for T2 for a certain component and probability level.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   fLevel             The probability level, .95 means 95% probability. 
   *                                  If -1, the default level from the project is used.
   *  @param[out]   pfT2Crit          A pointer to the T2Crit result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetT2Crit(SQ_Model pModel, int iComponent, float fLevel, float *pfT2Crit);
   /**
   *  Retrieves the T2Range matrix from a model.
   *  Hotelling T2Range, is a combination of all the scores (t) 
   *  in the selected range of components.
   *  It is a measure of how far away an observation is from the center
   *  of a PC or PLS model hyperplane in the selected range of components.
   *  If the prediction handle comes from the observation level of a batch project, 
   *  the results will be T2RangeBCC.
   *  T2RangeBCC is a combination of all the scores in the selected range computed at every time point.
   *  It is a measure of how far a batch time point is from the average trajectory at the same time point.
   *  The function fails if the model doesn't have any components.
   *  T2Range for an OPLS model is only valid for component 1 to last (Predictive + X Side orthogonal).
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iCompFrom          The first component in the requested range.
   *  @param[in]   iCompTo            The last component in the requested range. 
   *                                  Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                  OPLS which always requires the last component.
   *  @param[out]   pT2Range          A pointer to the T2Range matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetT2Range(SQ_Model pModel, int iCompFrom, int iCompTo, SQ_VectorData *pT2Range);
   /**
   *  Retrieves the T2RangeCrit from a model.
   *  The critical limit for T2 for a certain component and probability level.
   *  The function fails if the model doesn't have any components.
   *  T2RangeCrit for an OPLS model is only valid for component 1 to last (Predictive + X Side orthogonal).
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponentFrom     Index of the first component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iComponentTo       Index of the second component to use
   *                                  Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                  OPLS which always requires the last component.
   *  @param[in]   fLevel             The probability level, .95 means 95% probability. 
   *                                  If -1, the default level from the project is used.
   *  @param[out]   pfT2RangeCrit          A pointer to the T2Crit result.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetT2RangeCrit(SQ_Model pModel, int iComponentFrom, int iComponentTo, float fLevel, float *pfT2RangeCrit);

   /**
   *  Retrieves the To matrix from a model.
   *  Scores that summarizes the X variation orthogonal to Y.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pTo               A pointer to the To matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pTo);

   /**
   *  Retrieves the Tocv matrix from a model.
   *  Cross validated scores that summarizes the X variation orthogonal to Y.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[out]   pTocv             A pointer to the Tocv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTocv(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pTocv);

   /**
   *  Retrieves the TocvSE matrix from a model.
   *  Standard error of the cross validated scores orthogonal to Y.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pTocvSE           A pointer to the TocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTocvSE(SQ_Model pModel, int iComponent, SQ_VectorData* pTocvSE);

   /**
   *  Retrieves the ToCorr matrix from a model.
   *  Scores scaled as correlation coefficients that summarizes the X variation orthogonal to Y.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]  pModel                        The model handle to use
   *  @param[in]  pComponents                   A list of component indices to use.
   *                                            1 for component 1 in the model, 2 for component 2 and so on.
   *                                            NULL if all predictive components in the model will be used.
   *  @param[in]  pComponentIsPredictiveVector  A bool vector with the same size as the pComponents vector. 
   *                                            Set to true for each component that represents a predictive component
   *                                            and to false for the components that represents an orthogonal component.
   *                                            If NULL, the components are assumed to be predictive.tocorr
   *  @param[out]  pToCorr                      A pointer to the ToCorr results.
   *                                            Number of rows in matrix = number of components chosen (length of pComponents).
   *                                            Number of columns in matrix = number of observations in the model.
   *  @return                                   Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetToCorr(SQ_Model pModel, SQ_IntVector* pComponents, SQ_BoolVector* pComponentIsPredictiveVector, SQ_VectorData* pToCorr);

/*********************************************************************************************************************
 * U VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the U matrix from a model.
   *  The u scores, one vector for each model dimension, are new variables summarizing Y
   *  so as to maximizes the correlation with the scores t.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pU                A pointer to the U matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetU(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pU);

   /**
   *  Retrieves the Uo matrix from a model.
   *  Scores that summarizes the Y variation orthogonal to X.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[out]   pUo               A pointer to the Uo matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetUo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_VectorData* pUo);

   /**
   *  Retrieves the Uocv matrix from a model.
   *  Orthogonal Y score uo of the Y-part of the OPLS model, computed from the selected cross validation rounds.
   *  The function fails if the model is not a OPLS model or if the model doesn't have any Y side orthogonal components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pYCompontentsList  A list of Y side orthogonal components to use.
   *                                  NULL if all Y components in the model should be used
   *  @param[out]   pUocv             A pointer to the Uocv matrix.
   *                                  Number of rows in matrix = number of predictive components.
   *                                  Number of columns in matrix = number of y-variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetUocv(SQ_Model pModel, SQ_IntVector* pYCompontentsList, SQ_VectorData* pUocv);

   /**
   *  Retrieves the Ucv matrix from a model.
   *  Y score u for the selected model dimension, computed from the selected cross validation rounds.
   *  The function fails if the model is not a OPLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pCompontentsList   A list of predictive components to use.
   *                                  NULL if all Y components in the model should be used
   *  @param[out]  pUcv               A pointer to the Ucv matrix.
   *                                  Number of rows in matrix = number of predictive components.
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetUcv(SQ_Model pModel, SQ_IntVector* pCompontentsList, SQ_VectorData* pUcv);

 /*********************************************************************************************************************
 * VIP VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the VIP matrix from a model.
   *  The influence on all the Y's (all responses) of every term (xk) in the model. 
   *  Term with VIP>1 have an above average influence on Y.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pVIP              A pointer to the VIP matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIP(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIP);
   /**
   *  Retrieves the VIPCI matrix from a model.
   *  The confidence interval for VIP.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pVIPCI              A pointer to the VIPCI matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPCI(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIPCI);
   /**
   *  Retrieves the VIPcv matrix from a model.
   *  The VIP, computed from all the cross validation rounds in the model.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pVIPcv            A pointer to the VIPcv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds in the model.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPcv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIPcv);
   /**
   *  Retrieves the VIPcvSE matrix from a model.
   *  The jack knife standard error of the VIP computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pVIPcvSE          A pointer to the VIPcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIPcvSE);
   /**
   *  Retrieves the VIPLag matrix from a model.
   *  VIP of a lagged variable as a function of the Lags.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]  pVIPLag            A pointer to the VIPLag matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPLag(SQ_Model pModel, int iComponent, int iColumnXIndex, SQ_VectorData* pVIPLag);

   /**
   * Get predictive VIP matrix from an OPLS model.
   * @param[in]   pModel The model handle to use
   *  @param[in]  bReconstruct   If the project is a wavelet spectral compressed project
   *                             and this parameter is true the returned matrix will be back transformed
   *                             to the original domain.
   * @param[out]  pVIP A pointer to the VIP matrix
   *                                   Number of rows in matrix = number of components in the model.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPPredictive(SQ_Model pModel, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIP);

   /**
   * Get orthogonal VIP matrix from an OPLS model.
   * @param[in]   pModel The model handle to use
   *  @param[in]  bReconstruct   If the project is a wavelet spectral compressed project
   *                             and this parameter is true the returned matrix will be back transformed
   *                             to the original domain.
   * @param[out]  pVIP A pointer to the VIP matrix
   *                                   Number of rows in matrix = number of components in the model.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPOrthogonal(SQ_Model pModel, SQ_ReconstructState bReconstruct, SQ_VectorData* pVIP);
   
   /**
   *  Retrieves the predictive VIPLag matrix from a model.
   *  VIP of a lagged variable as a function of the Lags.
   *  The function fails if the model is not a OPLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]  pVIPLag            A pointer to the VIPLag matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPPredictiveLag(SQ_Model pModel, int iColumnXIndex, SQ_VectorData* pVIPLag);

   /**
   *  Retrieves the orthogonal VIPLag matrix from a model.
   *  VIP of a lagged variable as a function of the Lags.
   *  The function fails if the model is not a OPLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]  pVIPLag            A pointer to the VIPLag matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVIPOrthogonalLag(SQ_Model pModel, int iColumnXIndex, SQ_VectorData* pVIPLag);

/*********************************************************************************************************************
 * W VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the W matrix from a model.
   *  The Weights that combine the X variables (1st dimension) 
   *  or their residuals (subsequent dimensions) to form the scores t. 
   *  These weights are selected to maximize the correlation between t and u. 
   *  Large w's indicate high correlation with u (Y).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is SQ_True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pW                A pointer to the W matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_GetW(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pW);
   /**
   *  Retrieves the Wcv matrix from a model.
   *  The w weights for a selected model dimension, computed from all the cross validation rounds.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is SQ_True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWcv              A pointer to the Wcv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWcv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWcv);
   /**
   *  Retrieves the WcvSE matrix from a model.
   *  The jack knife standard error of the weights w computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWcvSE            A pointer to the WcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWcvSE);
   /**
   *  Retrieves the degrees of freedom of WcvSE from a model.
   *  
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfWcvSEDF         The DoF of WcvSE.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWcvSEDF(SQ_Model pModel, int iComponent, float *pfWcvSEDF);
   /**
   *  Retrieves the WLag matrix from a model.
   *  Weights w of a lagged variable as a function of the Lags.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]   pWLag             A pointer to the WLag matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWLag(SQ_Model pModel, int iComponent, int iColumnXIndex, SQ_VectorData* pWLag);
   /**
   *  Retrieves the W* matrix from a model.
   *  The Weights that combine the original X's (not their residuals) to form the scores t. 
   *  In the first dimension w =w*. X variables with large w* are highly correlated with u and (Y).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWStar            A pointer to the WStar matrix.
   *                                  Number of rows in matrix = number of components chosen (length of pComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStar(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pWStar);
   /**
   *  Retrieves the W*C matrix from a model.
   *  Combined weights w* and c (see w* and c).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents       A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWStarC           A pointer to the WStarC matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarC(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pWStarC);
   /**
   *  Retrieves the WStarCcvSE matrix from a model.
   *  The jack knife standard error of the w* and c weights computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWStarCcvSE       A pointer to the WStarCcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarCcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWStarCcvSE);
   /**
   *  Retrieves the WStarcv matrix from a model.
   *  The w* weights for a selected model dimension, computed from all the cross validation rounds.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWStarcv          A pointer to the WStarcv matrix.
   *                                  Number of rows in matrix = number of cross-validation rounds.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarcv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWStarcv);
   /**
   *  Retrieves the WStarcvSE matrix from a model.
   *  The jack knife standard error of the weights w* computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWStarcvSE        A pointer to the WStarcvSE matrix.
   *                                  Number of rows in matrix = 1
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarcvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWStarcvSE);
   /**
   *  Retrieves the Degrees of freedom for WStarcvSE from a model.
   *  
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         Index of the component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[out]   pfWStarcvSEDF     the DoF of WStarcvSE.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarcvSEDF(SQ_Model pModel, int iComponent, float *pfWStarcvSEDF);
   /**
   *  Retrieves the WStarLag matrix from a model.
   *  Weights w* of a lagged variable as a function of the Lags.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iColumnXIndex      The index of X column to use
   *  @param[out]   pWStarLag         A pointer to the WStarLag matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of lagged variables in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWStarLag(SQ_Model pModel, int iComponent, int iColumnXIndex, SQ_VectorData* pWStarLag);

   /**
   *  Retrieves the Wo matrix from a model.
   *  Weights that combine the X variables (first dimension) or the X residuals 
   *  (subsequent dimensions) to form the scores To. These weights are selected 
   *  so as to minimize the correlation between To and U, thereby indirectly between To and Y.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pComponents        A list of component indices to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *                                  NULL if all components should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWo               A pointer to the Wo matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWo(SQ_Model pModel, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData* pWo);

   /**
   *  Retrieves the Wocv matrix from a model.
   *  Orthogonal weights Wo from the X-part of the model, for a selected model dimension,
   *  computed from the selected cross validation round.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWocv             A pointer to the Wocv matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWocv(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWocv);

   /**
   *  Retrieves the WocvSE matrix from a model.
   *  Standard error of the cross validated weights wo.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *  to get the confidence interval, see GetcvSEPercentile(...
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use.
   *                                  1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pWocvSE           A pointer to the WocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetWocvSE(SQ_Model pModel, int iComponent, SQ_ReconstructState bReconstruct, SQ_VectorData* pWocvSE);

/*********************************************************************************************************************
 * X VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the XObs matrix from a model.
   *  The X part of an observation in the workset, in original units.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bUnscaled          If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                  If False, the returned x-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the x-values will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case x-values will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the x-values in the unscaled untransformed 
   *                                  metric of the workset. If False the returned x-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pObservations      A list of observation indices that exist in the model, 
   *                                  NULL if all observations that exist in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pXObs             A pointer to the pXObs matrix.
   *                                  Number of rows in matrix = number of observations chosen (length of pnObservationList).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXObs(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations, SQ_ReconstructState bReconstruct,SQ_VectorData* pXObs);
   /**
   *  Retrieves the XObsPred matrix from a model.
   *  A reconstructed observation as X=TP? from the training set
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The number of the component in the model we want the results from
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bUnscaled          If true, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                  If false, the returned x-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is false, the x-values will still 
   *                                  be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                  In this case x-values will always be unscaled.
   *  @param[in]   bBackTransformed   If true, the function will return the x-values in the unscaled untransformed 
   *                                  metric of the workset. If false the returned x-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is true, the returned
   *                                  x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pObservations      A list of observation Indices that exist in the model, 
   *                                  NULL if all observations that exist in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is true the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pXObsPred         A pointer to the pXObsPred matrix.
   *                                  Number of rows in matrix = number of observations chosen (length of pnObservationList).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXObsPred(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations, SQ_ReconstructState bReconstruct,SQ_VectorData* pXObsPred);
   /**
   *  Retrieves the XObsRes matrix from a model.
   *  The residuals of an X observation in the workset, in original units.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The number of the component in the model we want the results from
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   bUnscaled          If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                  If False, the returned residuals will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case residuals will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the residuals in the unscaled untransformed 
   *                                  metric of the workset. If False the returned residuals will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pObservations      A list of observation indices that exist in the model, 
   *                                  NULL if all observations that exist in the model should be used.
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pXObsRes          A pointer to the pXObsRes matrix.
   *                                  Number of rows in matrix = number of observations chosen (length of pnObservationList).
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXObsRes(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations, SQ_ReconstructState bReconstruct,SQ_VectorData* pXObsRes);
   /**
   *  Retrieves the XOffsets matrix from a model. (Xavg in SIMCA)
   *  The center of X variables, as specified, in original units.
   *  When the variable is UV scaled the center is the variable average. 
   *  If the variable is transformed, the center is in the transformed metric.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]  pXOffsets          A pointer to the XOffsets matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXOffsets(SQ_Model pModel, SQ_ReconstructState bReconstruct, SQ_VectorData* pXOffsets);
   /**
   *  Retrieves the XVar matrix from a model.
   *  X variable from the workset, in original units.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bUnscaled          If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                  If False, the returned x-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the x-values will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case x-values will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the x-values in the unscaled untransformed 
   *                                  metric of the workset. If False the returned x-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pColumnXIndices    A list of X column indices to use.
   *                                  NULL if all x columns in the model should be used
   *  @param[out]   pXVar             A pointer to the XVar matrix.
   *                                  Number of rows in matrix = number of x-variables chosen (length of pnColumnXIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVar(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnXIndices,SQ_VectorData* pXVar);
   /**
   *  Retrieves the XVarRes matrix from a model.
   *  X Variable residuals unscaled, i.e. original units, for observations in the workset.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component and zero are the only valid ones.
   *  @param[in]   pColumnXIndices    A list of X column indices to use.
   *                                  NULL if all x columns in the model should be used
   *  @param[in]   bUnscaled          If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                  If False, the returned residuals will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case residuals will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the residuals in the unscaled untransformed 
   *                                  metric of the workset. If False the returned residuals will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   bStandardized      If True, the function will use the standardized residuals 
   *                                  (the unscaled residuals divided by their standard deviation).
   *  @param[out]   pXVarRes          A pointer to the XVarRes matrix.
   *                                  Number of rows in matrix = number of x-variables chosen (length of pnColumnXIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarRes(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnXIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized,SQ_VectorData* pXVarRes);

   /**
   *  Retrieves the XVarPred matrix from a model.
   *  For PLS and PCA models, X variables, from the training set, reconstructed as X=TP'.
   *  For OPLS/O2PLS models it is the X-values predicted from from the given Y-values.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnXIndices    A list of X column indices to use.
   *                                  NULL if all x columns in the model should be used
   *  @param[in]   bUnscaled          If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                  If False, the returned residuals will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case residuals will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the residuals in the unscaled untransformed 
   *                                  metric of the workset. If False the returned residuals will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out]   pXVarPred         A pointer to the XVarPred matrix.
   *                                  Number of rows in matrix = number of x-variables chosen (length of pnColumnXIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarPred(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnXIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData* pXVarPred);

   /**
   *  Retrieves the XVarResYRelated matrix from a model.
   *  The estimated pure profiles of the underlying constituents in X under the
   *  assumption of additive Y-variables.
   *  Estimation includes a linear transformation of the Coefficient matrix, Bp(BpTBp)-1, where Bp
   *  is the Coefficient matrix using only the predictive components to compute the Coefficient
   *  matrix (i.e., the components orthogonal to Y are not included in the computation of Bp).
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bUnscaled          If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                  If False, the returned x-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the x-values will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case x-values will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the x-values in the unscaled untransformed 
   *                                  metric of the workset. If False the returned x-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   bStandardized      If True, the function will return the standardized residuals 
   *                                  (the unscaled residuals divided by their standard deviation).
   *  @param[in]   pColumnXIndices    A list of X column Indices to use.
   *                                  NULL if all X columns in the model should be used
   *  @param[out]   pXVarResYRelated  A pointer to the XVarResYRelated matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarResYRelated(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized, SQ_IntVector* pColumnXIndices, SQ_VectorData* pXVarResYRelated);

   /**
   *  Retrieves the XVarResO2PLS matrix from a model.
   *  The function is only available for O2PLS models that have X-Side orthogonal components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iPredComponent        The predictive component to use.
   *                                   1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iXSideOrthoComponent  The orthogonal component to use.
   *                                   1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   iYSideOrthoComponent  The unrelated to X component to use.
   *                                   1 for component 1 in the model, 2 for component 2 and so on.
   *  @param[in]   bUnscaled             If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                   If False, the returned x-values will be in the scaled and centered metric of 
   *                                   the workset. Note that if bBackTransformed is False, the x-values will still 
   *                                   be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                   In this case x-values will always be unscaled.
   *  @param[in]   bBackTransformed      If True, the function will return the x-values in the unscaled untransformed 
   *                                   metric of the workset. If False the returned x-values will be transformed
   *                                   in the same way as the workset. Note that if this variable is True, the returned
   *                                   x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   bStandardized      If True, the function will return the standardized residuals 
   *                                  (the unscaled residuals divided by their standard deviation).
   *  @param[in]   pColumnXIndices       A list of X column Indices to use.
   *                                   NULL if all X columns in the model should be used
   *  @param[out]   pXVarResO2PLS        A pointer to the XVarResO2PLS matrix.
   *                                   Number of rows in matrix = number of components chosen (length of piComponents).
   *                                   Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarResO2PLS(SQ_Model pModel, int iPredComponent, int iXSideOrthoComponent, int iYSideOrthoComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized, SQ_IntVector* pColumnXIndices, SQ_VectorData* pXVarResO2PLS);

   /**
   *  Retrieves the XWeights matrix from a model.
   *  Scaling weights of the X Variables.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bReconstruct       If the project is a wavelet spectral compressed project
   *                                  and this parameter is True the returned matrix will be back transformed 
   *                                  to the original domain.
   *  @param[out]   pXWeights         A pointer to the XWeights matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of x-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXWeights(SQ_Model pModel, SQ_ReconstructState bReconstruct,SQ_VectorData* pXWeights);

/*********************************************************************************************************************
 * Y VECTORS
 *********************************************************************************************************************/
   /**
   *  Retrieves the YObs matrix from a model.
   *  The Y part of an observation in the workset, in original units.
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   bUnscaled          If True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                  If False, the returned y-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the y-values will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case y-values will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the y-values in the unscaled untransformed 
   *                                  metric of the workset. If False the returned y-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pObservations      A list of observation indices that exist in the model, 
   *                                  NULL if all observations that exist in the model should be used.
   *  @param[out]   pYObs             A pointer to the pYObs matrix.
   *                                  Number of rows in matrix = number of observations chosen (length of pnObservationList).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */

   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYObs(SQ_Model pModel, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations,SQ_VectorData* pYObs);
   /**
   *  Retrieves the YObsRes matrix from a model.
   *  The residuals of a Y observation in the workset, in original units.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The number of the component in the model we want the results from
   *  @param[in]   bUnscaled          If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                  If False, the returned residuals will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case residuals will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the residuals in the unscaled untransformed 
   *                                  metric of the workset. If False the returned residuals will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pObservations      A list of observation indices that exist in the model, 
   *                                  NULL if all observations that exist in the model should be used.
   *  @param[out]   pYObsRes          A pointer to the pYObsRes matrix.
   *                                  Number of rows in matrix = number of observations chosen (length of pnObservationList).
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYObsRes(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations,SQ_VectorData* pYObsRes);
   /**
   *  Retrieves the YOffsets matrix from a model.
   *  The center of Y variables, as specified, in original units.
   *  When the variable is UV scaled the center is the variable average.
   *  If the variable is transformed, the center is in the transformed metric.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pYOffsets         A pointer to the YOffsets matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYOffsets(SQ_Model pModel,SQ_VectorData* pYOffsets);
   /**
   *  Retrieves the YPred matrix from a model.
   *  Predicted values of Y variables for observations in the workset, used to fit the model.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component or zero are the only valid ones.
   *  @param[in]   bUnscaled          If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                  If false, the returned y-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                  be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                  In this case y-values will always be unscaled.
   *  @param[in]   bBackTransformed   If true, the function will return the y-values in the unscaled untransformed 
   *                                  metric of the workset. If false the returned y-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is true, the returned
   *                                  y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pYPred            A pointer to the YPred matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   *
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPred(SQ_Model pModel, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, SQ_VectorData* pYPred);
   /**
   *  Retrieves the YPredCV matrix from a model.
   *  Prediction for an observation in the workset computed from the model with that observation removed.
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pYPredCV          A pointer to the YPredCV matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredCV(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices,SQ_VectorData* pYPredCV);
   /**
   *  Retrieves the YPredCVErr matrix from a model.
   *  Prediction error for an observation in the workset computed from the model with that observation removed.
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pYPredCVErr       A pointer to the YPredCVErr matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredCVErr(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices,SQ_VectorData* pYPredCVErr);
   /**
   *  Retrieves the YPredCVErrSE matrix from a model.
   *  Jack knife standard error of the fitted Y computed from the cross validation rounds
   *  The function fails if the model is not a PLS model, if the model doesn't have any components.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pYPredCVErrSE     A pointer to the YPredCVErrSE matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of components in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredCVErrSE(SQ_Model pModel, SQ_IntVector* pColumnYIndices,SQ_VectorData* pYPredCVErrSE);
   /**
   *  Retrieves the YVar matrix from a model.
   *  Y variable from the workset, in original units.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pColumnYIndices    A list of Y column indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bUnscaled          If True, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                  If False, the returned y-values will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the y-values will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case y-values will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the y-values in the unscaled untransformed 
   *                                  metric of the workset. If False the returned y-values will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out]   pYVar             A pointer to the YVar matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYVar(SQ_Model pModel, SQ_IntVector* pColumnYIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed,SQ_VectorData* pYVar);
   /**
   *  Retrieves the YVarRes matrix from a model.
   *  Y Variable residuals unscaled, i.e. original units, for observations in the workset.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   iComponent         The component to use
   *                                  For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]   pColumnYIndices    A list of Y column indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[in]   bUnscaled          If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                  If False, the returned residuals will be in the scaled and centered metric of 
   *                                  the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                  be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                  In this case residuals will always be unscaled.
   *  @param[in]   bBackTransformed   If True, the function will return the residuals in the unscaled untransformed 
   *                                  metric of the workset. If False the returned residuals will be transformed
   *                                  in the same way as the workset. Note that if this variable is True, the returned
   *                                  residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]   bStandardized      If True, the function will return the standardized residuals 
   *                                  (the unscaled residuals divided by their standard deviation).
   *  @param[out]   pYVarRes          A pointer to the YVarRes matrix.
   *                                  Number of rows in matrix = number of y-variables chosen (length of pnColumnYIndices).
   *                                  Number of columns in matrix = number of observations in the model.
   *
   *  @return                         Returns SQ_E_OK if success or an error code
   *  @see SQ_GetColumnYIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYVarRes(SQ_Model pModel, int iComponent, SQ_IntVector* pColumnYIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized,SQ_VectorData* pYVarRes);
   /**
   *  Retrieves the YWeights matrix from a model.
   *  Scaling weights of the Y Variables.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   pYWeights         A pointer to the YWeights matrix.
   *                                  Number of rows in matrix = 1.
   *                                  Number of columns in matrix = number of y-variables in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYWeights(SQ_Model pModel,SQ_VectorData* pYWeights);

   /**
   *  Retrieves the YRelatedProfile matrix from a model.
   *  X residuals where the systematic variation orthogonal to Y has been removed.
   *  The function fails if the model is not an OPLS/O2PLS model.
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[in]   pColumnYIndices    A list of Y column Indices to use.
   *                                  NULL if all y columns in the model should be used
   *  @param[out]   pYRelated         A pointer to the PocvSE matrix.
   *                                  Number of rows in matrix = number of components chosen (length of piComponents).
   *                                  Number of columns in matrix = number of observations in the model.
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYRelatedProfile(SQ_Model pModel, SQ_IntVector* pColumnYIndices, SQ_VectorData* pYRelated);

   /**
   *  Retrieves the model alarm limits (used in SIMCA-online) formatted as a JSON string
   *
   *  @param[in]   pModel             The model handle to use
   *  @param[out]   szJsonLimits      Json formatted string containing the alarm and limits for this model. 
   *                                  Note that returned value will be of maximum iBufferLength.
   *                                  Available types are : var, t, DModX, DModX+, PModX, PModX+ and T2Range. For each type of vector there are 
   *                                  target, lolo, lo, hi and hihi limits. Each alarm type can have triggers to raise an alarm.
   *                                  Example 1 : A score vector alarm with limits. Extra argument is which component the alarm is for.
   *                                  "[{\"type\" : \"DModX\", \"comp\" : 1, \"limits\" : { \"lolo\" : -4, \"lo\" : -3, \"hi\" : 3, \"hihi\" : 4 }]"
   *                                  Example 2 : A variable vector alarm with limits. Extra argument is 'name' to specify
   *                                  the variable name to set the alarm for.
   *                                  "[{\"type\" : \"var\", \"name\" : \"Agitation speed\", \"limits\" : { \"lolo\" : -100, \"target\" : 100, \"hihi\" : 200 }]"
   *                                  Example 3 : A DModX vector alarm.
   *                                  "[{\"type\" : \"DModX\", \"limits\" : { \"hihi\" : 20 }]"
   *                                  Example 4 : A T2Range vector alarm. Extra argument \"comp\" is used for setting the component range. The \"comp\"
   *                                  argument is not used for OPLS models.
   *                                  "[{\"type\" : \"T2Range\", \"comp\" : { \"from\" : 1, \"to\" : 4 }, \"limits'\" : { \"hihi\" : 20 }]"
   *                                  Triggers are represented by a "triggerLogic" member to the alarm.
   *                                  "{ \"numObservations' : 3, \"missing\" : True, \"numObsInWindow\" : 8, \"sizeWindow\" : 100 }"
   *                                  Available members are 'numObservations' which is how many consecutive observations
   *                                  that can be outside the limits before triggering alarm. 'missing' is used when a missing observation should trigger alarms.
   *                                  "numObsInWindow" and "sizeWindow" fields are used for how many observations that can be outside limits withing a window before triggering
   *                                  an alarm. This is a moving window across all observations. The settings "numObsInWindow" : 8 and "sizeWindow" : 100
   *                                  means that an alarm will be triggered if there are more than 8 observations in a window of 100 observations.
   *  @param[in]    iBufferLength     The maximum length of szJsonLimits. 
   *  @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelAlarmLimits(SQ_Model pModel, char* szJsonLimits, int iBufferLength);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQModel_H_ */
