/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQMODELSTATISTICS_H_
#define _SQMODELSTATISTICS_H_

#include "SQDef.h"

/** @file SQModelStatistics.h

   This file list the SQModelStatistics object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQIntVector.h"
#include "SQFloatVector.h"

   /**
      A a handle to data with model statistics information.
      IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_ModelStatistics 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_ModelStatistics;

   /**
   *  Removes the model statistics, this function must be called for every 
   *  statistics object that has been received from SQ_GetStatistics.
   *  @see tagSQ_ModelStatistics
   *
   *  @param[in]   pStatistics   The model statistics object to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearModelStatistics(SQ_ModelStatistics* pStatistics);

   /**
   *  Retrieves the number of observations from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pObservations  Number of observations for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetObservationsFromStatistics(SQ_ModelStatistics pStatistics, SQ_IntVector* pObservations);

   /**
   *  Retrieves the number of missing values in percent from model statistics
   *
   *  @param[in]  pStatistics   The model statistics object to get information from.
   *  @param[out] pMisVals       Number of missing values in percent for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMissingFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMisVals);

   /**
   *  Retrieves the minimum values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pMin           Minimum values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMinFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMin);

   /**
   *  Retrieves the maximum values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pMax           Maximum values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMaxFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMax);

   /**
   *  Retrieves the minimum divided by maximum values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pMinMax        Minimum divided by maximum values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMinMaxFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMinMax);

   /**
   *  Retrieves the mean values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pMean          Mean values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMeanFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMean);

   /**
   *  Retrieves the median values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pMedian        Median values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMedianFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pMedian);

   /**
   *  Retrieves the standard deviation values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pStdDev        Standard deviation values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStdDevFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pStdDev);

   /**
   *  Retrieves the standard deviation divided by mean values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pStdDevMean    Standard deviation divided by mean values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStdDevMeanFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pStdDevMean);

   /**
   *  Retrieves the skewness values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pSkewness      Skewness values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSkewnessFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pSkewness);

   /**
   *  Retrieves the skewness test values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pSkewnessTest  Skewness test values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSkewnessTestFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pSkewnessTest);

   /**
   *  Retrieves the kurtosis values from model statistics
   *
   *  @param[in]   pStatistics   The model statistics object to get information from.
   *  @param[out] pKurtosis      Kurtosis values for each variable
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetKurtosisFromStatistics(SQ_ModelStatistics pStatistics, SQ_FloatVector* pKurtosis);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQMODELSTATISTICS_H_ */
