/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQPREDICTION_H_
#define _SQPREDICTION_H_

#include "SQDef.h"

/** @file SQPrediction.h

   This file list the SQ_Prediction object used to get data from a prediction.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"
#include "SQIntVector.h"
#include "SQVectorData.h"

   /**
   The handle used to identify a prediction object.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_Prediction 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Prediction;

   /**
   *  Removes the allocated memory for the Prediction object.
   *  This function must be called for every Prediction object that is created,
   *  if not a memory leak will occur.
   *
   *  @param[in]   pPrediction   The Prediction object to remove.
   *
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearPrediction(SQ_Prediction* pPrediction);

 /*********************************************************************************************************************
 * Contribution PS VECTORS
 *********************************************************************************************************************/

   /**
   * Get the score single weight contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iObs1Ix           Index in the observation matrix (the predictionset)
   *                                for the reference observation (from observation).
   *                                0 if the average is to be used.
   *  @param[in]  iObs2Ix           Index in the observation matrix (the predictionset) of the observation 
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *                                For an OPLS model with weight CoeffCS or VIP the component must be the last predictive.
   *  @param[in]  iYVar             The index of the Y variable to use if eWeightType is CoeffCS or CoeffCSRaw. 
   *                                If eWeightType is something else, this parameter is ignored. 
   *                                @see GetColumnYIndexByName
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the Scores single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScorePSSingleWeight(SQ_Prediction pPrediction, int iObs1Ix, int iObs2Ix, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);
   /**
   * Get the score single weight group contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pObs1Ix           A list of indices in the observation matrix (the predictionset)
   *                                for the reference observations (from observation).
   *                                NULL if the average is to be used.
   *  @param[in]  pObs2Ix           A list of indices in the observation matrix (the predictionset) of the observations 
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, Raw, RX or P.
   *                                If the model is a PLS model this parameter can be any weight defined in SQ_WeightType.
   *  @param[in]  iComponent        The component of the weight. Ignored if eWeightType=Normalized or Raw.
   *                                For an OPLS model with weight CoeffCS or VIP the component must be the last predictive.
   *  @param[in]  iYVar             The index of the Y variable to use if eWeightType is CoeffCS or CoeffCSRaw. 
   *                                If eWeightType is something else, this parameter is ignored. 
   *                                @see GetColumnYIndexByName
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the Scores single weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScorePSSingleWeightGroup(SQ_Prediction pPrediction, SQ_IntVector* pObs1Ix, SQ_IntVector* pObs2Ix, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the score multi weight contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iObs1Ix           Index in the observation matrix (the predictionset). 
   *                                for the reference observation (from observation).
   *                                0 if the average is to be used.
   *  @param[in]  iObs2Ix           Index in the observation matrix (the predictionset) of the observation 
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums.
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                for weight Po the component argument is the orthogonal component number and the model must be OPLS
   *                                for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the Scores multi weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScorePSMultiWeight(SQ_Prediction pPrediction, int iObs1Ix, int iObs2Ix, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the score multi weight contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pObs1Ix           A list of indices in the observation matrix (the predictionset).
   *                                for the reference observations (from observation).
   *                                NULL if the average is to be used.
   *  @param[in]  pObs2Ix           A list of indices in the observation matrix (the predictionset) of the observations 
   *                                for which the contributions are to be calculated (to observation).
   *  @param[in]   pWeightType      An int vector containing SQ_WeightType enums
   *                                If the model is a PCA model this parameter must be P.
   *                                If the model is a PLS model this parameter must be P or WStar.
   *                                If the model is a OPLS model this parameter must be P or PO.
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                for OPLS models and PRange weight the component argument is ignored, all components are used,
   *                                for weight Po the component argument is the orthogonal component number and the model must be OPLS
   *                                for P_Po, supply 2 components, one predictive component and one orthogonal component.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the Scores multi weight contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsScorePSMultiWeightGroup(SQ_Prediction pPrediction, SQ_IntVector* pObs1Ix, SQ_IntVector* pObs2Ix, SQ_IntVector* pWeightType, SQ_IntVector* pComponents, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);
   
   /**
   * Get the DModX contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iObsIx            Index in the observation matrix (the predictionset). 
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight.
   *                                For an OPLS model the component must be the last predictive.
   *  @param[in]  iYVar             The index of the Y variable to use if eWeightType is CoeffCS. 
   *                                If eWeightType is something else, this parameter is ignored. 
   *                                @see GetColumnYIndexByName
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the DModXPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModXPS(SQ_Prediction pPrediction, int iObsIx, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   * Get the DModX contributions from the predicted results.
   * Contribution is used to understand how an observation differs from the others in an X score(t) or in DModX.
   * See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pObsIx            A list of indices in the observation matrix (the predictionset). 
   *  @param[in]  eWeightType       The type of weight.
   *                                If the model is a PCA model this parameter must be Normalized, RX.
   *                                If the model is a PLS model this parameter must be Normalized, RX, CoeffCS or VIP.
   *  @param[in]  iComponent        The component of the weight.
   *                                For an OPLS model the component must be the last predictive.
   *  @param[in]  iYVar             The index of the Y variable to use if eWeightType is CoeffCS. 
   *                                If eWeightType is something else, this parameter is ignored. 
   *                                @see GetColumnYIndexByName
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the DModXPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */  
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModXPSGroup(SQ_Prediction pPrediction, SQ_IntVector* pObsIx, SQ_WeightType eWeightType, int iComponent, int iYVar, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);
   
   /**
   *  Get the DModY contributions from the predicted results.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components
   *  or if the model isn't a PLS model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iObsIx            Index in the observation matrix (the predictionset).
   *  @param[in]  eWeightType       The type of weight.
   *                                This parameter must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *                                For an OPLS model the component must be the last predictive.
   *  @param[out] pVectorData       A pointer to the DModYPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of Y-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModYPS(SQ_Prediction pPrediction, int iObsIx, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pVectorData);
   /**
   *  Get the DModY group contributions from the predicted results.
   *  Contribution is used to understand how an observation differs from the others in DModY.
   *  See the document "SIMCA-Q Interface Description.pdf" for a more detailed description on contributions.
   *  The function fails if the model doesn't have any components
   *  or if the model isn't a PLS model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pObsIx            A list of indices in the observation matrix (the predictionset).
   *  @param[in]  eWeightType       The type of weight.
   *                                This parameter must be Normalized or RY.
   *  @param[in]  iComponent        The component of the weight.
   *                                For an OPLS model the component must be the last predictive.
   *  @param[out] pVectorData       A pointer to the DModYPS contribution results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of Y-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetContributionsDModYPSGroup(SQ_Prediction pPrediction, SQ_IntVector* pObsIx, SQ_WeightType eWeightType, int iComponent, SQ_VectorData *pVectorData);

/*********************************************************************************************************************
 * DModPS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the DModXPS from the predicted data.
   *  Distance to the model in the X space (row residual SD), after xx components (the selected dimension),
   *  for new observations in the predictionset. 
   *  If you select component 0, it is the Stdev of the observations as scaled in the workset.
   *
   *  @param[in]  pPrediction             The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents             A list of component indices to use.
   *                                      1 for component 1 in the model, 2 for component 2 and so on.
   *                                      NULL if all components in the model should be used.
   *                                      If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                      the first row will contain the data after 0 components.
   *                                      For an OPLS model, the last predictive component and zero are the only valid ones.
   *  @param[in]  bNormalized             If true, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If false, they will be in absolute values.
   *  @param[in]  bModelingPowerWeighted  If true, the function will weight the residuals by the modeling power of the variables.
   *  @param[out] pVectorData             A pointer to the DModXPS results.
   *                                      Number of rows in matrix = number of components chosen (length of pComponents).
   *                                      Number of columns in matrix = number of observations in the prediction.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDModXPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData *pVectorData);
      /**
   *  Retrieves the DModXPS+ matrix from the predicted data.
   *  Combination of DModXPS plus Hotelling T2 when the latter is outside the critical limit.
   *  For observations in the predictionset. 
   *  If you select component 0, it is the Stdev of the observations as scaled in the workset.
   *
   *  @param[in]  pPrediction             The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents             A list of component Indices to use.
   *                                      1 for component 1 in the model, 2 for component 2 and so on.
   *                                      NULL if all components in the model should be used.
   *                                      If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                      the first row will contain the data after 0 components.
   *                                      For an OPLS model, the last predictive component and zero are the only valid ones.
   *  @param[in]  bModelingPowerWeighted  If true, the function will weight the residuals by the modeling power of the variables.
   *  @param[in]  bNormalized             If True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                      If False, they will be in absolute values.
   *  @param[out] pVectorData             A pointer to the DModXPS+ results.
   *                                      Number of rows in matrix = number of components chosen (length of pnComponentList).
   *                                      Number of columns in matrix = number of observations that was sent to Predict()/BatchPredict().
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDModXPSCombined(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_NormalizedState bNormalized, SQ_ModelingPowerWeightedState bModelingPowerWeighted, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the DModYPS from the predicted data.
   *  Distance to the model in the Y space (row residual SD), after xx components (the selected dimension),
   *  for new observations in the predictionset. 
   *  If you select component 0, it is the Stdev of the observations as scaled in the workset.
   *
   *  @param[in]  pPrediction          The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents          A list of component Indices to use.
   *                                   1 for component 1 in the model, 2 for component 2 and so on.
   *                                   NULL if all components in the model should be used.
   *                                   If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                   the first row will contain the data after 0 components.
   *                                   For an OPLS model, the last predictive component and zero are the only valid ones.
   *  @param[in]  bNormalized          If True, the results will be in units of standard deviation of the pooled RSD of the model
   *                                   If False, they will be in absolute values.
   *  @param[out] pVectorData          A pointer to the DModYPS results.
   *                                   Number of rows in matrix = number of components chosen (length of pComponents).
   *                                   Number of columns in matrix = number of observations in the prediction.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDModYPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_NormalizedState bNormalized, SQ_VectorData *pVectorData);

/*********************************************************************************************************************
 * PModPS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the PModXPS from the predicted data.
   *  Probability of belonging to the model in the X space for observations in the prediction set. 
   *  Component 0 corresponds to a point model (center of coordinate system). 
   *  Observations with PModX less than 5% are considered non members.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                the first row will contain the data after 0 components.
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[out] pVectorData       A pointer to the PModXPS results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPModXPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the PModXPS+ from the predicted data.
   *  Combination of PModXPS plus Hotelling T2 when the latter is outside the critical limit. 
   *  Component 0 corresponds to a point model (center of coordinate system). 
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                the first row will contain the data after 0 components.
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[out] pVectorData       A pointer to the PModXPS+ results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPModXPSCombined(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the PModYPS from the predicted data.
   *  Probability of belonging to the model in the Y space for observations in the prediction set. 
   *  Component 0 corresponds to a point model (center of coordinate system). 
   *  Observations with PModY less than 5% are considered non members.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *                                If NULL is used, the returned matrix will contain the number of components + 1 rows, i.e.
   *                                the first row will contain the data after 0 components.
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[out] pVectorData       A pointer to the PModYPS results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of observations that was in the prediction set.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPModYPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * TPS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the TPS from the predicted data.
   *  The predicted scores, (for new observations), one vector for each model dimension. 
   *  They are new variables, computed from the model. 
   *  They summarize X, to best approximate X (PC model), and both approximate X and predict Y (PLS model).
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components in the model should be used.
   *  @param[out] pVectorData       A pointer to the TPS results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the ToPS from a prediction.
   *  Scores that summarizes the X variation orthogonal to Y for the prediction set.
   *  The function fails if the model is not an O2PLS or OPLS model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component indices to use.
   *                                1 for component 1 in the model, 2 for component 2 and so on.
   *                                NULL if all components should be used.
   *  @param[out] pVectorData       A pointer to the ToPS results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetToPS(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the TPScv from the predicted data.
   *  The predicted scores tPS computed from all the Cross Validation rounds.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *  @param[out] pVectorData       A pointer to the TPScv results.
   *                                Number of rows in matrix = number of cross-validation rounds.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTPScv(SQ_Prediction pPrediction, int iComponent, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the TPScvSE from the predicted data.
   *  Jack knife standard error of the predicted scores tPS computed from the cross validations.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *  @param[out] pVectorData       A pointer to the TPScvSE results.
   *                                Number of rows in matrix = 1
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetTPScvSE(SQ_Prediction pPrediction, int iComponent, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the T2RangePS matrix from the predicted data.
   *  Hotelling T2RangePS is a combination of all the scores (tPS) 
   *  in the selected range of components.
   *  It is a measure of how far away an observation is from the center 
   *  of a PC or PLS model hyperplane in the selected range of components.
   *  If the prediction handle comes from the observation level of a batch project, 
   *  the results will be T2RangeBCCPS.
   *  T2RangeBCCPS is a combination of all the scores in the selected range computed at every time point.
   *  It is a measure of how far a batch time point is from the average trajectory at the same time point.
   *  The function fails if the model doesn't have any components.
   *  T2RangePS for an OPLS model is only valid for component 1 to last (Predictive + X Side orthogonal).
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iCompFrom         The first component in the requested range.
   *  @param[in]  iCompTo           The last component in the requested range.
   *                                Use -1 if you want it to be the last component in model. This is the preferred way for
   *                                OPLS which always requires the last component.
   *  @param[out] pVectorData       A pointer to the T2RangePS results.
   *                                Number of rows in matrix = 1.
   *                                Number of columns in matrix = number of observations that was sent to Predict()/BatchPredict().
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetT2RangePS(SQ_Prediction pPrediction, int iCompFrom, int iCompTo, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * Classification PS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the misclassification table.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  bAllCombinations  A boolean to retrieve all different combinations between classes
   *  @param[out] pVectorData       A pointer to the misclassification results.
   *                                 
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMisClassification(SQ_Prediction pPrediction, SQ_Bool bAllCombinations, SQ_VectorData *pVectorData);

   /*********************************************************************************************************************
   * ROC calculations
   *********************************************************************************************************************/

   /**
   * Returns the receiver operating characteristic for the current prediction set (see wikipedia).
   * @param[in] pPrediction          The handle to the predictions, is retrieved from GetPrediction()
   *                                 The model must be a PLS-DA model with two classes or a class model.
   * @param[in] iYVar                If the model is a PLS-DA model this is the index of the Y-variable (class)
   *                                 that the ROC is to be calculated for. This argument is ignored for class models.
   * @param[out] pROCCalculations    A vector data with three rows, the first contains the true positive rate, the second
   *                                 the false positive rate and the third, the threshold for the points.
   * @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetROC(SQ_Prediction pPrediction, int iYVar, SQ_VectorData* pROCCalculations);

 /*********************************************************************************************************************
 * RMSEP VECTOR
 *********************************************************************************************************************/

   /**
   *  Retrieves the RMSEP matrix.
   *  Root mean square error of the fit for observations in the predictionset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[out] pVectorData       A pointer to the RMSEP results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of y-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRMSEP(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * MBEP VECTOR
 *********************************************************************************************************************/

   /**
   *  Retrieves the MBEP matrix.
   *  Root mean square error of the fit for observations in the predictionset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  pComponents       A list of component Indices to use.
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[out] pVectorData       A pointer to the MBEP results.
   *                                Number of rows in matrix = number of components chosen (length of pComponents).
   *                                Number of columns in matrix = number of y-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetMBEP(SQ_Prediction pPrediction, SQ_IntVector* pComponents, SQ_VectorData *pVectorData);
   
 /*********************************************************************************************************************
 * Serr PS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves SerrLPS.
   *  Lower limit of the standard error of the predicted response Y
   *  (As selected) for a new observation in the prediction set.
   *  The function fails if the model is not a PLS model or 
   *  if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If False, the returned x-values will be in the scaled and centerd metric of 
   *                                the workset. Note that if bBackTransformed is false, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the SerrLPS results.
   *                                Number of rows in matrix = number of y-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSerrLPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);
      
   /**
   *  Retrieves SerrUPS.
   *  Upper limit of the standard error of the predicted response Y (as selected) 
   *  for a new observation in the prediction set.
   *  SerrUPS is always in original units, 
   *  i.e., back transformed when the response Y was transformed.
   *  The function fails if the model is not a PLS model or 
   *  if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If False, the returned x-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the SerrUPS results.
   *                                Number of rows in matrix = number of y-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetSerrUPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * XObsPS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the XObsResPS matrix from the predicted data.
   *  The residuals of an X observation in the predictionset, in original units.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                If false, the returned residuals will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the residuals will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case residuals will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the residuals in the unscaled untransformed 
   *                                metric of the workset. If false the returned residuals will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pObservations     A list of observation Indices in the prediction, 
   *                                NULL if all observations in the prediction should be used.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the XObsResPS results.
   *                                Number of rows in matrix = Number of observations chosen.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXObsResPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the XObsPredPS from the predicted data.
   *  A reconstructed observation as X=TP from the predictions set.
   *  The function fails if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                If false, the returned residuals will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the residuals will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case residuals will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the residuals in the unscaled untransformed 
   *                                metric of the workset. If false the returned residuals will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pObservations     A list of observation Indices in the prediction, 
   *                                NULL if all observations in the prediction should be used.
   *  @param[in]  bReconstruct      If the project is a wavelet spectral compressed project
   *                                and this parameter is true the returned matrix will be back transformed 
   *                                to the original domain.
   *  @param[out] pVectorData       A pointer to the XObsPredPS results.
   *                                Number of rows in matrix = Number of observations chosen.
   *                                Number of columns in matrix = number of x-variables in the model.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXObsPredPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pObservations, SQ_ReconstructState bReconstruct, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * XVarPS VECTORS
 *********************************************************************************************************************/
   
   /**
   *  Retrieves XVarPS.
   *  X variable from the predictionset, in original units.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  bUnscaled         If True, the function will return the x-values in the (unscaled) metric of the dataset.
   *                                If False, the returned x-values will be in the scaled and centerd metric of 
   *                                the workset. Note that if bBackTransformed is false, the x-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case x-values will always be unscaled.
   *  @param[in]  bBackTransformed  If True, the function will return the x-values in the unscaled untransformed 
   *                                metric of the workset. If False the returned x-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                x-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnXIndices   A list of X column Indices to use.
   *                                NULL if all x columns in the model should be used
   *                                @see GetColumnXIndexByName
   *  @param[out] pVectorData       A pointer to the XVarPS results.
   *                                Number of rows in matrix = number of x-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarPS(SQ_Prediction pPrediction, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnXIndices, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the XVarPredPS matrix from a prediction.
   *  For PLS and PCA models, X variables, from the prediction set, reconstructed as X=Tps*P'.
   *  For OPLS/O2PLS models it is the X-values predicted from from the given Y-values.
   *  The function fails if the model doesn't have any components
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The component to use
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  pColumnXIndices   A list of X column indices to use.
   *                                NULL if all x columns in the model should be used
   *  @param[in]  bUnscaled         If True, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                If False, the returned residuals will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is False, the residuals will still 
   *                                be transformed. Note also that if bBackTransformed is True, this parameter is ignored.
   *                                In this case residuals will always be unscaled.
   *  @param[in]  bBackTransformed  If True, the function will return the residuals in the unscaled untransformed 
   *                                metric of the workset. If False the returned residuals will be transformed
   *                                in the same way as the workset. Note that if this variable is True, the returned
   *                                residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the XVarPredPS results.
   *                                Number of rows in matrix = number of x-variables chosen (length of pColumnXIndices).
   *                                Number of columns in matrix = number of observations in the prediction.
   *
   *  @return                       Returns SQ_E_OK if success or an error code
   *  @see SQX_GetColumnXIndexByName
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarPredPS(SQ_Prediction pPrediction, int iComponent, SQ_IntVector* pColumnXIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the XVarResPS from the predictionset.
   *  X Variable residuals in original units, for observations in the Predictionset.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  pColumnXIndices   A list of X column Indices to use.
   *                                NULL if all x columns in the model should be used
   *                                @see GetColumnXIndexByName
   *  @param[in]  bUnscaled         If true, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                If false, the returned residuals will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the residuals will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case residuals will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the residuals in the unscaled untransformed 
   *                                metric of the workset. If false the returned residuals will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  bStandardized     If true, the function will use the standardized residuals 
   *                                (the unscaled residuals divided by their standard deviation).
   *  @param[out] pVectorData       A pointer to the XVarResPS results.
   *                                Number of rows in matrix = number of x-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetXVarResPS(SQ_Prediction pPrediction, int iComponent, SQ_IntVector* pColumnXIndices, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized, SQ_VectorData *pVectorData);

 /*********************************************************************************************************************
 * Y PS VECTORS
 *********************************************************************************************************************/

   /**
   *  Retrieves the YPredPS.
   *  Predicted values of Y variables, in original units for observations in the predictionset.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the YPredPS results.
   *                                Number of rows in matrix = number of y-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the predicted YPredPSConfInt+.
   *  The confidence intervals for predictions (basically GetYPredPScvSE * +t).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[in]   fLevel           The probability level, .95 means 95% probability. 
   *                                If -1, the default level from the model is used.
   *  @param[out] pVectorData       A pointer to the YPredPSConfInt+ results.
   *                                Number of rows in matrix = number of y-variables chosen. 
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredPSConfIntPlus(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, float fLevel, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the predicted YPredPSConfInt-.
   *  The confidence intervals for predictions (basically GetYPredPScvSE * -t).
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[in]   fLevel           The probability level, .95 means 95% probability. 
   *                                If -1, the default level from the model is used.
   *  @param[out] pVectorData       A pointer to the YPredPSConfInt- results.
   *                                Number of rows in matrix = number of y-variables chosen. 
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredPSConfIntMinus(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, float fLevel, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the predicted YPredPScv 
   *  xx cross-validated predictions for observations that were not in the training set.
   *  These predictions are computed using the xx submodels of the training set 
   *  generated from the xx rounds of cross validation.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  iColumnYIndex     Index of the Y column to use.
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the YPredPScv results.
   *                                Number of rows in matrix = number of y-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredPScv(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, int iColumnYIndex, SQ_VectorData *pVectorData);

   /**
   *  Retrieves the predicted YPredPScvSE.
   *  Jack-knife standard error of predictions for observations not in the training set
   *  computed from the cross validations.
   *  The function fails if the model is not a PLS model or if the model doesn't have any components.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the YPredPScvSE results.
   *                                Number of rows in matrix = number of y-variables chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYPredPScvSE(SQ_Prediction pPrediction, int iComponent, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the predicted YObsResPS 
   *  The residuals of an observation (Y space) in the prediction set. 
   *  Since no predictions are performed on Y, this function will only return missing values,
   *  except for the observation level of a batch project were we have Maturity or Time as Y.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *  @param[in]  pObservations     A list of observation Indices in the prediction, 
   *                                NULL if all observations in the prediction should be used.
   *  @param[in]  bUnscaled         If true, the function will return the residuals in the (unscaled) metric of the dataset.
   *                                If false, the returned residuals will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the residuals will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case residuals will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the residuals in the unscaled untransformed 
   *                                metric of the workset. If false the returned residuals will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                residuals will always be unscaled irrespective of the value of bUnscaled.
   *  @param[out] pVectorData       A pointer to the YObsResPS results.
   *                                Number of rows in matrix = number of y-variables in the model.
   *                                Number of columns in matrix = number of observations chosen.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYObsResPS(SQ_Prediction pPrediction, int iComponent, SQ_IntVector* pObservations, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the predicted YVarPS.
   *  Since no predictions are performed on Y, this function will only return missing values,
   *  except for the observation level of a batch project were we have Maturity or Time as Y.
   *  The function fails if the model is not a PLS model
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the YVarPS results.
   *                                Number of rows in matrix = number of y-variables in chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYVarPS(SQ_Prediction pPrediction, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);
   
   /**
   *  Retrieves the predicted YVarResPS.
   *  Y Variable residuals for observations in the prediction set.
   *  Since no predictions are performed on Y, this function will only return missing values,
   *  except for the observation level of a batch project were we have Maturity or Time as Y.
   *  The function fails if the model is not a PLS model.
   *
   *  @param[in]  pPrediction       The handle to the predictions, is retrieved from GetPrediction()
   *  @param[in]  iComponent        The number of the component in the model we want the results from
   *                                For an OPLS model, the last predictive component is the only valid one.
   *  @param[in]  bUnscaled         If true, the function will return the y-values in the (unscaled) metric of the dataset.
   *                                If false, the returned y-values will be in the scaled and centered metric of 
   *                                the workset. Note that if bBackTransformed is false, the y-values will still 
   *                                be transformed. Note also that if bBackTransformed is true, this parameter is ignored.
   *                                In this case y-values will always be unscaled.
   *  @param[in]  bBackTransformed  If true, the function will return the y-values in the unscaled untransformed 
   *                                metric of the workset. If false the returned y-values will be transformed
   *                                in the same way as the workset. Note that if this variable is true, the returned
   *                                y-values will always be unscaled irrespective of the value of bUnscaled.
   *  @param[in]  bStandardized     If true, the function will use the standardized residuals 
   *                                (the unscaled residuals divided by their standard deviation).
   *  @param[in]  pColumnYIndices   A list of Y column Indices to use.
   *                                NULL if all y columns in the model should be used
   *                                @see GetColumnYIndexByName
   *  @param[out] pVectorData       A pointer to the YVarResPS results.
   *                                Number of rows in matrix = number of y-variables in chosen.
   *                                Number of columns in matrix = number of observations in the prediction.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetYVarResPS(SQ_Prediction pPrediction, int iComponent, SQ_UnscaledState bUnscaled, SQ_BacktransformedState bBackTransformed, SQ_StandardizedState bStandardized, SQ_IntVector* pColumnYIndices, SQ_VectorData *pVectorData);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQPREDICTION_H_ */
