/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQPREPAREBATCHPREDICTION_H_
#define _SQPREPAREBATCHPREDICTION_H_

#include "SQDef.h"

/** @file SQPrepareBatchPrediction.h

   This file list the SQ_PrepareBatchPrediction object used to set data for a prediction.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQVariable.h"
#include "SQBatchPrediction.h"
#include "SQStringMatrix.h"

  /** @cond SQPPlus */
   
   /**
   The handle used to identify the object that sets data for a batch prediction.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_PrepareBatchPrediction 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_PrepareBatchPrediction;

   /**
   *  Removes the allocated memory for the PrepareBatchPrediction object.
   *  This function must be called for every PrepareBatchPrediction object that is created,
   *  if not a memory leak will occur.
   *
   *  @param[in]  pPreparePrediction   The PrepareBatchPrediction object to remove.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearPrepareBatchPrediction(SQ_PrepareBatchPrediction* pPreparePrediction);

   /** 
   *  Retrieves the variables that the current model are based on.
   *  These are the variables that are needed to perform a prediction.
   *  If data is missing for any of the variables, set them to "missing value".
   *  Both quantitative and qualitative variables and X and Y will be returned.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  iPhase               The index of the phase to set the data for. 
   *  @param[out] pVariables           The variables needed as input to GetBatchPrediction.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariablesForBatchPrediction(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_VariableVector* pVariables);
   
   /** 
   *  Sets qualitative data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iPhase               The index of the phase to set the data for. 
   *  @param[in]  iRow                 The row/observation to set the data for. 
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  iColumn              The column/variable to set the data for.  
   *  @param[in]  szData               The string data to predict, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeBatchData(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, int iRow, int iColumn, const char* szData);

   /** 
   *  Sets qualitative data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iPhase               The phase to set the data for. 
   *  @param[in]  pMatrix              A matrix with the same number of columns as the phase it should set prediction data for.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeBatchDataMatrix(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_StringMatrix pMatrix);
   /** 
   *  Sets quantitative data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iPhase               The phase to set the data for. 
   *  @param[in]  iRow                 The row/observation to set the data for. 
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  iColumn              The column/variable to set the data for.  
   *  @param[in]  fVal                 The float data to predict.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeBatchData(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, int iRow, int iColumn, const float fVal);

   /** 
   *  Sets quantitative data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iPhase               The phase to set the data for. 
   *  @param[in]  pMatrix              A matrix with the same number of columns as the phase it should set prediction data for.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeBatchDataMatrix(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_FloatMatrix pMatrix);

   /** 
   *  Sets quantitative data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iPhase               The phase to set the data for. 
   *  @param[in]  iNumberOfRows        The number of rows included in the prediction data 
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  pValues              A pointer to a matrix of float values with the same number of columns as the phase it should set prediction data for.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeBatchDataRaw(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, int iNumberOfRows, const float* pValues);
    
 /*********************************************************************************************************************
 * Local Centering
 *********************************************************************************************************************/

   /**
   *  A request for the local centering connected with the given model number.
   *  If the model is a batch model and local centering was not specified for a variable in this phase, 
   *  but has been specified for the variable in another phase, the average of all phases will be returned for this phase.
   *
   *  @param[in]   pPreparePrediction  The PreparePrediction object to use for the prediction. 
   *  @param[in]   iPhase              The phase to get the Local Centering information for. 
   *  @param[out]  pVariables          The variables that local centering data was found for.
   *  @param[out]  pDefaultValues      The default values.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLocalCenteringInfo(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_VariableVector* pVariables, SQ_FloatVector* pDefaultValues);
   /**
   *  Will specify new local centering values that will be used when performing predictions. 
   *
   *  @param[in]  pPreparePrediction      The PreparePrediction object to use for the prediction. 
   *  @param[in]  iPhase                  The phase to specify the Local Centering information for. 
   *  @param[in]  pVariables              The variables that local centering data was found for.
   *  @param[in]  pLocalCenteringValue    The new center values to use.
   *  @return                             Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SpecifyLocalCentering(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_VariableVector pVariables, SQ_FloatVector pLocalCenteringValue);

/*********************************************************************************************************************
 * BATCH, PHASE AND PHASE ITERATION CONDITIONS
 *********************************************************************************************************************/
   
   /** 
   *  Retrieves the batch and phase condition variables that are included in the BLM.
   *  If data is missing for any of the variables, set them to "missing value".
   *  Both quantitative and qualitative variables and X and Y will be returned.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[out] pBCVariables         The Batch condition variables needed as input to GetBatchPrediction.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetBatchConditionVariablesForBatchPrediction(SQ_PrepareBatchPrediction pPreparePrediction, SQ_VariableVector* pBCVariables);

   /**
   *  Retrieves the phase iteration condition variables that are included in the BLM.
   *  If data is missing for any of the variables, set them to "missing value".
   *  Both quantitative and qualitative variables and X and Y will be returned.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.
   *  @param[out] pBCVariables         The phase iteration condition variables needed as input to GetBatchPrediction.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPhaseIterationConditionVariablesForBatchPrediction(SQ_PrepareBatchPrediction pPreparePrediction, SQ_VariableVector* pBCVariables);

   /** 
   *  Sets qualitative Batch/Phase Condition data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iBCIndex             The index of the Batch/Phase Condition variable to set the data for.  
   *  @param[in]  szData               The string data to predict, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeBatchConditionData(SQ_PrepareBatchPrediction pPreparePrediction, int iBCIndex, const char* szData);

   /**
   *  Sets the phase iteration for each observation, the vectur muste bee sorted ascending.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.
   *  @param[in]  iPhase               The index of the phase to set the data for. 
   *  @param[in]  pPhaseIterations     The phase iteration indexes for each observation in the phase, the vector must be sorted ascending.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPhaseIterations(SQ_PrepareBatchPrediction pPreparePrediction, int iPhase, SQ_IntVector pPhaseIterations);

   /** 
   *  Sets qualitative Batch/Phase Condition data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  pData                The string data for batch predictions. Must be the number of columns as batch and phase condition variables.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeBatchConditionDataVector(SQ_PrepareBatchPrediction pPreparePrediction, SQ_StringVector pData);
   /** 
   *  Sets quantitative Batch/Phase Condition data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  iBCIndex             The index of the Batch/Phase Condition variable to set the data for.  
   *  @param[in]  fVal                 The float data to predict.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeBatchConditionData(SQ_PrepareBatchPrediction pPreparePrediction, int iBCIndex, const float fVal);
   /** 
   *  Sets quantitative Batch/Phase Condition data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[in]  pData                The string data for batch predictions. Must be the number of columns as batch and phase condition variables.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeBatchConditionDataVector(SQ_PrepareBatchPrediction pPreparePrediction, SQ_FloatVector pData);

   /**
   *  Sets qualitative Phase Iteration Condition data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.
   *  @param[in]  iPCVarIndex          The index of the Phase Iteration Condition variable to set the data for.
   *  @param[in]  iPItCIndex           The index of the Phase Iteration Condition to set the data for.
   *  @param[in]  szPhase              The name of the Phase.
   *  @param[in]  szData               The string data to predict, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativePhaseIterationConditionData(SQ_PrepareBatchPrediction pPreparePrediction, int iPCVarIndex, const char* szPhase, int iPItCIndex, const char* szData);

   /**
   *  Sets qualitative Phase Iteration Condition data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.
   *  @param[in]  szPhase              The name of the Phase.
   *  @param[in]  pData                The string data for batch predictions. Must be one row per phase iteration and one column per phase iteration condition variable.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativePhaseIterationConditionDataMatrix(SQ_PrepareBatchPrediction pPreparePrediction, const char* szPhase, SQ_StringMatrix pData);
   /**
   *  Sets quantitative Phase Iteration Condition data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.
   *  @param[in]  iPCVarIndex          The index of the Phase Iteration Condition variable to set the data for.
   *  @param[in]  szPhase              The name of the Phase.
   *  @param[in]  iPItCIndex           The index of the Phase Iteration Condition in that pahse to set the data for.
   *  @param[in]  fVal                 The float data to predict.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativePhaseIterationConditionData(SQ_PrepareBatchPrediction pPreparePrediction, int iPCVarIndex, const char* szPhase, int iPItCIndex, const float fVal);
   /**
   *  Sets quantitative Phase Iteration Condition data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.
   *  @param[in]  szPhase              The name of the Phase.
   *  @param[in]  pData                The string data for batch predictions. Must be one row per phase iteration and one column per phase iteration condition variable.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativePhaseIterationConditionDataMatrix(SQ_PrepareBatchPrediction pPreparePrediction, const char* szPhase, SQ_FloatMatrix pData);

/*********************************************************************************************************************
 * BATCH PREDICTION
 *********************************************************************************************************************/
   /** 
   *  Performs a batch prediction on the data that have been set with SQ_SetQualitativeBatchData and SQ_SetQuantitativeBatchData.
   *  The returned object must be released by SQ_ClearBatchPrediction.
   *
   *  @param[in]  pPreparePrediction   The data to use for the prediction.  
   *  @param[out] pPrediction          The prediction object that should be used to retrieve the prediction results.   
   *                                   Must be released by SQ_ClearBatchPrediction.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetBatchPrediction(SQ_PrepareBatchPrediction pPreparePrediction, SQ_BatchPrediction* pPrediction);

  /** @endcond // end of SQPPlus */

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQPREPAREBATCHPREDICTION_H_ */
