/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQPREPAREPREDICTION_H_
#define _SQPREPAREPREDICTION_H_

#include "SQDef.h"

/** @file SQPreparePrediction.h

   This file list the SQ_PreparePrediction object used to set data for a prediction.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQVariable.h"
#include "SQPrediction.h"
#include "SQStringMatrix.h"

   /**
   The handle used to identify the object that sets data for a prediction.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_PreparePrediction 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_PreparePrediction;

   /**
   *  Removes the allocated memory for the PreparePrediction object.
   *  This function must be called for every PreparePrediction object that is created,
   *  if not a memory leak will occur.
   *
   *  @param[in]  pPreparePrediction  The PreparePrediction object to remove.
   *
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearPreparePrediction(SQ_PreparePrediction* pPreparePrediction);

   /** 
   *  Retrieves the variables that the current model are based on.
   *  These are the variables that are needed to perform a prediction.
   *  If data is missing for any of the variables, set them to "missing value".
   *  Both quantitative and qualitative variables and X and Y will be returned.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[out] pVariables           The variables needed as input to GetPrediction.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariablesForPrediction(SQ_PreparePrediction pPreparePrediction, SQ_VariableVector* pVariables);
   /** 
   *  Sets qualitative data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  iRow                 The row/observation to set the data for. 
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  iVariableIndex       The variable to set the data for, represents the variable index returned by SQ_GetVariablesForPrediction.  
   *  @param[in]  szData               The string data to predict, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeData(SQ_PreparePrediction pPreparePrediction, int iRow, int iVariableIndex, const char* szData);

   /** 
   *  Sets qualitative data for the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  oData                A string matrix to use for the predictions
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeDataMatrix(SQ_PreparePrediction pPreparePrediction, SQ_StringMatrix oData);

   /** 
   *  Sets qualitative data for a lagged variable to be used in the prediction.
   *  This function can be used even for quantitative data, the string will then be converted to a float.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  pVar                 The variable to set the lagged data for.
   *  @param[in]  iLagStep             The lag step to set the value for.  
   *  @param[in]  szData               The string data to predict, UTF-8 encoded.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQualitativeLagData(SQ_PreparePrediction pPreparePrediction, SQ_Variable pVar, int iLagStep, const char* szData);
   /** 
   *  Sets quantitative data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  iRow                 The row/observation to set the data for. 
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  iVariableIndex       The variable to set the data for, represents the variable index returned by SQ_GetVariablesForPrediction.  
   *  @param[in]  fVal                 The float data to predict.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeData(SQ_PreparePrediction pPreparePrediction, int iRow, int iVariableIndex, const float fVal);

   /**
   *  Sets class for an observation in the prediction set
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.
   *  @param[in]  iRow                 The row/observation to set the data for.
   *  @param[in]  iClass               The new class.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetPredictClass(SQ_PreparePrediction pPreparePrediction, int iRow, int iClass);

   /** 
   *  Sets quantitative data for the prediction with raw float values. Must be of the size NumberOfVariablesToPredict x iRows
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  iRows                Number of rows that to set values for
   *                                   The matrix will automatically be increased to the correct size.
   *  @param[in]  pfVals               The float data to predict. Must be of correct size!
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeDataRaw(SQ_PreparePrediction pPreparePrediction, int iRows, const float* pfVals);

   /** 
   *  Sets quantitative data for the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  pMatrix              The matrix to set as prediction data
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL  SQ_SetQuantitativeDataMatrix(SQ_PreparePrediction pPreparePrediction, SQ_FloatMatrix pMatrix);
   /** 
   *  Sets quantitative data for a lagged variable to be used in the prediction.
   *  This function can be used even for qualitative data, the index of the settings should then be used.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[in]  pVar                 The variable to set the lagged data for.
   *  @param[in]  iLagStep             The lag step to set the value for.  
   *  @param[in]  fVal                 The float data to predict.   
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetQuantitativeLagData(SQ_PreparePrediction pPreparePrediction, SQ_Variable pVar, int iLagStep, const float fVal);
   /** 
   *  Performs a prediction on the data that have been set with SQ_SetQualitativeData and SQ_SetQuantitativeData.
   *  The returned object must be released by SQ_ClearPrediction.
   *
   *  @param[in]  pPreparePrediction   The PreparePrediction object to use for the prediction.  
   *  @param[out] pPrediction          The prediction object that should be used to retrieve the prediction results.   
   *                                   Must be released by SQ_ClearPrediction or is removed when the model is closed.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetPrediction(SQ_PreparePrediction pPreparePrediction, SQ_Prediction* pPrediction);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQPREPAREPREDICTION_H_ */
