/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQPROJECT_H_
#define _SQPROJECT_H_

#include "SQDef.h"

/** @file SQProject.h
   This file list the SQ_Project object and all functions that can be performed on a project.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

   /** 
   The project handle used to identify an opened project.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_ProjectHandle
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Project;

#include "SQErrorCodes.h"
#include "SQModel.h"
#include "SQDataset.h"
#include "SQCommon.h"

   /**
   *  Open an existing project (.usp).
   *  SIMCA-Q can handle multiple project that are identified with a project handle.
   *  It is the users responsibility to close the project when finished (SQ_CloseProject).
   *
   *  @param[in]  szProjectName     The full path to the SIMCA project file (.usp), UTF-8 encoded.
   *  @param[in]  szPassword        If the usp is encrypted, supply the password that was used to encrypt, UTF-8 encoded, else use NULL.
   *  @param[out] pProject          The handle to this project.
   *
   *  @see SQ_CloseProject
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_OpenProject(const char *szProjectName, const char *szPassword, SQ_Project *pProject);
   
   /**
   *  Close an opened project.
   *  Every project that has been opened with OpenProject must be closed with this function.
   *
   *  @param[in]  pProject       The project to close.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CloseProject(SQ_Project* pProject);

   /**
   *  Retrieves the name of the project.
   *
   *  @param[in]     pProject          The project handle to get the name from.
   *  @param[in,out] pszProjectName    A buffer to where the project name should be stored, UTF-8 encoded.
   *                                   The user is responsible for allocation/deallocation.
   *  @param[in]     iLength           The size of the buffer.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetProjectName(SQ_Project pProject, char* pszProjectName, int iLength);

   /**
   *  Retrieves if the project is encrypted or not.
   *
   *  @param[in]  szProjectName     The full path to the SIMCA project file (.usp), UTF-8 encoded.
   *  @param[out] pIsEncrypted      True if the project is encrypted, otherwise false.
   *  @return                          Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsProjectEncrypted(const char *szProjectName, SQ_Bool* pIsEncrypted);

   /**
   *  Check if the project is reduced, all results will not be available for reduced projects.
   *  @see SQ_AreDatasetsRemoved and @see SQ_AreResidualsRemoved
   *
   *  @param[in]  pProject          The project.
   *  @param[out] pIsReduced        True if the project is reduced
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsProjectReduced(SQ_Project pProject, SQ_Bool* pIsReduced);

   /**
   *  Check if the project is reduced, all results will not be available for reduced projects.
   *  @see SQ_IsProjectReduced
   *
   *  @param[in]  pProject          The project.
   *  @param[out] pAreDatasetsRemoved True if datasets are removed from the project
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AreDatasetsRemoved(SQ_Project pProject, SQ_Bool* pAreDatasetsRemoved);

   /**
   *  Check if the project is reduced, all results will not be available for reduced projects.
   *  @see SQ_IsProjectReduced
   *
   *  @param[in]  pProject          The project.
   *  @param[out] pAreResidualsRemoved True if the model residuals are removed
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_AreResidualsRemoved(SQ_Project pProject, SQ_Bool* pAreResidualsRemoved);

   
   /**
   *  Does the project contain embedded python scripts, predictions will fail if the project contains scripts and EnableEmbeddedScripts is not called.
   *
   *  @param[in]  pProject          The project.
   *  @param[out] pIsHasScripts     True if the project contains embedded python scripts
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ProjectHasPythonScripts(SQ_Project pProject, SQ_Bool* pIsHasScripts);

   /**
   *  Enable embedded python scripts, predictions will fail if the project contains embedded scripts and this is not called.
   *
   *  @param[in]  pProject          The project.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_EnableEmbeddedScripts(SQ_Project pProject);
   
   /**
   *  Retrieves a model from the project. The models are closed automatically when the parent project is closed.
   *
   *  @param[in]  pProject       The project to get the model from.
   *  @param[in]  iModelNumber   The model number of the model (not model index).
   *  @param[out] pModel         The handle to the model
   *
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModel(SQ_Project pProject, int iModelNumber, SQ_Model* pModel);

   /**
   *  Retrieves information about a model from the project.
   *
   *  @param[in]  pProject       The project to get the model from.
   *  @param[in]  iModelNumber   The model number of the model (not model index).
   *  @param[out] pModelInfo  The struct to fill with model information
   *
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelInfo(SQ_Project pProject, int iModelNumber, SQ_ModelInfo* pModelInfo);

   /**
   *  Retrieves the number of models in the project, including unfitted models. @see SQ_IsModelFitted.
   *
   *  @param[in]  pProject       The project to get the number of models from
   *  @param[out] pnNrOfModels   The total number of models in the project.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfModels(SQ_Project pProject, int *pnNrOfModels);

   /**
   *  Retrieves the model number from an index of a model.
   *
   *  @param[in]  pProject       The project to get the model number from
   *  @param[in]  iModelIndex    The index of the model to get the number of,
   *                             between 1 and the result from GetNumberOfModels.
   *  @param[out] piModelNumber  The model number.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetModelNumberFromIndex(SQ_Project pProject, int iModelIndex, int *piModelNumber);

   /**
   *  Retrieves a dataset from the project. The datasets are closed automatically when the project is closed.
   *
   *  @param[in]  pProject          The project to get the dataset from.
   *  @param[in]  iDatasetNumber    The dataset number of the dataset (not dataset index).
   *  @param[out] pDataset          The handle to the dataset.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataset(SQ_Project pProject, int iDatasetNumber, SQ_Dataset* pDataset);
   /**
   *  Retrieves the number of data sets in the project.
   *
   *  @param[in]  pProject          The project to use
   *  @param[out] pnNrOfDatasets    The number of data sets in the project.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumberOfDatasets(SQ_Project pProject, int *pnNrOfDatasets);
   /**
   *  Retrieves the number of a data set given an index.
   *
   *  @param[in]  pProject          The project to use
   *  @param[in]  iDatasetIndex     The index of the project to get the number of.
   *  @param[out] piDatasetNumber   The number of the data set.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDatasetNumberFromIndex(SQ_Project pProject, int iDatasetIndex, int *piDatasetNumber);

   /**
   *  Checks if the project is read-only.
   *
   *  @param[in]   pProject         The project to use
   *  @param[out]  bIsReadOnly      True if the project is read-only, False otherwise.
   *  @return                       Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsReadOnly(SQ_Project pProject, SQ_Bool* bIsReadOnly);

   /**
   * Save the project.
   *
   * @param[in]   pProjHandle        The project handle to use
   * @return                         Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_Save(SQ_Project pProjHandle);

   /**
   * Checks if a project has data that has not been saved to disk
   *
   * @param[in]   pProjHandle        The project handle to use
   * @param[out]  pbIsDirty          Will be True if the project has data that has not been saved,
   *                                   otherwise False.
   *  @return                        Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsProjectDirty(SQ_Project pProjHandle, SQ_Bool *pbIsDirty);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQPROJECT_H_ */
