/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/

#ifndef SQ_STRINGMATRIX_H
#define SQ_STRINGMATRIX_H

#include "SQDef.h"

/** @file SQStringMatrix.h

   This file list the SQStringMatrix object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"


   /**
   A string matrix that is used to get and send strings to many functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_StringMatrix 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_StringMatrix;

   /**
   *  Initialize a string matrix with its size.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to be initialized.
   *  @param[in]   iRow          Number of rows to give the matrix.
   *  @param[in]   iColumn       Number of columns to give the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitStringMatrix(SQ_StringMatrix* pMatrix, int iRow, int iColumn);

   /**
   *  Get a copy of the string matrix 
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pInMatrix        The matrix to copy.
   *  @param[in, out]   pOutMatrix  The result matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyStringMatrix(SQ_StringMatrix pInMatrix, SQ_StringMatrix* pOutMatrix);

   /**
   *  Removes the matrix, this function must be called for every matrix
   *  that has been initialized with SQ_InitStringMatrix.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearStringMatrix(SQ_StringMatrix* pMatrix);

   /**
   *  Sets the string in the given cell.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the string in.
   *  @param[in]   iRow          The row to set the string in.
   *  @param[in]   iColumn       The column to set the string in.
   *  @param[in]   szString      The string to set the cell with, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetStringInMatrix(SQ_StringMatrix pMatrix, int iRow, int iColumn, const char* szString);

   /**
   *  Gets the string from a given cell.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the string from.
   *  @param[in]   iRow          The row to get the string from.
   *  @param[in]   iColumn       The column to get the string from.
   *  @param[out]  pszString     The string in the cell. A pointer to where the string should be stored, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStringFromMatrix(SQ_StringMatrix pMatrix, int iRow, int iColumn, char** pszString);
   
   /**
   *  Sets the strings in the given row.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the strings in.
   *  @param[in]   iRow          The row to set the strings in.
   *  @param[in]   pszStrings    An array of strings to set the row with. Must be of same length as number of columns, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetRowStringsInMatrix(SQ_StringMatrix pMatrix, int iRow, const char** pszStrings);

   /**
   *  Sets the strings in the given column.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the strings in.
   *  @param[in]   iColumn       The column to set the strings in.
   *  @param[in]   pszStrings    An array of strings to set the column with. Must be of same length as number of rows, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetColumnStringsInMatrix(SQ_StringMatrix pMatrix, int iColumn, const char** pszStrings);

   /**
   *  Sets the strings in the matrix.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to set the strings in.
   *  @param[in]   pszStrings    A matrix of strings to fill the StringMatrix, UTF-8 encoded. Must be of same size as the StringMatrix.
   *                             The format is Row 1, Row 2, etc.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetStringMatrix(SQ_StringMatrix pMatrix, const char* const* pszStrings);

   /**
   *  Gets the strings from a given row.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the strings from.
   *  @param[in]   iRow          The row to get the strings from.
   *  @param[out]  pszStrings    The strings in given row, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRowStringsFromMatrix(SQ_StringMatrix pMatrix, int iRow, char*** pszStrings);

   /**
   *  Gets the strings from a given column.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the strings from.
   *  @param[in]   iColumn       The column to get the strings from.
   *  @param[out]  pszStrings    The strings in the given column, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnStringsFromMatrix(SQ_StringMatrix pMatrix, int iColumn, char*** pszStrings);

   /**
   *  Gets the strings from the matrix.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the strings from.
   *  @param[out]  pszStrings    A pointer to a matrix to be filled with strings from the StringMatrix, UTF-8 encoded.
   *                             The format is Row 1, Row 2, etc.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStringMatrix(SQ_StringMatrix pMatrix, char*** pszStrings);

   /**
   *  Returns the number of rows in the matrix.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the number of rows from.
   *  @param[out]  piNumRows     The number of rows in the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumRowsInStringMatrix(SQ_StringMatrix pMatrix, int* piNumRows);

   /**
   *  Returns the number of columns in the matrix.
   *  @see tagSQ_StringMatrix
   *
   *  @param[in]   pMatrix       The matrix to get the number of columns from.
   *  @param[out]  piNumCols     The number of columns in the matrix.
   *  @return                    Returns SQ_E_OK if success or an error code.
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumColumnsInStringMatrix(SQ_StringMatrix pMatrix, int* piNumCols);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* SQ_STRINGMATRIX_H */
