/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQSTRINGVECTOR_H_
#define _SQSTRINGVECTOR_H_

#include "SQDef.h"

/** @file SQStringVector.h

   This file list the SQStringVector object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"

   /**
   A string vector that is used to get and send variable names to many functions. Index base = 1.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_StringVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_StringVector;

   /**
   *  Initialize a string vector with its size.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The vector to be initialized.
   *  @param[in]   iSize         Number of strings to give the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_InitStringVector(SQ_StringVector* pVector, int iSize);

   /**
   *  Get a copy of the string vector 
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pInVector        The vector to copy.
   *  @param[in, out]   pOutVector  The result vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_CopyStringVector(SQ_StringVector pInVector, SQ_StringVector* pOutVector);
   /**
   *  Removes the Vector, this function must be called for every Vector that is created.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The vector to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearStringVector(SQ_StringVector* pVector);
   /**
   *  Sets the string in the given position.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The vector to set the string in.
   *  @param[in]   iPos          The position to set the string in.
   *  @param[in]   szString      The string to set, UTF-8 encoded.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetStringInVector(SQ_StringVector pVector, int iPos, const char *szString);
   /**
   *  Set the strings in vector.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The vector to set the string in.
   *  @param[in]   pszString     The strings to set, UTF-8 encoded. Must be of the same length as the vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_SetStringVector(SQ_StringVector pVector, const char **pszString);
   /**
   *  Get the strings from the vector.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The vector to set the string in.
   *  @param[out]  pszString     The strings in the vector, UTF-8 encoded. This is an array of strings of the same length as StringVector.
   *                             IMPORTANT: it's up to the user to free the allocated memory by calling free() for each
   *                             string in the array;
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStringVector(SQ_StringVector pVector, char ***pszString);
   /**
   *  Gets the string from a given position in the Vector.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The Vector to get the string from.
   *  @param[in]   iPos          The position in the Vector to get the string from.
   *  @param[in,out] pszString   A buffer to where the string should be stored, UTF-8 encoded.
   *                             The user is responsible for allocation/deallocation.
   *  @param[in]   iLength       The size of the buffer.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetStringFromVector(const SQ_StringVector pVector, int iPos, char *pszString, int iLength);
   /**
   *  Returns the number of strings in the Vector.
   *  @see tagSQ_StringVector
   *
   *  @param[in]   pVector       The Vector to get the number of strings from.
   *  @param[out]  piNumStrings  The number of strings in the Vector or a negative number if an error occurred.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumStringsInVector(const SQ_StringVector pVector, int* piNumStrings);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQSTRINGVECTOR_H_ */
