/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQVARIABLE_H_
#define _SQVARIABLE_H_

#include "SQDef.h"

/** @file SQVariable.h

   This file list the SQVariable object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQCommon.h"
#include "SQIntVector.h"
#include "SQStringVector.h"

   /**
   The object used to identify a vector of variables.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_VariableVector 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_VariableVector;

   /**
   The object used to identify a variable.
   */
   typedef struct tagSQ_Variable 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_Variable;
   
/*********************************************************************************************************************
 * SQ_VariableVector
 *********************************************************************************************************************/

   /**
   *  Returns the number of variables in the Vector.
   *
   *  @param[in]   pVariables    The VariableVector to get the number of variables from.
   *  @param[out]  iSize         The number of variables in the Vector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetNumVariablesInVector(SQ_VariableVector pVariables, int* iSize);
   /**
   *  Gets the variable from a given position in the Vector.
   *
   *  @param[in]   pVariables    The VariableVector to get the variable from.
   *  @param[in]   iIndex        The index in the Vector to get the variable from.
   *  @param[out]  pVariable     The variable requested.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariableFromVector(SQ_VariableVector pVariables, int iIndex, SQ_Variable* pVariable);

/*********************************************************************************************************************
 * SQ_Variable
 *********************************************************************************************************************/

   /**
   *  Gets the name of a variable
   *
   *  @param[in]     pVariable      The variable to get the name of.
   *  @param[in]     iNameID        The ID of the name to get.
   *  @param[in,out] szVariableName A buffer to where the name should be stored, UTF-8 encoded.
   *                                The user is responsible for allocation/deallocation.
   *  @param[in]     iBufferLength  The size of the buffer.
   *  @return                       Returns SQ_E_OK if success or an error code
   *  @see SQ_GetVariableIDs
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariableName(SQ_Variable pVariable, int iNameID, char* szVariableName, int iBufferLength);
   /**
   *  Gets the variable IDs used in the current model/dataset.
   *
   *  @param[in]   pVariable     The variable to use.
   *  @param[out]  pVariableIDs  The variable IDs used.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetVariableIDs(SQ_Variable pVariable, SQ_IntVector* pVariableIDs);
   /**
   *  Gets information whether the variable is qualitative or not.
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] bIsQualitative SQ_True if the variable is qualitative. 
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsQualitative(SQ_Variable pVariable, SQ_Bool* bIsQualitative);
   /**
   *  Gets the settings for the qualitative variable.
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] pSettings      The settings, empty if the variable is not qualitative.
   *                             The vector must be cleared with SQ_ClearStringVector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetQualitativeSettings(SQ_Variable pVariable, SQ_StringVector *pSettings);
   /**
   *  Gets information whether the variable has been lagged or not.
   *  Only valid if the variable is from a model.
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] bIsLagged      SQ_True if the variable is lagged. 
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsLagged(SQ_Variable pVariable, SQ_Bool* bIsLagged);

   /**
   *  Gets information whether the variable has dynamic lags or not, if so predictions might quire more lagged data if the process is faster then when modeled.
   *  Only valid if the variable is from a model
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] bHasDynamicLags   SQ_True if the variable is lagged.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_HasDynamicLags(SQ_Variable pVariable, SQ_Bool* bHasDynamicLags);

   /**
   *  Gets the lag steps for the variable.
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] pLagSteps      The lag steps for the variable.
   *                             The vector must be cleared with SQ_ClearIntVector.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetLagSteps(SQ_Variable pVariable, SQ_IntVector* pLagSteps);

   /**
   *  Gets information whether the variable belongs to the X or Y block.
   *  Only valid if the variable is from a model.
   *
   *  @param[in]  pVariable      The variable to get the name of.
   *  @param[out] bIsXVariable   SQ_True if the variable is an X variable. 
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsXVariable(SQ_Variable pVariable, SQ_Bool* bIsXVariable);

   /**
   *  Gets information whether the variable is phase iteration condition or not.
   *  Only valid if the variable is from a model.
   *
   *  @param[in]  pVariable      The variable to get the information of.
   *  @param[out] bIsPhaseIterVariable   SQ_True if the variable is a phase iteration condition variable.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_IsPhaseIterationVariable(SQ_Variable pVariable, SQ_Bool* bIsPhaseIterVariable);

#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /* _SQVARIABLE_H_ */
