/*
Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
*/
#ifndef _SQVECTORDATA_H_
#define _SQVECTORDATA_H_

#include "SQDef.h"

/** @file SQVectorData.h
   This file list the SQ_VectorData object used in Umetrics SIMCA-Q products.
*/
#ifdef __cplusplus
extern "C" {
#endif 

#ifdef _WIN32
#pragma pack(push, 8)
#endif

#include "SQErrorCodes.h"
#include "SQFloatMatrix.h"
#include "SQStringVector.h"

   /**
   A vector data is used to get the results for requested vectors.
   It consists of a SQ_FloatMatrix with the data, a SQ_StringVector for the row names and a SQ_StringVector for the column names.
   IMPORTANT: Always initialize it to NULL!
   */
   typedef struct tagSQ_VectorData 
   {
      /** Reserved, only used internally. */
      void* reserved;
   } *SQ_VectorData;

   /**
   *  Removes the VectorData, this function must be called for every VectorData that is returned by any SQ function.
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to be removed.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_ClearVectorData(SQ_VectorData* pVectorData);
   /**
   *  Get the data matrix for the VectorData. DO NOT REMOVE this SQ_FloatMatrix, it will be removed by SQ_ClearVectorData
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to get the matrix from.
   *  @param[out]  pMatrix       The data matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataMatrix(SQ_VectorData pVectorData, SQ_FloatMatrix* pMatrix);

   /**
   *  Get the data matrix for the VectorData. This is a copy of the matrix kept by vector data and must be manually removed
   *  using SQ_ClearStringMatrix.
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to get the matrix from.
   *  @param[out]  pMatrix       The data matrix.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetDataMatrixCopy(SQ_VectorData pVectorData, SQ_FloatMatrix* pMatrix);
   /**
   *  Get the row names for the VectorData. DO NOT REMOVE this SQ_StringVector, it will be removed by SQ_ClearVectorData
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to get the row names from.
   *  @param[out]  pRowNames     The names of the rows.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRowNames(SQ_VectorData pVectorData, SQ_StringVector* pRowNames);
   
   /**
   *  Get the row alias names (the name of the primary and secondary IDs ) for the VectorData. 
   *  @see SQ_VectorData pAliasNames should be freed with SQ_ClearStringVector
   *
   *  @param[in]   pVectorData   The VectorData to get the row names from.
   *  @param[out]  pAliasNames   The names of the primary and secondary ID's.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRowAliases(SQ_VectorData pVectorData, SQ_StringVector* pAliasNames);

   /**
   *  Get the row names for the VectorData. pRowNames should be freed with SQ_ClearStringVector
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to get the row names from.
   *  @param[in]   iAlias        The index of the ID to get, 1 is the primary alias and the same as calling SQ_GetRowNames.
   *  @param[out]  pRowNames     The names of the rows.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetRowNames2(SQ_VectorData pVectorData, int iAlias, SQ_StringVector* pRowNames);

   /**
   *  Get the column names for the VectorData. DO NOT REMOVE this SQ_StringVector, it will be removed by SQ_ClearVectorData
   *  @see SQ_VectorData
   *
   *  @param[in]   pVectorData   The VectorData to get the row names from.
   *  @param[out]  pColumnNames  The names of the columns.
   *  @return                    Returns SQ_E_OK if success or an error code
   */
   SQCEXPORT SQ_ErrorCode SQCDECL SQ_GetColumnNames(SQ_VectorData pVectorData, SQ_StringVector* pColumnNames);


#ifdef _WIN32
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif /*__cplusplus*/


#endif /* _SQVECTORDATA_H_ */
