import sys
import random
from win32com import client as win32


# Function to create a fake prediction set
def create_fake_predictionset(model):
    prepare_prediction = model.PreparePrediction()
    variable_vector = prepare_prediction.GetVariablesForPrediction()
    num_variables = variable_vector.GetSize()
    num_observations = 1

    for index in range(1, num_variables + 1):
        variable = variable_vector.GetVariable(index)
        is_qualitative = variable.IsQualitative()

        for row in range(1, num_observations + 1):
            if is_qualitative == True:
                qualitative_settings = variable.GetQualitativeSettings()
                prepare_prediction.SetQualitativeData(row, index, qualitative_settings.Get(1))
            else :
                prepare_prediction.SetQuantitativeData(row, index, random.uniform(1, 100))
    
    return prepare_prediction


# Function to print all prediction values
def print_prediction_values(vector_data):
    data_matrix = vector_data.GetDataMatrix()
    row_names = vector_data.GetRowNames()
    column_names = vector_data.GetColumnNames()
    num_rows = data_matrix.GetNumberOfRows()
    num_columns = data_matrix.GetNumberOfCols()
    
    # Print all column names 
    for column in range(1, num_columns + 1):
        print(column_names.GetData(column), end=" ")

    print()
    # Print the data
    for row in range(1, num_rows + 1):
        # Print the row name
        print(f'{row_names.GetData(row)}:', end=" ")
        
        # Print the values
        for column in range(1, num_columns + 1):
            print(data_matrix.GetData(row, column), end=" ")
        
        print()


def simcaq_example():
    
    # The SIMCA project to use
    project_path = r"path\to\simca\project.usp"

    # Get the COM interface for SIMCA-Q
    simcaq = win32.gencache.EnsureDispatch('Umetrics.SIMCAQ')
    #simcaq = win32.Dispatch('Umetrics.SIMCAQ')
    CONSTANTS = win32.constants

    try:
        # Check that there is a valid license file
        if not simcaq.IsLicenseFileValid():
            sys.exit("Invalid license file")

        # Open the project and get some information
        project = simcaq.OpenProject(project_path, "")

        # Get the first model
        model_number = project.GetModelNumberFromIndex(1)
        model = project.GetModel(model_number)

        # Print some model info
        print(f'Model name: {model.GetName()}')
        print(f'Model type: {model.GetModelTypeString()}')
        print(f'Number of components: {model.GetNumberOfComponents()}')

        # Create a fake prediction set. In a real product, you should get prediction values from actual data
        prediction_data = create_fake_predictionset(model)

        # Do the actual prediction
        prediction = prediction_data.GetPrediction()

        # Get the predicted values for all components
        tPS = prediction.GetTPS(None)
        print_prediction_values(tPS)

        dmodXPS = prediction.GetDModXPS(None, CONSTANTS.eNormalized_True, CONSTANTS.eModelingPowerWeighted_False)
        print_prediction_values(dmodXPS)

        # Close the project
        project.DisposeProject()
    except Exception as e: 
        print(simcaq.GetErrorDescription(e.excepinfo[5]))

if __name__ == '__main__':
    simcaq_example()