/*
 * linux/arch/m68knommu/platform/5307/memory_io.c
 *
 * This file contains a non interruptible memory i/o function
 */
#include <linux/syscalls.h>
#include <linux/interrupt.h>
#include <asm/sag_efg.h>

asmlinkage int sys_cmio (void *dst_ptr, const void* src_ptr,unsigned number_of_bytes)
{
	unsigned long flags;

	save_flags(flags); cli();
	memcpy(dst_ptr,src_ptr,number_of_bytes);
	restore_flags(flags);
	return 0;
}

// example
// bits=01010000
// mask=11110000
//
// sys_write_mask((BYTE*)0x....,bits,mask);

//  set= 01010000
//  res  10100000

//  byte     abcdefgh
//  byte|set a1c1efgh
//  ~res =   01011111
//  &~res    0101efgh
//
asmlinkage int sys_write_mask(unsigned char*byte,unsigned char data,unsigned char mask)
{
	unsigned long flags;
	unsigned set=  data &mask;
	unsigned res=(~data)&mask;
	save_flags(flags); cli();
	*byte=(*byte|set)&~res;
	restore_flags(flags);
	return 0;
}

#define MAX_EFG 16


static DECLARE_WAIT_QUEUE_HEAD(efgwqueue);   /* use ver 2.4 static declaration - ron */
static DECLARE_MUTEX(efgmutex);

static long efgs[MAX_EFG]={0};

int __init cefg_init(void);

int __init cefg_init()                                  /* the __init added by ron  */
{
	memset(efgs,0,sizeof(efgs));
	init_waitqueue_head(&efgwqueue);    /* was init_waitqueue() --Ron */
	printk ("Sartorius EventFlags installed\n");
	return 0;
}


static void efg_timer(unsigned long data)
{
	*(int*)data=1;
	wake_up(&efgwqueue);	/* transfer finished */
}

static int and_test(long*efg,long mask)
{
	return (*efg&mask)==mask;
}

static int or_test(long*efg,long mask)
{
	return (*efg&mask)!=0;
}

static int and_rst(long*efg,long mask)
{
	if((*efg&mask)==mask)
	{
		*efg&=~mask;
		return 1;
	}
	return 0;
}

static int or_rst(long*efg,long mask)
{
	mask&=*efg;
	if(mask)
	{
		*efg&=~mask;
		return 1;
	}
	return 0;
}

static long test_or_efgs(int*result,long evtmask,long mask)
{
	int i;
	for(i=0;i<MAX_EFG;i++)
	{
		int event=(1<<i);
		if(evtmask&event)
		{
			if(or_rst(&efgs[i],mask))
			{
				*result|=event;
			}
		}
	}
	return *result;
}


asmlinkage int sys_cefg(unsigned func,unsigned efg,unsigned mask,int*result,unsigned long timeout)
{
	int rc=0;
	int(*tst_func)(long*,long)=0;
	// check parameters
	if(!result) return -EINVAL;
	if(efg>=MAX_EFG) return -EINVAL;	// invalid argument
	if(func>MAX_EFG_FUNC) return -EINVAL;

	// lock mutex
	down(&efgmutex);
	*result=0;	// assume no change
	switch((E_EFG_FUNC)(func))
	{
		case RSTEFG:
			{
				unsigned old=efgs[efg];
				efgs[efg]&=~mask;
				if(old!=efgs[efg]) *result=1;
				break;
			}
		case SETEFG:
			{
				unsigned old=efgs[efg];
				efgs[efg]|=mask;
				if(old!=efgs[efg])
				{
					*result=1;
					wake_up_interruptible_all(&efgwqueue);	/* transfer finished */
				}
				break;
			}
		case ANDTST:
			{
				if((efgs[efg]&mask)==mask)
					*result=1;
				break;
			}
		case ORTST:
			{
				if((efgs[efg]&mask)!=0)
					*result=1;
				break;
			}
		case CPYEFG:
			{
				*(long*)result=efgs[efg];
				break;
			}
		case WAITAND:
			{
				tst_func=and_test;
				goto try_or_wait;
			}
		case WAITOR:
			{
				tst_func=or_test;
				goto try_or_wait;
			}
		case WAITANDRST:
			{
				tst_func=and_rst;
				goto try_or_wait;
			}
		case WAITORRST:
			{
				tst_func=or_rst;
				goto try_or_wait;
			}
		case WAITOREFGS:
			{
				*result=test_or_efgs(result,efg,mask);
				if(*result)
				{
					break;	// all done
				}
				{
					struct timer_list timer;
					int timeint=0;
					int sts;
					up(&efgmutex);
					if(timeout)
					{
						timeout=(timeout+1000/HZ-1)/(1000/HZ);
						init_timer(&timer);
						timer.expires = jiffies + timeout;	// + 3 seconds
						timer.data = (unsigned long)&timeint;
						timer.function=&efg_timer;	/* timer handler */
						add_timer(&timer);
					}
					sts=wait_event_interruptible(efgwqueue,timeint||test_or_efgs(result,efg,mask));
					down(&efgmutex);	// will be upped at exit
					if(timeout)
					{
						if(timer_pending(&timer)) timeout=0;
						del_timer(&timer);
					}
					if(sts==ERESTARTSYS)	// durch signal beendet
						rc=-EINTR;
					//set_current_state(TASK_RUNNING);
					break;
				}
			}
		try_or_wait:
			{
				if(tst_func(&efgs[efg],mask))
				{
					*result=1;
					break;	// all done
				}
			}
		//must_wait:
			{
				struct timer_list timer;
				int timeint=0;
				int sts;
				up(&efgmutex);
				if(timeout)
				{
					timeout=(timeout+1000/HZ-1)/(1000/HZ);
					init_timer(&timer);
					timer.expires = jiffies + timeout;	// + 3 seconds
					timer.data = (unsigned long)&timeint;
					timer.function=&efg_timer;	/* timer handler */
					add_timer(&timer);
				}
				sts=wait_event_interruptible(efgwqueue,timeint||tst_func(&efgs[efg],mask));
				down(&efgmutex);	// will be upped at exit
				if(timeout)
				{
					if(timer_pending(&timer)) timeout=0;
					del_timer(&timer);
				}
				if(sts==ERESTARTSYS)	// durch signal beendet
					rc=-EINTR;
				else if(timeout)		// condition not fullfilled->timeout
				{
					*result=0;	// timeout
				}
				else
				{
					*result=1;	// success
				}
				//set_current_state(TASK_RUNNING);
				break;
			}
		default:
			rc=-EINVAL;
			break;
	}
	up (&efgmutex);
	return rc;
}

module_init(cefg_init);

