/***************************************************************************/

/*
 *	linux/arch/m68knommu/platform/5282/dma_timer.c
 *
 *	Sub-architcture dependant dma timer code for the Motorola
 *	5282 and 5270 CPU.
 *
 *	Copyright (C) 2005, Andreas Horn (andreas.horn@sartorius.com)
 */

/***************************************************************************/
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <asm/dma.h>
#include <asm/mcf528x_dma_timer.h>
#if defined(CONFIG_M532x)
/*
 * DMA timer channel base address
*/

struct S_MCF_DMA_TIMER* dma_timer[MAX_M68K_DMA_CHANNELS];

spinlock_t t_lock = SPIN_LOCK_UNLOCKED;

struct S_MCF_DMA_TIMER* request_hrt_timer(int *intr_request_vector)
{
	int i;
	spin_lock(t_lock);
	for(i=0;i<MAX_M68K_DMA_CHANNELS;i++)
	{
#if defined(CONFIG_SAG_PR5800_CORE)
		if(i==1) continue;	// reserved by keyboard click in bios
#endif
#if defined(CONFIG_SAG_LABPRO_CORE)
		if(i==3) continue;	// reserved by dma beep
#endif
		if(!dma_timer[i])
		{
			struct S_MCF_DMA_TIMER*timer=&MCF_DMA_TIMER(i);
			dma_timer[i] =timer;
			spin_unlock(t_lock);
			*intr_request_vector = MCFDTIM0_IRQ_INT+i;
			// one time initialisation of high resolution timer
			timer->dtmr=DTMR_PS(MCF_BUSCLK/(1000*1000))	// prescale at maximum speed
			                   |DTMR_CE_NONE	// capture no edge
			                   |DTMR_OM_0	// output mode pulse (dont care)
			                   |DTMR_ORRI_0	// disable interrupt
			                   |DTMR_FRR_0	// free run timer, dont reset at reference value
			                   |DTMR_CLK_DIV1	// clock is system/1
			                   |DTMR_RST_1;	// enable timer
			timer->dtxmr=DTXMR_DMAEN_0	// no dma request on reference match
			                    |DTXMR_MODE16_0;	// dont use 16bit mode
			MCF_INTC0_ICR(MCFDTIM0_IRQ_INT+i) = 4;
#if defined(CONFIG_M527x)
			MCF_INTC0_IMRL &= ~((1 << (MCFDTIM0_IRQ_INT+i)) | 1);
#endif
#if defined(CONFIG_M532x)
			MCF_INTC0_IMRH &= ~(1 << (MCFDTIM0_IRQ_INT+i-32)) ;
#endif
			return timer;
		}
	}
	spin_unlock(t_lock);
	printk(KERN_ERR "no more free dma timer\n");
	return 0;
}

void release_hrt_timer(struct S_MCF_DMA_TIMER* timer)
{
	int i;
	spin_lock(t_lock);
	for(i=0;i<MAX_M68K_DMA_CHANNELS;i++)
	{
		if(dma_timer[i] == timer)
		{
			dma_timer[i] =0;
			// disable interrupt
#if defined(CONFIG_M527x)
			MCF_INTC0_IMRL |=((1 << (pInfo->intr_request_vector)));
#endif
#if defined(CONFIG_M532x)
			MCF_INTC0_IMRH |=((1 << (MCFDTIM0_IRQ_INT+i-32)));
#endif
			spin_unlock(t_lock);
			return;
		}
	}
	spin_unlock(t_lock);
	printk(KERN_ERR "can't relase dma timer %8lx\n",(long)timer);
}

void reinit_hrt_timer(struct S_MCF_DMA_TIMER* timer)
{
	int i;
	for(i=0;i<MAX_M68K_DMA_CHANNELS;i++)
	{
		if(dma_timer[i] == timer)
		{
			timer->dtmr=DTMR_PS(MCF_BUSCLK/(1000*1000))	// prescale at maximum speed
						|DTMR_CE_NONE	// capture no edge
						|DTMR_OM_0	// output mode pulse (dont care)
						|DTMR_ORRI_0	// disable interrupt
						|DTMR_FRR_0	// free run timer, dont reset at reference value
						|DTMR_CLK_DIV1	// clock is system/1
						|DTMR_RST_1;	// enable timer
			timer->dtxmr=DTXMR_DMAEN_0		// no dma request on reference match
						|DTXMR_MODE16_0;	// dont use 16bit mode
			MCF_INTC0_ICR(i) = 4;
			return;
		}
	}
}

void start_hrt_timer(struct S_MCF_DMA_TIMER* timer,unsigned long delay)
{
	timer->dtrr=delay;
	timer->dtcn=0;			/* clear count register */
	timer->dter = 3;		/* clear pending interrupt */
	timer->dtmr|=DTMR_ORRI_1;	/* start timer */
}
void stop_hrt_timer(struct S_MCF_DMA_TIMER* timer)
{
	timer->dter = 3;		/* clear pending interrupt */
	timer->dtmr&=~DTMR_ORRI_1;	/* stop timer */
}

#endif

