/************************************************************************/
/*                                                                      */
/*  edgeport.c - utility routines for enable/disable interrupt		*/
/*  (C) Copyright 2004, Sartorius Hamburg GmbH                          */
/*                                                                      */
/************************************************************************/
#include <linux/interrupt.h>
#include <asm/coldfire.h>
#include <asm/mcfedgeport.h>

#if defined(CONFIG_M528x)||defined(CONFIG_M527x)
#include <asm/m528xsim.h>
#endif
#if defined(CONFIG_M532x)
#include <asm/m532xsim.h>
#endif
#define MCFINTC0_BASEVEC       64

/**
 * init a edgeport input as a extrnal interrupt line, request the interrupt.
 * @param port  number of edgeport
 * @param mode  0 = level triggered, 1 rising edge, 2 falling edge,3 both
 *               falling and rising edge triggered
 * @return -1 if somthing goes wrong else 0
*/
const char *const edgeport[8]=
{"reserved",
"edge port 1",
"edge port 2",
"edge port 3",
"edge port 4",
"edge port 5",
"edge port 6",
"edge port 7"};

int enable_edgeport(int port,int mode,irqreturn_t(*intr)(int,void*,struct pt_regs*))
{
	unsigned mask;
// 	volatile unsigned long *imrp;
	if(port<1 || port >7) return -1;
	if(mode<0 || mode >3) return -1;

	if(request_irq(MCFINTC0_BASEVEC+port, intr, SA_INTERRUPT,
		edgeport[port], NULL) < 0 ) {
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,port+MCFINTC0_BASEVEC);
// 		return -1;
	}

	// enable interrupt inside interrupt controller
	MCF_INTC0_CIMR=port;
	MCF_INTC0_ICR(port)=port;	// set int level like the port

	/* set the edgeport level detection mode */
	mask = MCF_EPORT_EPPAR;
	mask &= ~(3<<(2*port));
	mask |= ((mode)<<(2*port));
	MCF_EPORT_EPPAR = mask;

	/* enable edgeport interrupt */
	MCF_EPORT_EPIER |=(1<<port);	// enable interrupt
	return 0;
}

int enable_edgeport_dev(int port,int mode,irqreturn_t(*intr)(int,void*,struct pt_regs*),void * dev_id)
{
	unsigned mask;
// 	volatile unsigned long *imrp;
	if(port<1 || port >7) return -1;
	if(mode<0 || mode >3) return -1;

	if(request_irq(MCFINTC0_BASEVEC+port, intr, SA_INTERRUPT,
	   edgeport[port], dev_id) < 0 ) {
		   printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			   __FILE__,__LINE__,port+MCFINTC0_BASEVEC);
// 		return -1;
	   }
	// enable interrupt inside interrupt controller
	MCF_INTC0_CIMR=port;

	MCF_INTC0_ICR(port)=port;	// set int level like the port
	   /* set the edgeport level detection mode */
	   mask = MCF_EPORT_EPPAR;
	   mask &= ~(3<<(2*port));
	   mask |= ((mode)<<(2*port));
	   MCF_EPORT_EPPAR = mask;

	   /* enable edgeport interrupt */
	   MCF_EPORT_EPIER |=(1<<port);	// enable interrupt
	   return 0;
}

/**
 *  disable the interrupt on the edgeport, free the irq.
*/
void disable_edgeport(int port)
{
	if(port<1 || port >7) return ;
	// disable interrupt
	MCF_INTC0_SIMR = port;
	MCF_EPORT_EPIER &=~(1<<port);	// disable interrupt
	free_irq(MCFINTC0_BASEVEC+port,0);
}


