/***************************************************************************
 * usb-mcf532x.c - Platform level (mcf532x) USB initialization.
 *
 * Andrey Butok Andrey.Butok@freescale.com.
 * Copyright Freescale Semiconductor, Inc 2006
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ***************************************************************************
 * Changes:
 *   v0.02	12 July 2006	Andrey Butok
 *		UDC Device registration added.
 *   v0.01	31 March 2006	Andrey Butok
 *   		Initial Release - developed on uClinux with 2.6.15.6 kernel.
 *
 */

#if defined(CONFIG_USB_GADGET_MCF532x) || defined(CONFIG_USB_EHCI_HCD)

#undef	DEBUG

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/usb_otg.h>
#include <asm/coldfire.h>

/* Start address of HC registers.*/
#define MCF532x_USB_HOST_REG_START		(0xfc0b4000)
/* End address of HC registers */
#define MCF532x_USB_HOST_REG_END		(MCF532x_USB_HOST_REG_START+0x1a8)
/* USB Host Interrupt number */
#define MCF532x_USB_HOST_INT_NUMBER		(128+48)

/* Start address of OTG (Host/Device) module registers.*/
#define MCF532x_USB_OTG_HOST_REG_START		(0xfc0b0000)
/* End address of OTG (Host/Device) module registers */
#define MCF532x_USB_OTG_HOST_REG_END		(MCF532x_USB_OTG_HOST_REG_START+0x1a8)

#ifdef	CONFIG_USB_GADGET_MCF532x
/* Start address of OTG (Device) module registers.*/
#define MCF532x_USB_OTG_DEVICE_REG_START	(0xfc0b01ac)
/* End address of OTG (Device) module registers */
#define MCF532x_USB_OTG_DEVICE_REG_END		(MCF532x_USB_OTG_DEVICE_REG_START+0x20)
#endif

/* USB OTG (Host/Device) Interrupt number */
#define MCF532x_USB_OTG_INT_NUMBER		(128+47)


static struct otg_transceiver *transceiver;

/**
 * otg_get_transceiver - find the (single) transceiver driver
 *
 * Returns the transceiver driver, after getting a refcount to it; or
 * null if there is no such transceiver.  The caller is responsible for
 * releasing that count.
 */
struct otg_transceiver *otg_get_transceiver(void)
{
	if (transceiver)
		get_device(transceiver->dev);
	return transceiver;
}
EXPORT_SYMBOL(otg_get_transceiver);

int otg_set_transceiver(struct otg_transceiver *x)
{
	if (transceiver && x)
		return -EBUSY;
	transceiver = x;
	return 0;
}
EXPORT_SYMBOL(otg_set_transceiver);



/*-------------------------------------------------------------------------*/

static void
usb_release(struct device *dev)
{
	/* normally not freed */
}

/*
 * USB Host module structures
 */
static struct resource ehci_host_resources[] = {
	{	/* registers */
		.start = MCF532x_USB_HOST_REG_START,
		.end = MCF532x_USB_HOST_REG_END,
		.flags = IORESOURCE_MEM,
	}, {	/* IRQ */
		.start = MCF532x_USB_HOST_INT_NUMBER,
		.flags = IORESOURCE_IRQ,
	}
};

static struct platform_device ehci_host_device = {
	.name = "ehci",
	.id = 1,
	.dev = {
		.release = usb_release,
		.dma_mask = 0x0},
	.num_resources = ARRAY_SIZE(ehci_host_resources),
	.resource = ehci_host_resources,
};

/*
 * USB OTG module structures.
 */
static struct resource ehci_otg_resources[] = {
	{	/* registers */
		.start = MCF532x_USB_OTG_HOST_REG_START,
		.end = MCF532x_USB_OTG_HOST_REG_END,
		.flags = IORESOURCE_MEM
	}, {	/* IRQ */
		.start = MCF532x_USB_OTG_INT_NUMBER,
		.flags = IORESOURCE_IRQ,
	},
};

static struct platform_device ehci_otg_device = {
	.name = "ehci",
	.id = 0,
	.dev = {
		.release = usb_release,
		.dma_mask = 0x0},
	.num_resources = ARRAY_SIZE(ehci_otg_resources),
	.resource = ehci_otg_resources,
};

#ifdef	CONFIG_USB_GADGET_MCF532x
static struct resource udc_otg_resources[] = {
	{	/* registers */
		.start	= MCF532x_USB_OTG_DEVICE_REG_START,
		.end	= MCF532x_USB_OTG_DEVICE_REG_END,
		.flags	= IORESOURCE_MEM,
	}, {	/* IRQ */
		.start	= MCF532x_USB_OTG_INT_NUMBER,
		.flags	= IORESOURCE_IRQ,
	}
};

static struct platform_device udc_otg_device = {
	.name		= "mcf532x_udc",
	.id		= -1,
	.dev = {
		.release = usb_release,
		.dma_mask = 0x0},
	.num_resources	= ARRAY_SIZE(udc_otg_resources),
	.resource	= udc_otg_resources,
};
#endif

#if defined(CONFIG_ULPI_M532x)
static struct platform_device usb_ulpi_device = {
	.name		  = "mcf532x-ulpi",
	.id		  = -1
};
#endif

static int __init
mcf532x_usb_init(void)
{
	int status;

	/*
	 * Initialize the clock divider for the USB:
	 */
#if MCF_CLK == 240000000
	/*
	 * CPU oerating on 240Mhz (MISCCR[USBDIV]=1)
	 */
	(*(volatile u16 *) (0xFC0A0010)) |= (0x0002);
#elif MCF_CLK == 180000000
	/*
	 * CPU oerating on 180Mhz (MISCCR[USBDIV]=0)
	 */
	(*(volatile u16 *) (0xFC0A0010)) &= ~(0x0002);
#else
#error "Processor clock must be 240MHz or 180MHz for MCF532x to use USB."
#endif

#ifdef CONFIG_USB_EHCI_HCD
	/*
	 * Register USB (HOST module) EHCI device:
	 */
	status = platform_device_register(&ehci_host_device);
	if (status) {
		pr_info
		    ("USB-MCF532x: Can't register (HOST module) EHCI device, %d\n",
		     status);
		return -ENODEV;
	}
	pr_info("USB-MCF532x: (HOST module) EHCI device is registered\n");
#endif

#if defined(CONFIG_USB_EHCI_HCD) && !defined(CONFIG_USB_GADGET_MCF532x)
	/*
	 * Register USB (OTG module) EHCI device:
	 */
	status = platform_device_register(&ehci_otg_device);
	if (status) {
		pr_info
		    ("USB-MCF532x: Can't register (OTG module) EHCI device, %d\n",
		     status);
		return -ENODEV;
	}
	pr_info("USB-MCF532x: (OTG module) EHCI device is registered\n");
#endif

#ifdef	CONFIG_USB_GADGET_MCF532x
	/*
	 * Register USB (OTG module) UDC device:
	 */
	status = platform_device_register(&udc_otg_device);
	if (status) {
		pr_info
		    ("USB-MCF532x: Can't register (OTG module) UDC device, %d\n",
		     status);
		return -ENODEV;
	}
	pr_info("USB-MCF532x: (OTG module) UDC device is registered\n");
#endif

#if defined(CONFIG_ULPI_M532x)
	/*
	 * Register USB (OTG module) ULPI device:
	 */
	status = platform_device_register(&usb_ulpi_device);
	if (status) {
		pr_info
		    ("USB-MCF532x: Can't register (OTG module) ULPI device, %d\n",
		     status);
		return -ENODEV;
	}
	pr_info("USB-MCF532x: (OTG module) ULPI device is registered\n");
#endif

	return 0;
}

subsys_initcall(mcf532x_usb_init);

#endif
