/************************************************************************/
/*                                                                      */
/*  hms.c - driver for HMS fieldbus cards                               */
/*                                                                      */
/*  (C) Copyright 2004, Sartorius Hamburg GmbH                          */
/*                                                                      */
/************************************************************************/
#include <linux/module.h>
#include <linux/version.h>
#include <asm/delay.h>
#include <asm/irq.h>
#include <asm/mcfedgeport.h>

#include <linux/fs.h>
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <asm/uaccess.h> /* for put_user */
#include <linux/init.h>  
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/pm.h>

#include <asm/coldfire.h>
#include <linux/config.h>
#include "sag_hms.h"
#include <asm/sag_hms.h>

#if defined(CONFIG_M527x)
#include <asm/m528xsim.h>   
#endif

/* for use with Sartorius boards */
#define MCFINTC0_BASEVEC       64

#define DEVICE_NAME "hms"
static DECLARE_MUTEX(sem);

/* struct wait_queue *wqueue;   */
static DECLARE_WAIT_QUEUE_HEAD(wqueue);  

local_data status;
typedef volatile unsigned int vu32;
typedef volatile unsigned long vul32;
typedef volatile unsigned char vuc32;

int __init hms_init(void);
void __exit hms_exit(void);


static int hms_state;
static int is_open=0;
#if defined(CONFIG_SAG_PR5220_CORE)
#define HMS_RESET_PORT (*(vuc32*)0x40100001)
#define HMS_RESET_BIT 0x04
#endif

#define MOVEB(d,s) asm volatile(" move.b (%0)+,(%1)+;nop;"\
		: "=a"(s),"=a"(d) \
		:  "0"(s), "1"(d) : "memory");

#define MOVEIB(d,s) asm volatile(" move.b %1,(%0)+;nop;"\
		: "=a"(d) \
		:  "d"(s), "0"(d) : "memory");

void dpram_copy(volatile char*dst,const volatile char*src,long size)
{
	while(size>=16)
	{
		// move 16 words=32bytes
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		size-=16;
	}
	while(size>0)
	{
		MOVEB(dst,src);
		size-=1;
	}
}
void dpram_init(volatile char*dst,const char val,long size)
{
	while(size>=16)
	{
		// move 16 words=32bytes
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		size-=16;
	}
	while(size>0)
	{
		MOVEIB(dst,val);
		size-=1;
	}
}


static int hms_detect(void)
{
	//----------------------------
	// check slot 4 (3)
	//----------------------------
	unsigned char tmp[4];
	int bit;
	int n;
	vuc32 * dpram=(vuc32 *)(DPRAM_BASE+0x3800);
	status.status_reg = 0;
	status.found = 0;
	//--------------------------------------
	// pull reset low to get card inactive
	//--------------------------------------
	HMS_RESET_PORT &= ~HMS_RESET_BIT;
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(1);	// wait one HZ tick (5ms)
	// check if a card is inserted
	dpram_copy(dpram,"\x55\xAA\x55\xAA",4);
	dpram_copy(tmp,dpram,4);
	tmp[0]=~tmp[0];
	tmp[1]=~tmp[1];
	tmp[2]=~tmp[2];
	tmp[3]=~tmp[3];
	dpram_copy(dpram,tmp,4);
	dpram_copy(tmp,dpram,4);
	if(tmp[0]==0xAA && tmp[1]==0x55 && tmp[2]==0xAA && tmp[3]==0x55)
	{
		//---------------------------------------------
		// we have detected a dual ported ram
		//---------------------------------------------
	       	HMS_RESET_PORT|=HMS_RESET_BIT;		// reset high -> card will start
		bit=1;	// assume no reaction
		// wait max 1.5 seconds for card ready
		set_current_state(TASK_UNINTERRUPTIBLE);
		for(n=0; n<1500/*ms*/; n+=1000/HZ)
		{
			schedule_timeout(1);	// wait one HZ tick (5ms)
		}
		set_current_state(TASK_RUNNING);
		status.status_reg = *(dpram+HMS_STATUS_REG);
		status.found = 1;
		return 1;
	}
	return 0;
}



static int hms_ioctl(struct inode *inode, struct file *file,
				unsigned int cmd, unsigned long arg)
{
	int rc = 0;
	unsigned char *oprocessdata,*iprocessdata;
	unsigned long opsiz,ipsiz;
	unsigned char *omsgdata,*imsgdata;
	unsigned long omsgsiz,imsgsiz;
	unsigned long	flag;
	unsigned char ctrl_bits=0;
	unsigned io_flags;
	int __user *result=(int __user*)arg;
	hms_para *para=(hms_para*)arg; 
	vuc32 *dpram=(vuc32 *)DPRAM_BASE+0x3800;
	// if a powerfail was detected, the process must reinitialize this device
	get_user(io_flags,&para->status);
	io_flags=0;
	if(hms_state)
	{
		hms_state = 0;
		return -EINTR;
	}
	down(&sem);
	switch(cmd)
	{
        	case DATA_IO:
	        	// if no card is inserted
			if(status.found == 0) {
				rc =-ENXIO;
				goto error;
			}
			get_user(oprocessdata,&para->odat);
			get_user(opsiz,&para->osiz);
			if(oprocessdata)
                     	{
				dpram_copy ((void*)dpram,oprocessdata,opsiz);
				save_flags(flag); cli();
				status.control_reg = ((status.status_reg&SR_T)^SR_T)|CR_R;
				dpram[HMS_CONTROL_REG] = status.control_reg;
				interruptible_sleep_on_timeout(&wqueue,(HZ / 10));
				restore_flags(flag);
				if (signal_pending(current) || !status.initialized) {
					rc = -EINTR;
					goto error;
				}
                        }
			get_user(iprocessdata,&para->idat);
			get_user(ipsiz,&para->isiz);
			if(iprocessdata)
                   	{
				copy_to_user(iprocessdata,(void*)dpram+0x200,ipsiz);
			}
           		break;
         	case MBX_IO:
	        	// if no card is inserted
			if(status.found == 0) {
				rc =- ENXIO;
				goto error;
			}
			/* write process data */
			get_user(oprocessdata,&para->odat);
			get_user(opsiz,&para->osiz);
			if(oprocessdata)
				dpram_copy ((void*)dpram,oprocessdata,opsiz);
			/* write command to message buffer */
			get_user(omsgdata,&para->m_write);
			get_user(omsgsiz,&para->m_wr_size);
			if(omsgdata)
                     	{
				if(!(status.status_reg&SR_R))
					io_flags |= COULD_NOT_WRITE_MESSAGE;
				else
				{
					dpram_copy ((void*)dpram+0x3b00,omsgdata,omsgsiz);
					ctrl_bits = CR_M;
					// if this message is a command response or error response to a anybus message
					// set the R_BIT to get more messages from anybus
					if(!(omsgdata[4] & 0x40)|| (omsgdata[4] & 0x80))
						status.r_bit = CR_R;
				}
			}
			// toggle T-Bit
			save_flags(flag); cli();
			status.control_reg = ((status.status_reg&SR_T)^SR_T)|status.r_bit|ctrl_bits;
			dpram[HMS_CONTROL_REG] = status.control_reg;
			interruptible_sleep_on(&wqueue);
			restore_flags(flag);
			if (signal_pending(current)) {
				rc = -EINTR;
				goto error;
			}
			/* read from message buffer if available */
			get_user(imsgdata,&para->m_read);
			get_user(imsgsiz,&para->m_rd_size);
			if(imsgdata && (status.status_reg&SR_M))
                   	{
				copy_to_user (imsgdata,(void*)dpram+0x3d00,imsgsiz);
				io_flags |= HAVE_CLIENTMESSAGE;
				// clear our R_BIT to prevent further receiving of commands from anybus module
				if(imsgdata[4] & 0x40)
					status.r_bit = 0;
			}
			/* read processdata */
			get_user(iprocessdata,&para->idat);
			get_user(ipsiz,&para->isiz);
			if(iprocessdata)
			{
				copy_to_user(iprocessdata,(void*)dpram+0x100,ipsiz);
			}
			io_flags |= (status.status_reg&0x07)<<8;
			put_user(io_flags,&para->status);
			break;
               case HMS_RESET:
			rc = hms_detect();
			if(!rc) rc=-ENXIO;	// no card
			status.r_bit = CR_R;
			status.control_reg = ((status.status_reg&SR_T)^SR_T)|status.r_bit;
			save_flags(flag); cli();
			dpram[HMS_CONTROL_REG] = status.control_reg;
			interruptible_sleep_on(&wqueue);
			restore_flags(flag);
			
			if(status.status_reg&SR_R)
			{
				put_user(1,result);
				status.initialized=1;
			}
			else
				put_user(0,result);
			break;
		default:
			rc = -EINVAL;
			break;
   	}

error:	    
	up (&sem);
	return rc;
}

static int hms_release(struct inode *inode, struct file *file)
{
	disable_edgeport(MCFHMS_IRQ_INT);
	//--------------------------------------
	// pull reset low to get card inactive
	//--------------------------------------
       	HMS_RESET_PORT &= ~HMS_RESET_BIT;
	status.initialized=0;
	is_open = 0;
	return 0;
}
irqreturn_t hms_intr(int irq, void *dummy, struct pt_regs *fp)
{
        vul32 *dpram=(vul32 *)DPRAM_BASE;
	*(vuc32 *)(MCF_MBAR + MCF_EPFR) = 1<<(irq-64); /* reset interrupt */
	status.status_reg = dpram[0x3800+HMS_STATUS_REG];	// get new status
	wake_up(&wqueue);	/* transfer finished */
	return IRQ_HANDLED;
}


static int hms_open(struct inode *inode, struct file *file)
{
	int sts;
	if(is_open) return -EBUSY;
	is_open = 1;
	file->private_data = 0;
	sts = hms_detect();
	if(sts ==0) return -ENODEV;
	if(enable_edgeport(MCFHMS_IRQ_INT,FALLING_MODE,hms_intr)<0)
	{
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
		__FILE__,__LINE__,64+MCFHMS_IRQ_INT);
		return 0;
	}
	return 0;
}

#ifdef CONFIG_PM
static struct pm_dev *hms_pm_device;
static int hms_pm_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	switch (request) {
	case PM_SUSPEND:
		status.initialized=0;
		disable_edgeport(4);  /* disable hms interrupt */
		break;

	case PM_RESUME:
		// enable interrupt
		wake_up(&wqueue);	/* transfer finished */
		enable_edgeport(4,FALLING_MODE,hms_intr);
		break;
	}
	return 0;
}
#endif

static struct file_operations Fops = {
	owner:		THIS_MODULE,
	ioctl:		hms_ioctl,
	open:		hms_open,
	release:	hms_release  /* a.k.a. close */
};


/* init for compiled-in driver */
int __init hms_init()
{
        printk ("Sartorius HMS Anybus-CC device driver $Revision: 1.1 $ installed\n");
        if (register_chrdev(HMS_MAJOR, DEVICE_NAME, &Fops)) {
        	printk(KERN_NOTICE "Can't allocate major number %d for HMS devices.\n",
        	       HMS_MAJOR);
        	return -EAGAIN;;
        }
	init_waitqueue_head(&wqueue);
#ifdef CONFIG_PM
	hms_state =0;
	status.initialized=0;
	hms_pm_device = pm_register(PM_SYS_DEV, PM_SYS_COM, hms_pm_callback);
#endif
	return 0;
}

/* Cleanup - undid whatever init_module did */
void __exit hms_exit()
{
	/* Unregister the device */
	if ((unregister_chrdev(HMS_MAJOR, DEVICE_NAME)) < 0)
		printk("Error in unregister_chrdev\n");
#ifdef CONFIG_PM
	if(hms_pm_device)
	{
		pm_unregister(hms_pm_device);
		hms_pm_device =0;
	}
#endif
}

module_init(hms_init);
module_exit(hms_exit);    

MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("hms");
MODULE_DESCRIPTION("Direct character-device access to HMS devices");
