/***************************************************************************
                          sag_asycom.c  -  description
                             -------------------
    begin                : Mo Dec 14 2005
    copyright            : (C) 2005 by Sartorius Hamburg GmbH
    email                : andreas.horn@sartorius.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/string.h>   /* used in new tty drivers */
#include <linux/ioctl.h>
#include <linux/sag_asycom.h>
#include <asm/mcf528x_dma_timer.h>

//#define DEBUG

#ifdef DEBUG
#define TRACE_L(format, args...) printk("xbpi: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif


static int  asycom_open(struct tty_struct *tty);
static void asycom_close(struct tty_struct *tty);
static ssize_t asycom_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr);
static ssize_t asycom_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr);
static int asycom_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void asycom_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void asycom_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char fp);
static int  asycom_receive_room(struct tty_struct *tty);
static void asycom_tty_wakeup(struct tty_struct *tty);

static void wait_soh(void *pInfo, const unsigned char c,unsigned char flags);
static void wait_addr(void *pInfo, const unsigned char c,unsigned char flags);
static void wait_poll(void*pInfo, const unsigned char c,unsigned char flags);
static void wait_enq(void*pInfo, const unsigned char c,unsigned char flags);
static void wait_abort(void*pInfo, const unsigned char c,unsigned char flags);
static void rx_message(void *pInfo, const unsigned char c,unsigned char flags);
static void wait_bcc(void *p, const unsigned char c,unsigned char flags);
static void wait_rxenq(void *p, const unsigned char c,unsigned char flags);

int __init asycom_init(void);
void __exit asycom_exit(void);

static struct tty_ldisc tty_ldisc_N_ASYCOM =
    {
    .owner	 = THIS_MODULE,
    .magic	 = TTY_LDISC_MAGIC,
    .name	 = "ASYCOM",
    .open	 = asycom_open,
    .close	 = asycom_close,
    .read	 = asycom_read,
    .write	 = asycom_write,
    .ioctl	 = asycom_ioctl,
    .receive_buf = asycom_receive_buf,
    .receive_room= asycom_receive_room,
    .write_wakeup= asycom_tty_wakeup,
    .single_char_received=asycom_single_char_received
    };


static int  asycom_open(struct tty_struct *tty)
{
	struct asycom_info * pInfo;
	pInfo=kmalloc(sizeof(struct asycom_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "asycom: failed to alloc info structure\n");
		return -ENOMEM;
	}
	memset(pInfo,0,sizeof(struct asycom_info));
	pInfo->tx_buf = kmalloc(TX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->tx_buf)
	{
		printk(KERN_ERR "asycom: failed to alloc transmit buffer\n");
		kfree(pInfo);
		return -ENOMEM;
	}
	pInfo->write_state = TRANSMITT_BUFFER_EMPTY;
	pInfo->read_state = NO_MESSAGE;
	tty->disc_data = pInfo;
	pInfo->tty = tty;
	pInfo->nextchar = wait_soh;		/* the idle state */
	init_waitqueue_head(&pInfo->read_wait);
	init_waitqueue_head(&pInfo->write_wait);
	return 0;
}


static void asycom_close(struct tty_struct *tty)
{
	struct asycom_info *pInfo=(struct asycom_info*)tty->disc_data;
	if(pInfo->rx_buf) kfree(pInfo->rx_buf);
	kfree(pInfo->tx_buf);
	kfree(pInfo);
}

static ssize_t asycom_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr)
{
	int len;
	struct asycom_info * pInfo;
	pInfo = tty->disc_data;
	if(!pInfo) return -ENOENT;

	/* check for a valid input buffer */
	while(!pInfo->rx_buf[pInfo->last_rxbuf].in_use)
	{
		wait_event_interruptible(pInfo->read_wait,pInfo->read_state==RECEIVE_MESSAGE);
		if (signal_pending(current))
			return -EINTR;
	}
	len = pInfo->rx_buf[pInfo->last_rxbuf].rx_len;
	if(nr < len)
		return -EIO;
        copy_to_user(buf,pInfo->rx_buf[pInfo->last_rxbuf].rx_buf,len);
	pInfo->rx_buf[pInfo->last_rxbuf].in_use = 0;
	if(++pInfo->last_rxbuf == pInfo->nr_rxbuf)
		pInfo->last_rxbuf = 0;
	if(!pInfo->rx_buf[pInfo->last_rxbuf].in_use)
		pInfo->read_state = NO_MESSAGE;
	return len;
}

static ssize_t asycom_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr)
{
	int i=0;
	unsigned char c;
	unsigned char *ptxbuf;
	unsigned char tx_bcc;
	struct asycom_info * pInfo;
	pInfo = tty->disc_data;

	if(nr>TX_BUF_SIZE-5)
		return -EINVAL;
	while(pInfo->tx_busy) {
		wait_event_interruptible(pInfo->write_wait,pInfo->write_state==TRANSMITT_BUFFER_EMPTY);
		if (signal_pending(current))
			return -EINTR;
	}
	/* build the complete send string */
	ptxbuf = pInfo->tx_buf;
	*ptxbuf++=SOH;
	*ptxbuf++=pInfo->slv_adr;
	pInfo->tx_len=2;
	*ptxbuf++=STX;
	pInfo->tx_len++;
	tx_bcc=ETX;
	for(i=0;i<nr;i++)
	{
		get_user(c,&buf[i]);
		if(c==EOT||
			c==DLE||
			c==ETX||
			c==SOH)
		{
			*ptxbuf++=DLE;
			pInfo->tx_len++;
			tx_bcc^=DLE;
			if(pInfo->tx_len==TX_BUF_SIZE-2)
				return -E2BIG;
		}
		*ptxbuf++=c;
		pInfo->tx_len++;
		tx_bcc^=c;
		if(pInfo->tx_len==TX_BUF_SIZE-2)
			return -E2BIG;
	}
	*ptxbuf++=ETX;
	*ptxbuf++=tx_bcc;
	pInfo->tx_len+=2;
	pInfo->tx_busy=1;	/* buffer is valid */
	pInfo->write_state=TX_TRANSMITT;
	return nr;
}

static void put_char(struct asycom_info *pInfo, unsigned char ch)
{
	struct tty_struct *tty = pInfo->tty;

	if(tty==NULL)
		return;

	if(tty->driver->put_char)
	{
		tty->driver->put_char(tty, ch);
	}
}

/**
 * asycom_tty_wakeup - Callback for transmit wakeup
 * @tty	- pointer to associated tty instance data
 *
 * Called when low level device driver can accept more send data.
 */
static void asycom_tty_wakeup(struct tty_struct *tty)
{
	struct asycom_info *pInfo=(struct asycom_info*)tty->disc_data;

	if (!pInfo)
		return;
	if(!pInfo->tx_busy)
		return;			/* nothing more to send */
	if(pInfo->tty->stopped)
		return;

	if(pInfo->write_state==TX_TRANSMITT){
		if(pInfo->tx_write_idx < pInfo->tx_len)
			put_char(pInfo,pInfo->tx_buf[pInfo->tx_write_idx++]);
	}
}

static int asycom_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	int ret =0;
	int i=0;
	unsigned char c;
	unsigned char *ptxbuf;
	unsigned char tx_bcc;
	struct ASYCOM_PARA *parg=(struct ASYCOM_PARA *)arg;
	struct asycom_info *pInfo=(struct asycom_info*)tty->disc_data;
	struct ASYCOM_PARA para;
	struct S_MCF_DMA_TIMER *dmatimer;
	if(cmd <ASYCOM_SET_SLVADDR || cmd> ASYCOM_IO_TELEGRAMM)
		return  n_tty_ioctl (tty, file, cmd, arg);

	copy_from_user((void*)&para,(void*)arg,sizeof(struct ASYCOM_PARA));


	if( para.isiz>255
	 || para.osiz>255
	 || (para.ibuf && para.isiz==0)	// fehler: input buffer ohne groesse
	 || (para.obuf && para.osiz==0)	// fehler: output buffer ohne groesse
	)
	{
		put_user(IO_PARA_ERR,&parg->status);
		return -EINVAL;
	}

	dmatimer = pInfo->timer;
	switch(cmd)
	{
		/* set own slave address and create receive buffers.
		** The 1. byte from each receive buffer is the busy in use flag */
		case ASYCOM_SET_SLVADDR:
			if(para.nbuf==0)
				return -EINVAL;
			pInfo->rx_buf = kmalloc(sizeof(struct RX_BUF)*para.nbuf, GFP_KERNEL);
			if(!pInfo->rx_buf)
			{
				printk(KERN_ERR "asycom: failed to alloc receive buffer\n");
				return -ENOMEM;
			}
			memset(pInfo->rx_buf,0,sizeof(struct RX_BUF)*para.nbuf);
			pInfo->nr_rxbuf = para.nbuf;
			pInfo->slv_adr = para.id ; 		/* set the slave id */
			pInfo->curr_rxbuf=0;			/* start with rx buffer */
			pInfo->read_state = NO_MESSAGE;
			break;
		case ASYCOM_IO_TELEGRAMM:
			if(para.osiz)
			{
				while(pInfo->tx_busy) {
					wait_event_interruptible(pInfo->write_wait,pInfo->write_state==TRANSMITT_BUFFER_EMPTY);
					if (signal_pending(current))
						return -EINTR;
				}
				/* build the complete send string */
				ptxbuf = pInfo->tx_buf;
				*ptxbuf++=SOH;
				*ptxbuf++=pInfo->slv_adr;
				pInfo->tx_len=2;
				*ptxbuf++=STX;
				pInfo->tx_len++;
				tx_bcc=ETX;
				for(i=0;i<para.osiz;i++)
				{
					get_user(c,(unsigned char*)para.obuf+i);
					if(c==EOT||
					   c==DLE||
					   c==ETX||
					   c==SOH)
					{
						*ptxbuf++=DLE;
						pInfo->tx_len++;
						tx_bcc^=DLE;
						if(pInfo->tx_len==TX_BUF_SIZE-2)
							return -E2BIG;
					}
					*ptxbuf++=c;
					pInfo->tx_len++;
					tx_bcc^=c;
					if(pInfo->tx_len==TX_BUF_SIZE-2)
						return -E2BIG;
				}
				*ptxbuf++=ETX;
				*ptxbuf++=tx_bcc;
				pInfo->tx_len+=2;
				pInfo->tx_busy=1;	/* buffer is valid */
				pInfo->write_state=TX_TRANSMITT;
			}
			if(para.isiz)
			{
			}
			break;
	}
	return ret;
}


static void wait_soh(void *pInfo, const unsigned char c,unsigned char flags)
{

	if(flags != TTY_NORMAL) return;     /* stay in this state if parity error */
	if(c != SOH) return;
	if(c == EOT)
	{
		// FIXME must stop transmission !!! reset communication

		return;
	}
	((struct asycom_info*)pInfo)->nextchar=wait_addr;
}

static void wait_addr(void *pInfo, const unsigned char c,unsigned char flags)
{
	((struct asycom_info*)pInfo)->nextchar=wait_soh;
	if(flags != TTY_NORMAL ||(c < 'A' || c >'Z'))
	{
		return;
	}
	if(((struct asycom_info*)pInfo)->slv_adr == c)
		((struct asycom_info*)pInfo)->nextchar=wait_poll;
}
static void wait_poll(void *p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	if(flags != TTY_NORMAL ||(c !=POL && c != STX))
	{
		pInfo->nextchar=wait_soh;
		return;
	}
	if(c==POL)
		pInfo->nextchar=wait_enq;
	else if(c==STX)
	{
		/* check for free input buffer */
		if(pInfo->rx_buf[pInfo->curr_rxbuf].in_use)
			pInfo->rx_answer=EOT;
		else
			pInfo->rx_answer=0;
		pInfo->rx_position=pInfo->rx_buf[pInfo->curr_rxbuf].rx_buf;
		pInfo->rx_len=0;
		pInfo->rx_bcc=ETX;
		pInfo->rx_dle=0;
		pInfo->nextchar=rx_message;
	}
}
/* have SOH POL wait for ENQ*/
static void wait_enq(void*p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	if(flags != TTY_NORMAL ||(c !=ENQ))
	{
		pInfo->nextchar=wait_soh;
		return;
	}
	if(pInfo->tx_busy)
	{
		/* start transmission */
		pInfo->nextchar=wait_abort;
		pInfo->tx_write_idx=0;
		pInfo->tty->flags |= (1<<TTY_DO_WRITE_WAKEUP);
		put_char(pInfo,pInfo->tx_buf[pInfo->tx_write_idx++]);
	}
	else
	{
		/* nothing to send quit with EOT */
		put_char(pInfo,EOT);
		pInfo->nextchar=wait_soh;
	}
}
// while sending a message wait for a EOT for break
static void wait_abort(void*p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	struct tty_struct *tty = pInfo->tty;
	if(flags != TTY_NORMAL ||c ==EOT)
	{
		/* stop transmission */
		tty->hw_stopped = 1;
		tty->driver->flush_buffer(tty);
		pInfo->nextchar=wait_soh;
		tty->flags &= ~(1<<TTY_DO_WRITE_WAKEUP);
		return;
	}
	if(c==ACK||c==NAK)
	{
		tty->flags &= ~(1<<TTY_DO_WRITE_WAKEUP);
		pInfo->nextchar=wait_soh;
		if(c==ACK){
			/* free tx buffer for reuse */
			pInfo->write_state = TRANSMITT_BUFFER_EMPTY;
			pInfo->tx_busy=0;
			wake_up_interruptible(&pInfo->write_wait);	/* transfer finished */
		}
	}
}
/**
**  receive a command from host. If rx_answer is not 0 just wait until
**  the complete command is sent, than answer with EOT
*/
static void rx_message(void *p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	if(!pInfo->rx_dle)
	{
		switch(c)
		{
			/* end of message from host */
			case ETX:
				pInfo->nextchar=wait_bcc;
				return;
			/* asyncron break from host */
			case EOT:
				pInfo->nextchar=wait_soh;
				return;
			/* skip the first DLE, wait for next char */
			case DLE:
				pInfo->rx_dle=1;
				pInfo->rx_bcc ^=DLE;
				return;
		}
	}
	pInfo->rx_dle=0;
	if(pInfo->rx_answer) return; /* ignore char */
	pInfo->rx_position[pInfo->rx_len]=c;
	pInfo->rx_len++;
	pInfo->rx_bcc ^=c;
	if(pInfo->rx_len == RX_BUF_SIZE)
		pInfo->rx_answer = EOT;		/* buffer overflow */
}

static void wait_bcc(void *p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	if(pInfo->rx_answer)
	{
		pInfo->nextchar=wait_rxenq;
		return;
	}
	if(flags != TTY_NORMAL || pInfo->rx_bcc != c)
		pInfo->rx_answer = NAK;
	else
		pInfo->rx_answer = ACK;
	pInfo->nextchar=wait_rxenq;
}
static void wait_rxenq(void *p, const unsigned char c,unsigned char flags)
{
	struct asycom_info*pInfo = (struct asycom_info*)p;
	if(pInfo->rx_answer==EOT)
		put_char(pInfo,pInfo->rx_answer);
	else if(flags != TTY_NORMAL || c != ENQ)
		put_char(pInfo,NAK);
	else
	{
		put_char(pInfo,pInfo->rx_answer);
		pInfo->rx_buf[pInfo->curr_rxbuf].in_use = 1;		/* mark buffer as valid input */
		pInfo->rx_buf[pInfo->curr_rxbuf].rx_len = pInfo->rx_len;
		if(++pInfo->curr_rxbuf == pInfo->nr_rxbuf)
			pInfo->curr_rxbuf=0;
			pInfo->read_state = RECEIVE_MESSAGE;
		wake_up_interruptible(&pInfo->read_wait);
	}
	pInfo->nextchar=wait_soh;
}
/* entry for receiver interrupt */
static void asycom_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char flags)
{
	struct asycom_info *pInfo=(struct asycom_info*)tty->disc_data;

	if(pInfo->nextchar)
		pInfo->nextchar(pInfo,cp,flags);
}
static void asycom_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
int i;
char *f;
const unsigned char *p;
char flags=0;
	for (i=count, p = cp, f = fp; i; i--, p++)
	{
		if (f)
			flags = *f++;
		asycom_single_char_received(tty,*cp++,flags);
	}
}

static int asycom_receive_room(struct tty_struct *tty)
{
	return -1;
}




/* init for compiled-in driver */
int __init asycom_init(void)
{
	int status;
	printk ("Sartorius Asycom driver $Revision: 1.4 $ installed\n");
	/*
	 * Register the tty line discipline
	 */

	status = tty_register_ldisc (N_ASYCOM, &tty_ldisc_N_ASYCOM);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_ASYCOM);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_ASYCOM.flags,
		        tty_ldisc_N_MODBUS.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_ASYCOM.open);
		TRACE_L("tty_ldisc_N_ASYCOM = %x", (int)&tty_ldisc_N_ASYCOM);
	}
	else
	{
		printk (KERN_ERR "asycom: error registering line discipline: %d\n", status);
	}
	return status;
}
/* Cleanup - undid whatever init_module did */
void __exit asycom_exit(void)
{
	int status;


	status=tty_register_ldisc(N_ASYCOM, NULL);

	if(status!=0)
	{
		printk(KERN_ERR "asycom: error unregistering linediscipline: %d\n", status);
	}
}

module_init(asycom_init);
module_exit(asycom_exit);

MODULE_LICENSE("GPL");
MODULE_ALIAS_LDISC(N_ASYCOM);
