/*
 * bootprogress for Linux
 *
 * Copyright (C) 2007 Sartorius Mechatronics
 *
 * this driver allows write access to beep on labpro-cubis
 *
 * usage:
 *
 *   const long melody[]={
 *        f1,t1,f2,t2,f3,t3,...
 *         }
 *   write(fd,melody,sizeof(melody)
 *
 * f1,f2,f3 is frequency in Hz if >0
 *             pause if 0
 *          divider if <0
 * t1,t2,t3 is time in millisecs (rounded to timer ticks)
*/

#define BOOTPROGRESS_VERSION	"0.1"

#include <linux/module.h>
#include <linux/config.h>
#include <linux/sched.h>

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>

//------------------------------------------------------------------
// HW-Dependent defines
//------------------------------------------------------------------
#ifdef CONFIG_SAG_LABPRO_CORE
#include <asm/m532xsim.h>
#define DSP_BASE_ADDR 0x41fc0000
#define DSP_W 320
const int prog_x=93;	// position of progress bar in bootscreen
const int prog_y=152;	// position of progress bar in bootscreen
const int prog_h=5;
const int prog_w=203;

const unsigned PCR_MONOCROM=0
		|(0<<30)    // Monochrome
		|(2<<28)    // 4-bit panel bus width
		|(2<<25)    // 4BPP
		|(0<<24)    // Pixel polarity: active high
		|(0<<23)    // First line marker (FLM) polarity: active high
		|(0<<22)    // Line pulse (LP) polarity: active high
		|(0<<21)    // LCD shift clock polarity: active negative edge
		|(0<<20)
		|(1<<19)
		|(1<<18)    // Endian select: big endian
		|(1<<17)    // Swap Select: 8 bpp, 4 bpp, 2 bpp, 1 bpp mode
		|(0<<15)
		|(1<<7)
		|(12)       // Pixel clock divider
		;
const unsigned PCR_TFT_QVGA=0
		|(3<<30)    // TFT
		|(3<<25)    // 8BPP
		|(0<<24)    // active pol
		|(1<<23)
		|(1<<22)
		|(1<<21)
		|(0<<20)
		|(1<<19)
		|(1<<18)
		|(1<<17)
		|(1<<15)
		|(1<<7)
		|(12)
		;
#endif

// we need a mutex in case of multiple concurrent writes
static DECLARE_MUTEX(sem);

//=================================================
// this is the bad progress code
// derived from bios
//=================================================
static int bootprogress_did_write=0;
static unsigned bootprogress_cc_count=10;	// start value for kernel

void DrawProgress8(int x0,int x1)
{
	char*pl=(char*)(DSP_BASE_ADDR+DSP_W*prog_y+x0);
	int c,x;
	for(c=100;c<100+prog_h;++c)
	{
		char*pp=pl;
		for(x=x0;x<=x1;++x)
			*pp++=c;
		pl+=DSP_W;
	}
}

void DrawProgress8Mono(int x0,int x1)
{
static const unsigned char col[5]={~0,~0,~0,15,~0};
	char*pl=(char*)(DSP_BASE_ADDR+DSP_W*prog_y+x0);
	int c,x;
	for(c=0;c<0+prog_h;++c)
	{
		unsigned cc=col[c];
		char*pp=pl;
		for(x=x0;x<=x1+1;++x) {
			if((x>=x1)) cc=~0;
			*pp++=cc;
		}
		pl+=DSP_W;
	}
}

void DrawProgress4(int x0,int x1)
{
static const unsigned char col[5]={0,0,0,~0,0};
	char*pl=(char*)(DSP_BASE_ADDR+(DSP_W/2)*prog_y+(x0/2));
	int c,x;
	for(c=0;c<prog_h;++c)
	{
		unsigned cc=col[c];
		char*pp=pl;
		for(x=x0;x<=x1+2;++x)
		{
			if(x==x1) cc=0;
			if(x&1)
			{
				*pp=(*pp&0xF0)|(cc&0x0F);
				++pp;
			}
			else
			{
				*pp=(*pp&0x0F)|(cc&0xF0);
			}
		}
		
		pl+=DSP_W/2;
	}
}

void boot_progress(unsigned pc)
{
	if(pc>=prog_w) pc=prog_w-1;	// do a limit
	if(pc<=bootprogress_cc_count) return;	// nop
	if(MCF_LCDC_LPCR==PCR_MONOCROM)
		DrawProgress4(prog_x+bootprogress_cc_count,prog_x+pc);
	else
		if(MCF_LCDC_LPCR==PCR_TFT_QVGA)
			//DrawProgress8(prog_x+bootprogress_cc_count,prog_x+pc);
			DrawProgress8(prog_x,prog_x+pc);
		else
			DrawProgress8Mono(prog_x+bootprogress_cc_count,prog_x+pc);
	bootprogress_cc_count=pc;
}

void bootprogress_printk_lf(void)
{
	if(!bootprogress_did_write)
		boot_progress(bootprogress_cc_count+1);
}

//-------------------------------------------
// allow a single write of an absolute value
// or a relative prefixed with '+'
//-------------------------------------------
static ssize_t bootprogress_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	unsigned pc=0;
	int wr=0;
	int add=0;
	int percent=0;
	char c;
	down(&sem);
	for(;count>0;count-=1)
	{
		get_user(c,buf);	// get char
		++buf;
		++wr;
		if(c=='+')
		{
			add='+';
			continue;
		}
		if(c=='%')
		{
			percent='%';
			continue;
		}
		if(c>='0' && c<='9')
		{
			pc=pc*10+(c-'0');
			continue;
		}
		break;
	}
	if(percent=='%')
		pc=pc*prog_w/100;
	if(add=='+')
		pc+=bootprogress_cc_count;
	boot_progress(pc);
	bootprogress_did_write=1;
	up (&sem);
	return wr;
}

//-------------------------------------------
// allow to read current progress once (!)
//-------------------------------------------
static ssize_t bootprogress_read(struct file *file, char __user *buf, size_t size, loff_t *ppos)
{
	unsigned pc;
	int wr;
	char buff[32];
	char*p;
	if(file->f_pos) return 0;	// treat as end of file
	//-------------------------
	// get the count
	//-------------------------
	pc=bootprogress_cc_count;
	//-------------------------
	// convert to string
	//-------------------------
	p=buff+32;
	*--p=0;
	*--p='\n';	// add a linefeed
	do
	{
		*--p=pc%10U+'0';
		pc/=10U;
	}while(pc);
	wr=(buff+31)-p;
	//-------------------------
	// write to buffer
	//-------------------------
	if(wr>size) wr=size;
	copy_to_user(buf,p,wr);
	file->f_pos+=wr;
	if(ppos)
		*ppos+=wr;
	return wr;
}

int bootprogress_open(struct inode *inode, struct file *file)
{
	// this is a trivial
	file->f_pos=0;
	return 0;
}

static struct file_operations bootprogress_fops = {
	.owner		= THIS_MODULE,
	.write		= bootprogress_write,
	.read		= bootprogress_read,
//	.ioctl		= bootprogress_ioctl,
	.open		= bootprogress_open,
//	.release	= bootprogress_release,
};

static struct miscdevice bootprogress_dev = {
	BOOTPROGRESS_MINOR,
	"bootprogress",
	&bootprogress_fops
};

static int __init
bootprogress_init(void)
{
	int ret;

	ret = misc_register(&bootprogress_dev);
	if (ret) {
		printk(KERN_ERR "bootprogress: can't misc_register on minor=%d\n",
		       BOOTPROGRESS_MINOR);
		goto out;
	}

	ret = 0;
	printk(KERN_INFO "Sartorius bootprogress driver v" BOOTPROGRESS_VERSION "\n");
      out:
	return ret;
}

static void __exit
bootprogress_cleanup_module(void)
{
	misc_deregister(&bootprogress_dev);
}

module_init(bootprogress_init);
module_exit(bootprogress_cleanup_module);

MODULE_LICENSE("GPL");

MODULE_ALIAS_MISCDEV(BOOTPROGRESS_MINOR);
EXPORT_SYMBOL(bootprogress_printk_lf);
