/*
 * DMABEEP for Linux
 *
 * Copyright (C) 2007 Sartorius Mechatronics
 *
 * this driver allows write access to beep on labpro-cubis
 *
 * usage:
 *
 *   const long melody[]={
 *        f1,t1,f2,t2,f3,t3,...
 *         }
 *   write(fd,melody,sizeof(melody)
 *
 * f1,f2,f3 is frequency in Hz if >0
 *             pause if 0
 *            divider if <0
 * t1,t2,t3 is time in millisecs (rounded to timer ticks)
 * tn=long(0x80000000) can be used to leave last tone on
 * if fn=long(0x80000000) tn specifies volume 0..1000%%
*/

#define DMABEEP_VERSION	"0.1"

#include <linux/module.h>
#include <linux/config.h>
#include <linux/sched.h>

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/mcf528x_dma_timer.h>
#include <asm/m532xsim.h>

//------------------------------------------------------------------
// HW-Dependent defines
//------------------------------------------------------------------
#ifdef CONFIG_SAG_LABPRO_CORE
#define DMA_TIMER_BEEP MCF_DMA_TIMER(3)	/* timer used for beep output */
#define DMA_TIMER_BEEP_CLK 40000000	/* clock input for beep timer */
#define OUTPUT_OFF  MCF_GPIO_PAR_PWM=(MCF_GPIO_PAR_PWM & ~(3<<2))|(0<<2); // PWM3 is GPIO
#define OUTPUT_ON   MCF_GPIO_PAR_PWM=(MCF_GPIO_PAR_PWM & ~(3<<2))|(2<<2); // PWM3 is DT3OUT
#define VOLUME_ON   MCF_GPIO_PAR_PWM=(MCF_GPIO_PAR_PWM & ~(3<<4))|(1<<4); // PWM5 is PWM

#endif


#define T_NO_OFF (long)(0x80000000)	// value for no-off
#define F_VOLUME (long)(0x80000000)	// freq for volume setting
// we need a mutex in case of multiple concurrent writes
static DECLARE_MUTEX(sem);


static int sag_dmabeep_def_volume=128;

inline void beep(unsigned div)
{
	DMA_TIMER_BEEP.dtxmr=DTXMR_DMAEN_0	// no dma request on reference match
			|DTXMR_MODE16_0;	// dont use 16bit mode
	DMA_TIMER_BEEP.dtmr=DTMR_PS(1)	// prescale at sysclock
			|DTMR_CE_NONE	// capture no edge
			|DTMR_OM_1	// output mode toggle
			|DTMR_ORRI_0	// disable interrupt
			|DTMR_FRR_1	// restart when count is reached
			|DTMR_CLK_DIV1	// clock is system/1
			|DTMR_RST_1;	// enable timer
	DMA_TIMER_BEEP.dtrr=div;
	DMA_TIMER_BEEP.dtcn=0;
	OUTPUT_ON;
}

inline void setup_volume(unsigned vol)
{
	//------------------------------------
	// setup PWM
	//------------------------------------
	VOLUME_ON; 	// PWM5 is PWM
	MCF_PWM_pwmpol&=~(1<<5);	// start pol 0
	MCF_PWM_pwmcae&=~(1<<5);	// no center-align
	MCF_PWM_pwmclk&=~(1<<5);	// select clock B
	MCF_PWM_pwmctl&=~(1<<6);	// dont connect 45
	MCF_PWM_pwmper5=255;		// periode 255
	MCF_PWM_pwme|=(1<<5);		// enable PWM3
	MCF_PWM_pwmdty5=vol;
}

inline void beep_off(void)
{
	DMA_TIMER_BEEP.dtmr=0;
	OUTPUT_OFF;
}	// disable

static ssize_t dmabeep_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	long f,t=0;
	int wr=0;
	long initial_count=count;
	down(&sem);
	setup_volume(sag_dmabeep_def_volume);
	for(;count>=8;count-=8,	buf+=8,	wr+=8)
	{
		get_user(f,(long*)(buf+0));	// get f
		get_user(t,(long*)(buf+4));	// get t
		if(f==F_VOLUME)
		{
			if(t<0) t=0;
			if(t>1000) t=1000;
			t=255-(t*255)/1000;
			if(initial_count==8)	// only set default
			{
				sag_dmabeep_def_volume=t;
				t=T_NO_OFF;	// dont change pending tone
				break;
			}
			MCF_PWM_pwmdty5=t;
			continue;
		}
		if(t<0 && t!=T_NO_OFF)	// do sync to timeout
		{
			t=-t;
			set_task_state(current,TASK_INTERRUPTIBLE);
			schedule_timeout(1);
			if(signal_pending(current))
			{
				wr=-EINTR;
				break;
			}
		}
		if(f==0)	// pause
		{
			beep_off();
		}
		else
		{
			if(f>0)	// frequency in Hz
				f=DMA_TIMER_BEEP_CLK/f;
			else	// absolute divider
				f=-f;
			beep(f);
		}
		if(t==T_NO_OFF)
		{
			break;
		}
		// calculate ticks
		t=(t+1000/HZ-1)/(1000/HZ);
		set_task_state(current,TASK_INTERRUPTIBLE);
		schedule_timeout(t);
		if(signal_pending(current))
		{
			wr=-EINTR;
			break;
		}

	}
	if(t!=T_NO_OFF)
		beep_off();
	set_task_state(current,TASK_RUNNING);
	up (&sem);
	return wr;
}

static struct file_operations dmabeep_fops = {
	.owner		= THIS_MODULE,
	.write		= dmabeep_write,
//	.ioctl		= dmapeep_ioctl,
//	.open		= dmapeep_open,
//	.release	= dmapeep_release,
};

static struct miscdevice dmabeep_dev = {
	FANCYBEEP_MINOR,
	"dmabeep",
	&dmabeep_fops
};

static int __init
dmapeep_init(void)
{
	int ret;

	ret = misc_register(&dmabeep_dev);
	if (ret) {
		printk(KERN_ERR "dmabeep: can't misc_register on minor=%d\n",
		       FANCYBEEP_MINOR);
		goto out;
	}
	ret = 0;
	printk(KERN_INFO "Sartorius DmaBeep driver v" DMABEEP_VERSION "\n");
      out:
	return ret;
}

static void __exit
dmabeep_cleanup_module(void)
{
	misc_deregister(&dmabeep_dev);
}

module_init(dmapeep_init);
module_exit(dmabeep_cleanup_module);

MODULE_LICENSE("GPL");

MODULE_ALIAS_MISCDEV(DMABEEP_MINOR);
