/************************************************************************/
/*                                                                      */
/*  hms.c - driver for HMS fieldbus cards                               */
/*                                                                      */
/*  (C) Copyright 2004, Sartorius Hamburg GmbH                          */
/*                                                                      */
/************************************************************************/
#include <linux/module.h>
#include <linux/version.h>
#include <asm/delay.h>
#include <asm/irq.h>
#include <asm/mcfedgeport.h>

#include <linux/fs.h>
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <asm/uaccess.h> /* for put_user */
#include <linux/init.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/pm.h>

#include <asm/coldfire.h>
#include <linux/config.h>
#include "sag_hms.h"
#include <asm/sag_hms.h>

#if defined(CONFIG_M527x)
#include <asm/m528xsim.h>
#endif

/* for use with Sartorius boards */
#define MCFINTC0_BASEVEC       64

#define DEVICE_NAME "hms"
static DECLARE_MUTEX(sem);

/* struct wait_queue *wqueue;   */
static DECLARE_WAIT_QUEUE_HEAD(wqueue);

local_data status;
typedef volatile unsigned int vu32;
typedef volatile unsigned long vul32;
typedef volatile unsigned char vuc32;

int __init hms_init(void);
void __exit hms_exit(void);


static int hms_state;
#if defined(CONFIG_SAG_PR5800_CORE)
#define HMS_RESET_PORT (*(vuc32*)0x80010300)
#define HMS_RESET_BIT_MASK 0x01
#define HMS_RESET_BIT_NUMB 0
#endif

#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
#define HMS_RESET_PORT (*(vuc32*)0x40100001)
#define HMS_RESET_BIT_MASK 0x04
#define HMS_RESET_BIT_NUMB    2
#endif

#define MOVEB(d,s) asm volatile(" move.b (%0)+,(%1)+;nop;"\
		: "=a"(s),"=a"(d) \
		:  "0"(s), "1"(d) : "memory");

#define MOVEIB(d,s) asm volatile(" move.b %1,(%0)+;nop;"\
		: "=a"(d) \
		:  "d"(s), "0"(d) : "memory");

void dpram_copy(volatile char*dst,const volatile char*src,long size)
{
	while(size>=16)
	{
		// move 16 words=32bytes
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);	MOVEB(dst,src);
		size-=16;
	}
	while(size>0)
	{
		MOVEB(dst,src);
		size-=1;
	}
}
void dpram_init(volatile char*dst,const char val,long size)
{
	while(size>=16)
	{
		// move 16 words=32bytes
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);	MOVEIB(dst,val);
		size-=16;
	}
	while(size>0)
	{
		MOVEIB(dst,val);
		size-=1;
	}
}

#define BCLR(a,b) asm volatile("bclr %1,(%0)": :"a"(&a),"i"(b))
#define BSET(a,b) asm volatile("bset %1,(%0)": :"a"(&a),"i"(b))


#if defined(CONFIG_SAG_PR5410_CORE) || defined(CONFIG_SAG_PR5800_CORE) || defined(CONFIG_SAG_PR5220_CORE)
static int hms_reset(int mode)
{

	/* enable edgeport interrupt */

	volatile unsigned char *pepier=(volatile unsigned char *)(MCF_MBAR+MCF_EPIER);
	unsigned epier=*pepier;
	//----------------------------
	// check slot 4 (3)
	//----------------------------
	unsigned char tmp[4];
	int bit;
	int n;
        vuc32 * dpram=(vuc32 *)DPRAM_BASE;

	BCLR(*pepier,MCFHMS_IRQ_INT);

	status.app_control = 0;
	status.found = 0;
	//--------------------------------------
	// pull reset low to get card inactive
	//--------------------------------------
	BCLR(HMS_RESET_PORT,HMS_RESET_BIT_NUMB);
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(1);	// wait one HZ tick (5ms)
	// check if a card is inserted
	dpram_copy(dpram,"\x55\xAA\x55\xAA",4);
	dpram_copy(tmp,dpram,4);
	tmp[0]=~tmp[0];
	tmp[1]=~tmp[1];
	tmp[2]=~tmp[2];
	tmp[3]=~tmp[3];
	dpram_copy(dpram,tmp,4);
	dpram_copy(tmp,dpram,4);
	if(tmp[0]==0xAA && tmp[1]==0x55 && tmp[2]==0xAA && tmp[3]==0x55)
	{
		//---------------------------------------------
		// we have detected a dual ported ram
		//---------------------------------------------
		// read the byte to quit interrupts
		((volatile unsigned char*)dpram)[0xFFE];
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1);	// wait one HZ tick (5ms)
		((volatile unsigned char*)dpram)[0xFFF];
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1);	// wait one HZ tick (5ms)
		BSET(HMS_RESET_PORT,HMS_RESET_BIT_NUMB);	// reset high -> card will start
		bit=1;	// assume no reaction
		// wait max 5 seconds for card ready
		for(n=0; n<5000/*ms*/; n+=1000/HZ)
		{
			bit=(*HMS_INTERRUPT_PORT>>MCFHMS_IRQ_INT)&1;
			if(bit==0)	// card went ready
				break;
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(1);	// wait one HZ tick (5ms)
		}
		set_current_state(TASK_RUNNING);
		if(bit==0)	// we have an reaction from the cart
		{
			status.md_type=*(unsigned short*)(dpram+0x800+0x7E0);
			status.fb_type=*(unsigned short*)(dpram+0x800+0x7CC);
			status.any_control = *(dpram+HMS_ANY_CONTROL);
			status.found = 1;
		}
	}
	*(vuc32 *)(MCF_MBAR + MCF_EPFR) = 1<<(MCFHMS_IRQ_INT); /* reset interrupt */

	if(epier&(1<<MCFHMS_IRQ_INT))
		BSET(*pepier,MCFHMS_IRQ_INT);


	return status.found;
}
#else
#error "missing implemention of hms_reset"
#endif



static int hms_ioctl(struct inode *inode, struct file *file,
				unsigned int cmd, unsigned long arg)
{
	int rc = 0;
	unsigned char wmask=0;
	unsigned char rmask=0;
	unsigned char handshake=0;
	int wr_offs=0;
	int rd_offs=0;
	unsigned char *odat,*idat;
	unsigned long osiz,isiz;
	unsigned long	flag;
	int __user *result=(int __user*)arg;
	hms_para *para=(hms_para*)arg;
	vuc32 *dpram=(vuc32 *)DPRAM_BASE;
	// if a powerfail was detected, the process must reinitialize this device
	if(hms_state)
	{
		hms_state = 0;
		return -EINTR;
	}
	down(&sem);
	switch(cmd)
	{
        	case DATA_IO:
	        	// if no card is inserted
			if(status.found == 0) {
				rc =-ENXIO;
				goto error;
			}
         		wr_offs=0; rd_offs=0x200;
                	get_user(odat,&para->odat);
                	get_user(osiz,&para->osiz);
                    	if(odat)
                     	{
			      	// lock input area
                               	do
                               	{
                           		handshake = status.any_control^status.app_control;
                           		handshake = (handshake&0xe0)|0x14;
                             		status.app_control = handshake;
                           		save_flags(flag); cli();
                           		dpram[HMS_APP_CONTROL] = handshake;
                           		interruptible_sleep_on_timeout(&wqueue,(HZ / 10));
                           		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
                               	}while((status.any_control&0x4)==0);
                        	dpram_copy ((void*)dpram+wr_offs,odat,osiz);
                         	// release input
                         	do
				{
                           		handshake = status.any_control^status.app_control;
                           		handshake = (handshake&0xe0)|0x4;
                             		status.app_control = handshake;
                           		save_flags(flag); cli();
                           		dpram[HMS_APP_CONTROL] = handshake;
                           		interruptible_sleep_on_timeout(&wqueue,(HZ / 10));
                           		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
				}while((status.any_control&0x8)==0);
                        }
                        get_user(idat,&para->idat);
                 	get_user(isiz,&para->isiz);
                  	if(idat)
                   	{
			      	// lock output area
                               	do
                               	{
                           		handshake = status.any_control^status.app_control;
                           		handshake = (handshake&0xe0)|0x12;
                             		status.app_control = handshake;
                           		save_flags(flag); cli();
                           		dpram[HMS_APP_CONTROL] = handshake;
                           		interruptible_sleep_on_timeout(&wqueue,(HZ / 10));
                           		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
                               	}while((status.any_control&0x2)==0);
                                copy_to_user (idat,(void*)dpram+rd_offs,isiz);
                         	// release output
                         	do
				{
                           		handshake = status.any_control^status.app_control;
                           		handshake = (handshake&0xe0)|0x2;
                             		status.app_control = handshake;
                           		save_flags(flag); cli();
                           		dpram[HMS_APP_CONTROL] = handshake;
                           		interruptible_sleep_on_timeout(&wqueue,(HZ / 10));
                           		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
				}while((status.any_control&0x8)==0);
			}
           		break;
         	case MBX_IO:
	        	// if no card is inserted
			if(status.found == 0) {
				rc =- ENXIO;
				goto error;
			}
			wmask=0x80;rmask=0x40; wr_offs=0x400; rd_offs=0x520;
                	get_user(odat,&para->odat);
                	get_user(osiz,&para->osiz);
                    	if(odat)
                     	{
                        	while(((status.any_control^status.app_control)&wmask)!=0)
                                {
                        		save_flags(flag); cli();
                        		interruptible_sleep_on(&wqueue);
                        		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
                        	}

                        	dpram_copy ((void*)dpram+wr_offs,odat,osiz);
                                status.app_control^=wmask;
                        	save_flags(flag); cli();
                                dpram[HMS_APP_CONTROL]=status.app_control;

                        	interruptible_sleep_on(&wqueue);
                        	restore_flags(flag);
				if (signal_pending(current)) {
					rc = -EINTR;
					goto error;
				}
                	}
                        get_user(idat,&para->idat);
                 	get_user(isiz,&para->isiz);
                  	if(idat)
                   	{
                          	while(((status.any_control^status.app_control)&rmask)==0)
                                {
                        		save_flags(flag); cli();
                        		interruptible_sleep_on(&wqueue);
                        		restore_flags(flag);
					if (signal_pending(current) || !status.initialized) {
						rc = -EINTR;
						goto error;
					}
                        	}
                                copy_to_user (idat,(void*)dpram+rd_offs,isiz);
                                status.app_control^=rmask;
                                dpram[HMS_APP_CONTROL]=status.app_control;
                        }
             		break;
          	case CONTROL_READ:
	        	// if no card is inserted
			if(status.found == 0) {
				rc = -ENXIO;
				goto error;
			}
             		get_user(rd_offs,&para->ioffset);
                        get_user(idat,&para->idat);
                 	get_user(isiz,&para->isiz);
                	get_user(odat,&para->odat);
                	get_user(osiz,&para->osiz);
                   	if(rd_offs > 0x7fe) {
				rc = -EINVAL;
				goto error;
			}
			// lock control area
			do
			{
				handshake = status.any_control^status.app_control;
				handshake = (handshake&0xe0)|0x11;
				status.app_control = handshake;
				save_flags(flag); cli();
				dpram[HMS_APP_CONTROL] = handshake;
				interruptible_sleep_on(&wqueue);
				restore_flags(flag);
				if (signal_pending(current) || !status.initialized) {
					rc = -EINTR;
					goto error;
				}
			}while((status.any_control&0x1)==0);
                  	if(idat)
                                copy_to_user (idat,(void*)dpram+rd_offs,isiz);
			else if(odat)
                        	dpram_copy ((void*)dpram+wr_offs,odat,osiz);
			// release control arae
			do
			{
				handshake = status.any_control^status.app_control;
				handshake = (handshake&0xe0)|0x1;
				status.app_control = handshake;
				save_flags(flag); cli();
				dpram[HMS_APP_CONTROL] = handshake;
				interruptible_sleep_on(&wqueue);
				restore_flags(flag);
				if (signal_pending(current) || !status.initialized) {
					rc = -EINTR;
					goto error;
				}
			}while((status.any_control&0x8)==0);
             		break;
               // test if a mbx message is available
               // a MBX_IO read must be follow
               case DPV1_ACYCLIC_CMD:
	        	// if no card is inserted
			if(status.found == 0) {
				rc = -ENXIO;
				goto error;
			}
                	get_user(idat,&para->idat);
               		if((status.any_control^status.app_control)&0x40)
                               copy_to_user (idat,(void*)"1",1);
                 	else
                               copy_to_user (idat,(void*)"0",1);
                	break;
               case HMS_RESET:
               		rc = hms_reset(1);
			if(!rc) rc=-ENXIO;	// no card
			if(result && rc ) put_user(1,result);
			status.initialized=1;
               		break;
		default:
			rc = -EINVAL;
			break;
   	}

error:
	up (&sem);
	return rc;
}

static int hms_release(struct inode *inode, struct file *file)
{
	disable_edgeport(MCFHMS_IRQ_INT);
	//--------------------------------------
	// pull reset low to get card inactive
	//--------------------------------------
	BCLR(HMS_RESET_PORT,HMS_RESET_BIT_NUMB);
	status.initialized=0;
	return 0;
}
irqreturn_t hms_intr(int irq, void *dummy, struct pt_regs *fp)
{
        vul32 *dpram=(vul32 *)DPRAM_BASE;
        status.any_control = dpram[HMS_ANY_CONTROL]; // quit 1 interrupt
	*(vuc32 *)(MCF_MBAR + MCF_EPFR) = 1<<(irq-64); /* reset interrupt */
	wake_up(&wqueue);	/* transfer finished */
	return IRQ_HANDLED;
}


static int hms_open(struct inode *inode, struct file *file)
{
	int sts;
	file->private_data = 0;
	sts = hms_reset(0);
	if(sts ==0) return -ENODEV;
	status.initialized=1;
	if(enable_edgeport(MCFHMS_IRQ_INT,FALLING_MODE,hms_intr)<0)
	{
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
		__FILE__,__LINE__,64+MCFHMS_IRQ_INT);
		return 0;
	}
	return 0;
}

#ifdef CONFIG_PM
static struct pm_dev *hms_pm_device;
static int hms_pm_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	switch (request) {
	case PM_SUSPEND:
		status.initialized=0;
		disable_edgeport(MCFHMS_IRQ_INT);  /* disable hms interrupt */
		break;

	case PM_RESUME:
		// enable interrupt
		wake_up(&wqueue);	/* transfer finished */
		enable_edgeport(MCFHMS_IRQ_INT,FALLING_MODE,hms_intr);
		break;
	}
	return 0;
}
#endif

static struct file_operations Fops = {
	owner:		THIS_MODULE,
	ioctl:		hms_ioctl,
	open:		hms_open,
	release:	hms_release  /* a.k.a. close */
};


/* init for compiled-in driver */
int __init hms_init()
{
//	int sts;
//	sts = hms_reset(0);
        printk ("Sartorius HMS device driver $Revision: 1.17 $ installed\n");
        if (register_chrdev(HMS_MAJOR, DEVICE_NAME, &Fops)) {
        	printk(KERN_NOTICE "Can't allocate major number %d for HMS devices.\n",
        	       HMS_MAJOR);
        	return -EAGAIN;;
        }
	init_waitqueue_head(&wqueue);
#ifdef CONFIG_PM
	hms_state =0;
	status.initialized=0;
	hms_pm_device = pm_register(PM_SYS_DEV, PM_SYS_COM, hms_pm_callback);
#endif
	return 0;
}

/* Cleanup - undid whatever init_module did */
void __exit hms_exit()
{
	/* Unregister the device */
	if ((unregister_chrdev(HMS_MAJOR, DEVICE_NAME)) < 0)
		printk("Error in unregister_chrdev\n");
#ifdef CONFIG_PM
	if(hms_pm_device)
	{
		pm_unregister(hms_pm_device);
		hms_pm_device =0;
	}
#endif
}

module_init(hms_init);
module_exit(hms_exit);

MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("hms");
MODULE_DESCRIPTION("Direct character-device access to HMS devices");
