/*
 * 	ColdFire 527x I2C master support
 */
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <asm/irq.h>
#include <linux/init.h>
#include <linux/delay.h>

#include <linux/sag_i2c.h>
#include <asm/coldfire.h>
#include <asm/m527xi2c.h>

#include <asm/mcfsim.h>   /* gets us MCFSIM_ICR4, MCFSIM_PACNT */
#include <asm/mcfedgeport.h>

#define DEVICE_NAME "i2c"
#define I2C_MAJOR 89

#ifdef CONFIG_SAG_PR5410_CORE
#define KEY_CHANGE_EDGEPORT 1
#define I2C_KEYB_ADDR 0x40
#endif

//-------------------------------------------------------------
// zur Clockrate:
// tSCL-Low=4.7us, tSCL-Hi=4us tRise=1.0us tFall=0.3us
// => es sollten nicht mehr als 100 kHz sein
//
// bei SysClock=50 MHZ brauchen wir einen Divider von 500
// der nchst beste ist 512 => 97,65khz
//
// 1 clock = 10,24us
// ein Zeichen = 9 Bits = 92uS + interrupt zeiten
// wenn wir ein Timeout von 10ms pro Zeichen machen, reicht das fr 100 zeichen
// der Display-Controller braucht ca 500us pro dargestelltes Zeichen
// wir schreiben maximal 16 zeichen im Block, macht 8ms
// mit 20ms sollten wir da sicher sein
//-------------------------------------------------------------
#if defined(CONFIG_SAG_PR5410_CORE)
#define I2C_DIVIDER MCF_I2FDR_IC_512
#endif
#if defined(CONFIG_SAG_COBRA_CORE) || defined(CONFIG_SAG_LABPRO_CORE)
#define I2C_DIVIDER MCF_I2FDR_IC_896
#endif

#define I2C_TIMEOUT_MS 20 /* [ms] sollte> 2 clocks sein, also bei 100kHz 2*10us */
#define I2C_TIMEOUT_JIFFIES ((I2C_TIMEOUT_MS*HZ)/1000+1)

int __init i2c_init(void);
int init(void);
void __exit i2c_exit(void);
int init_mcf_i2c(void);

static int thread_id=0;

static DECLARE_WAIT_QUEUE_HEAD(wq_keyb_int);	// wait queue for keyboard interrupt
static DECLARE_WAIT_QUEUE_HEAD(wq_keyb_getkey);	// wait queue for keyboard interrupt
static DECLARE_COMPLETION( on_exit );
static DECLARE_MUTEX(i2c_dev_sem);	// semaphore for the device

#define BCLRM(a,b) asm volatile("bclr %1,%0": :"m"(a),"i"(b))
#define BSETM(a,b) asm volatile("bset %1,%0": :"m"(a),"i"(b))


#if defined(CONFIG_M527x)
#define MCF_INTC0_IMRL *(volatile unsigned long *)(MCF_MBAR+MCFICM_INTC0+MCFINTC_IMRL)
#define MCF_INTC0_ICR0 *(volatile unsigned char *)(MCF_MBAR + MCFICM_INTC0+ MCFINTC_ICR0)
#define MCF_INTC0_ICR(x) *(volatile unsigned char *)(MCF_MBAR + MCFICM_INTC0+ MCFINTC_ICR0 + x)
#endif


i2c_dev the_i2c_dev;

int keyb_int=0;	// have keyboard interrupt

int wait_bit_lo(volatile u8 *port,unsigned mask)
{
	int timeout;
	if((*port&mask)==0) return 1;
	for(timeout=I2C_TIMEOUT_JIFFIES;timeout>0;--timeout)
	{
		schedule_timeout(1);	// 1 tick
		if((*port&mask)==0) return 1;
	}
	return 0;
}

/*
** reset the bus. task has to be in state TASK_INTERRUPTIBLE
*/
void i2c_reset_bus(const char*why)
{
	u8 dummy;
	if(why) printk ("i2cbus:i2c_reset_bus(%s)\n",why);
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	MCF_I2CR  = 0x00;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	MCF_I2CR  = MCF_I2CR_RSTA|MCF_I2CR_TXAK;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	dummy = MCF_I2DR;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	MCF_I2SR  = 0x00;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	MCF_I2CR  = 0x00;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
	/* enable I2C module */
	MCF_I2CR = MCF_I2CR_IEN;
	schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
}

/* i2c interrupt
   universal
*/


irqreturn_t i2c_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	u8 dummy;
	i2c_dev	*dev_inuse=(i2c_dev*)dev_id;
	//printk("i2c interrupt ");
	/* Clear the I2C Interrupt Flag. */
	BCLRM(MCF_I2SR,MCF_I2SR_IIF_BIT);
	if(MCF_I2CR & MCF_I2CR_MSTA)	/* in master mode */
	{
		/* Master Mode - Check if this device is in Transmit or Receive Mode. */
		if(MCF_I2CR & MCF_I2CR_MTX)	/* master is in transmit mode */
		{
			if(MCF_I2SR & MCF_I2SR_RXAK)	/* ACK not received- generate STOP */
			{
				BCLRM(MCF_I2CR,MCF_I2CR_MSTA_BIT);
				dev_inuse->txrx_sts=-EIO;	// IO-Error
				wake_up_interruptible(&dev_inuse->xwait);	/* transfer finished */
			}
			else if(dev_inuse->tx_len)	// more to transmit ?
			{
				MCF_I2DR = dev_inuse->tx_buf[dev_inuse->tx_index++];
				dev_inuse->tx_len--;
			}
			else if(dev_inuse->rx_len)	// something to receive?
			{
				if(dev_inuse->rx_index==~0)	// already in repeated start
				{
					BCLRM(MCF_I2CR,MCF_I2CR_MTX_BIT);	// switch to receive mode
					dummy = MCF_I2DR;		// dummy read
					dev_inuse->rx_index=0;	// restore
				}
				else
				{
					// set repeated start
					BSETM(MCF_I2CR,MCF_I2CR_RSTA_BIT);

					dev_inuse->curr_slvid |= 1;
					MCF_I2DR = dev_inuse->curr_slvid;
					dev_inuse->rx_index=~0;	// to indicate repeated start
				}
			}
			else	// no thing to transmit or receive
			{
				/* last byte was transmitted
				generate stop signal by changing
				to slave mode */
				BCLRM(MCF_I2CR,MCF_I2CR_MSTA_BIT);
				dev_inuse->txrx_sts=0;	// success
				wake_up_interruptible(&dev_inuse->xwait);	/* transfer finished */
			}
		}
		else /* master is in receive mode*/
		{
			if(dev_inuse->rx_index==~0)	// already in repeated start
			{
				BCLRM(MCF_I2CR,MCF_I2CR_MTX_BIT);	// switch to receive mode
				dummy = MCF_I2DR;		// dummy read
				dev_inuse->rx_index=0;	// restore
			}
			else if(dev_inuse->rx_len)	// something to receive?
			{
				if(dev_inuse->rx_len==2)	// if next will be the last one
				{
					BSETM(MCF_I2CR,MCF_I2CR_TXAK_BIT);	// generate dontan ACK
				}
				else if(dev_inuse->rx_len==1)		// last byte to read
				{
					BCLRM(MCF_I2CR,MCF_I2CR_MSTA_BIT);	//  generate stop signal by changing to slave mode
				}
				// receive one byte
				dev_inuse->rx_buf[dev_inuse->rx_index++]=MCF_I2DR;
				dev_inuse->rx_len--;
				if(dev_inuse->rx_len==0)
				{
					dev_inuse->txrx_sts=0;	// success
					wake_up_interruptible(&dev_inuse->xwait);	/* transfer finished */
				}
			}
			else	// interrupt with no recive or transmiut
			{
				printk("i2cint:unexpected mst_rx_int\n");
				/* Restore module to it's idle (but active) state */
				MCF_I2CR = MCF_I2CR_IEN;
			}
		}
	}
	else
	{
		printk("i2cint:unexpected slave mode interrupt\n");
		/* Restore module to it's idle (but active) state */
		MCF_I2CR = MCF_I2CR_IEN;
	}
	return IRQ_HANDLED;
}

// io-control: do one write or one read transfer on I2C BUS
// assume i2c_dev_sem is lockec
static int i2c_ioctl_txrx(i2c_dev*dev,I2C_PARA *para)
{
	int timeout;
	const unsigned char *odat;
	unsigned char *idat;
	unsigned osiz;
	unsigned isiz;
	/* get parameters and check */
	get_user(odat,&para->obuf);
	get_user(osiz,&para->osiz);
	get_user(idat,&para->ibuf);
	get_user(isiz,&para->isiz);
	get_user(dev->curr_slvid,&para->slvid);
	if(osiz>TX_BUF_SIZE)
		return -EINVAL;
	if(isiz>RX_BUF_SIZE)
		return -EINVAL;
	if(osiz==0 && isiz==0)
		return -EINVAL;
	/*  Ensure slaves are in idle state */
	if(MCF_I2SR & MCF_I2SR_IBB)
	{
		set_task_state(current,TASK_INTERRUPTIBLE);
		i2c_reset_bus("busy in txrx");
	}
	if(osiz && odat)
		copy_from_user(&dev->tx_buf,odat,osiz);
	dev->txrx_sts=-ETIME;	// assume timeout
	dev->tx_len=osiz;
	dev->tx_index=0;
	dev->rx_len=isiz;	// nothing to received
	dev->rx_index=0;
	/* Put module in master TX mode (generates START) */
	MCF_I2CR = MCF_I2CR_IEN|MCF_I2CR_IIEN|MCF_I2CR_MSTA| MCF_I2CR_MTX ;
	/* Put target address into IBDR */
	if(osiz)
		dev->curr_slvid &= ~1;
	else
		dev->curr_slvid |= 1;
	MCF_I2DR = dev->curr_slvid;
	// wait for transfer complete
	timeout=interruptible_sleep_on_timeout(&dev->xwait,I2C_TIMEOUT_JIFFIES);
	if (signal_pending(current))
		return -EINTR;
	if(dev->txrx_sts==-ETIME)
	{
		i2c_reset_bus("timeout in txrx");
		return -ETIME;
	}
	if(dev->txrx_sts)
		return dev->txrx_sts;
	if(isiz && idat)
	{
		copy_to_user(idat,&dev->rx_buf,isiz);
		put_user(isiz,&para->icnt);
	}
	return 0;
}


static int i2c_ioctl(struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg)
{
	int rc=0;
	i2c_dev*dev=file->private_data;
	switch(cmd)
	{
		case IOCTL_INKEY:
		case IOCTL_GETKEY:
			{
				// para[0] is returnvalue
				// para[1] is timeout
				unsigned *para=(unsigned*)arg;
				unsigned timeout_ms;
				unsigned keyresult=0;
				get_user(timeout_ms,&para[1]);	// this is the timeout
				down(&dev->keyb_sem);

				if(!dev->keyb_count)	// if we have dont heave any keys
				{
					if(timeout_ms)	// wait some time for the key
					{
						unsigned timeout_ticks=(timeout_ms*HZ+999)/1000;
						up(&dev->keyb_sem);	// release again for some time
						wait_event_interruptible_timeout(wq_keyb_getkey,dev->keyb_count!=0,timeout_ticks);	/* transfer finished */
						if (signal_pending(current))
							return -EINTR;
						down(&dev->keyb_sem);
					}
					else if(cmd==IOCTL_GETKEY)	// else if getkey, wait forever
					{
						up(&dev->keyb_sem);	// release again for some time
						wait_event_interruptible(wq_keyb_getkey, dev->keyb_count!=0);
						if (signal_pending(current))
							return -EINTR;
						down(&dev->keyb_sem);
					}
				}
				if(dev->keyb_count)	// if we have any keys
				{
					keyresult=dev->keyb_buf[dev->keyb_rd_pos++];
					dev->keyb_rd_pos &= (RX_BUF_SIZE-1);
					dev->keyb_count--;
				}
				put_user((unsigned)keyresult,para);
				up(&dev->keyb_sem);	// release again for some time
				break;
			}
		case IOCTL_IO:	// do a real transfer
		{
			struct I2C_PARA *para=(I2C_PARA*)arg;
			// now we need the device
			down(&i2c_dev_sem);
			rc=i2c_ioctl_txrx(dev,para);
			up (&i2c_dev_sem);
			break;
		}
		default:
			return -ENOIOCTLCMD;
	}
	return rc;
}

static int i2c_open(struct inode *inode, struct file *file)
{
//	printk("i2cbus:i2c_open\n");
	file->private_data = &the_i2c_dev;
	down(&i2c_dev_sem);
	set_task_state(current,TASK_INTERRUPTIBLE);
	i2c_reset_bus(0);
	set_task_state(current,TASK_RUNNING);
	up(&i2c_dev_sem);
	return 0;
}

static int i2c_release(struct inode *inode, struct file *file)
{
	// nothing to do
	return 0;
}

static struct file_operations Fops = {
	owner:		THIS_MODULE,
	ioctl:		i2c_ioctl,
	open:		i2c_open,
	release:	i2c_release  /* a.k.a. close */
};


#if defined(CONFIG_PR5410)
static void keyb_write_char(struct i2c_dev *pInfo, const unsigned char keybits)
{
	down(&pInfo->keyb_sem);
	if((keybits&~pInfo->keyb_lastbits))	// only on rising edge
	{
		if(pInfo->keyb_count<RX_BUF_SIZE)	// dont write if buffer full
		{
			pInfo->keyb_buf[pInfo->keyb_wr_pos++] = keybits;
			pInfo->keyb_wr_pos &=(RX_BUF_SIZE-1);
			pInfo->keyb_count++;
			wake_up_interruptible(&wq_keyb_getkey);	/* transfer finished */
		}
	}
	pInfo->keyb_lastbits=keybits;
	up(&pInfo->keyb_sem);
}



irqreturn_t keyb_interrupt(int irq, void *data, struct pt_regs *fp)
{
	*(volatile unsigned long *)(MCF_MBAR+MCFICM_INTC0+MCFINTC_IMRL)|=1<<(irq-64); // disable interrupt
	*(volatile unsigned char *)(MCF_MBAR + MCF_EPFR) = 1<<(irq-64);		 /* quit interrupt */
	keyb_int++;
	wake_up_interruptible(&wq_keyb_int);	/* transfer finished */
	return IRQ_HANDLED;
}

/*
** read the keyboard bits once
*/

unsigned get_keyb_bits(i2c_dev*dev_inuse)
{
	unsigned key;
	down(&i2c_dev_sem);
	//------------------------------
	// make shure bus is in idle mode
	//------------------------------
	if(!wait_bit_lo(&MCF_I2SR,MCF_I2SR_IBB))
	{
		i2c_reset_bus("busy in keyb");
		// assume ok now, else will get timeout
	}

	dev_inuse->txrx_sts=-ETIME;	// no status
	dev_inuse->tx_len=0;	// nothing to send
	dev_inuse->rx_index=0;	//
	dev_inuse->rx_len=1;	// receive one byte

	/* Put module in master TX mode (generates START) */
	MCF_I2CR = (MCF_I2CR_IEN | MCF_I2CR_MSTA | MCF_I2CR_MTX |MCF_I2CR_IIEN);
	/* Put target address into IBDR */
	MCF_I2DR = I2C_KEYB_ADDR | 1;
	/* Wait for transfer done */
	interruptible_sleep_on_timeout(&dev_inuse->xwait,I2C_TIMEOUT_JIFFIES);
	// get result from transfer
	key=dev_inuse->rx_buf[0];
	// on timeout reset bus to keep alive
	if(dev_inuse->txrx_sts==-ETIME)
	{
		i2c_reset_bus("timeout in get_keyb_bits");
		key=~0;
	}
	up (&i2c_dev_sem);
	return key;
}


/* scan 6 digit keyboard from PCF8574
   from address 0x40 */
static int thread_code( void *data )
{
    i2c_dev*dev_inuse=&the_i2c_dev;
    unsigned key1,key2;
    unsigned equal;
    daemonize("I2C-thread");
    allow_signal( SIGTERM );

    set_task_state(current,TASK_INTERRUPTIBLE);	// needed for waits
    key1=get_keyb_bits(dev_inuse);
    set_task_state(current,TASK_RUNNING);
    dev_inuse->keyb_lastbits=(~key1)&0x3F;
    while(1)
    {
	// wait for the interrupt
	wait_event_interruptible(wq_keyb_int, keyb_int!=0);
	if(signal_pending(current)) goto reenable;
	set_task_state(current,TASK_INTERRUPTIBLE);	// needed for waits
	key1=get_keyb_bits(dev_inuse);
	equal=0;
	while(key1!=~0)	// on timeout dont loop
	{
		schedule_timeout(1);//I2C_TIMEOUT_JIFFIES);
		key2=get_keyb_bits(dev_inuse);
		if(key1==key2)
		{
			if(++equal==3)
				break;
		}
		else
			equal=0;
		key1=key2;
	}
	set_task_state(current,TASK_RUNNING);
	if(key1!=~0)
		keyb_write_char(dev_inuse,(~key1)&0x3f);	// only write 1 bits

	keyb_int=0;	// ack int
reenable:
	*(volatile unsigned char *)(MCF_MBAR + MCF_EPFR) = 1<<(KEY_CHANGE_EDGEPORT); /* reset pending interrupt */
	*(volatile unsigned long *)(MCF_MBAR+MCFICM_INTC0+MCFINTC_IMRL) &= ~((1 << (KEY_CHANGE_EDGEPORT)) | 1); /* enable intr */

    }
    thread_id = 0;
    complete_and_exit( &on_exit, 0 );
}
#endif

int init_i2c_dev(i2c_dev*dev_inuse)
{
	init_waitqueue_head(&dev_inuse->xwait);
	dev_inuse->rx_len=0;
	dev_inuse->tx_len=0;
	dev_inuse->keyb_count=0;
	dev_inuse->keyb_lastbits=0;
	dev_inuse->keyb_wr_pos=dev_inuse->keyb_rd_pos=0;
	sema_init(&dev_inuse->keyb_sem,1);
	return 0;
}

int init_mcf_i2c(void)
{
	/* setup SCL clock */
	MCF_I2FDR =  I2C_DIVIDER;		// systemclock/512
	/* setup own address */
	MCF_I2AR=0x00;		// no slave address for me
	/* quit all interrupts */
	MCF_I2SR = 0;
	/* enable I2C module (but not the interrupt yet */
	MCF_I2CR = MCF_I2CR_IEN;

	/* request interrupt vector from kernel */
	if (request_irq(MCFI2C_IRQ_VECTOR, i2c_interrupt, SA_INTERRUPT, "ColdFire I2C", &the_i2c_dev)) {
		printk("I2C: Unable to attach ColdFire I2C interrupt "
			"vector=%d\n", MCFI2C_IRQ_VECTOR);
		return -EINVAL;
	}



	/* setup interrupt in CPU */
	MCF_INTC0_ICR(MCFI2C_IRQ_INT) = 056;         /* autovector on, il=5, ip=6 (octal)     */
	MCF_INTC0_IMRL &= ~((1 << (MCFI2C_IRQ_INT)) | 1);  /* enable i2c interrupt */

	return 0;
}

#if defined(CONFIG_PR5410)
int init_keyb_int(void)
{
	if(enable_edgeport(KEY_CHANGE_EDGEPORT,FALLING_MODE,keyb_interrupt)<0) {
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,KEY_CHANGE_EDGEPORT);
		return-EINVAL;
	}
	keyb_int=0;
	init_waitqueue_head(&wq_keyb_int);	// wait queue for keyboard interrupt
	init_waitqueue_head(&wq_keyb_getkey);	// wait for inkey/getkey
	return 0;
}
#endif

/* init for compiled-in driver: call from mem.c */
int __init i2c_init()
{
	int sts;
	if (register_chrdev(I2C_MAJOR, DEVICE_NAME, &Fops)) {
		printk(KERN_NOTICE "Can't allocate major number %d for I2C devices.\n",
		       I2C_MAJOR);
		return -EAGAIN;;
	}
        printk ("Sartorius I2C device driver installed\n");
	init_i2c_dev(&the_i2c_dev);
	sts=init_mcf_i2c();
	if(sts<0) return sts;
#if defined(CONFIG_PR5410)
	sts=init_keyb_int();
	if(sts<0) return sts;
	thread_id=kernel_thread(thread_code, NULL, CLONE_KERNEL );
	if( thread_id==0 )
		return -EIO;
#endif
	return 0;
}

/* Cleanup - undid whatever init_module did */
void __exit i2c_exit()      /* the __exit added by ron  */
{

	int ret=0;
	/* zero our IPL */
	MCF_INTC0_IMRL |=  ~((1 << (MCFI2C_IRQ_INT)));  /* disable qspi interrupt */
#if defined(CONFIG_PR5410)
	disable_edgeport(1);
#endif
	free_irq(MCFI2C_IRQ_VECTOR, NULL);

	/* Unregister the device */
	if ((unregister_chrdev(I2C_MAJOR, DEVICE_NAME)) < 0)
		printk("Error in unregister_chrdev: %d\n", ret);

	if( thread_id ) kill_proc( thread_id, SIGTERM, 1 );
	wait_for_completion( &on_exit );
}

module_init(i2c_init);
module_exit(i2c_exit);

MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("mcf_i2c");
MODULE_DESCRIPTION("Direct character-device access to i2c devices");


