/*
 * $Id: sag_keyb.c,v 1.25 2006/10/26 07:49:06 dec3ban Exp $
 *
 */
/**
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * This driver creates 4 virtual tty (0..3) devices which will be used
 * as input devices for user processes for receiving keycodes from
 * the combix keyboard as well as from an external PS/2 keyboard and
 * from VNC and IEC 61131 applications
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/input.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/init.h>
#include <linux/serio.h>
#include <asm/coldfire.h>
#include <asm/mcftimer.h>
#include <asm/uaccess.h>
#include <asm/mcfsim.h>
#include <asm/mcfpit.h>
#include <asm/irq.h>
#include <linux/input.h>
#include <linux/major.h>
#include <linux/device.h>
#include <linux/ioctl.h>
#include <linux/pm.h>
#include "sag_keyb.h"

#include <linux/kd.h>
#include <linux/kbd_kern.h>

//#define DEBUG

#ifdef DEBUG
#define TRACE_L(format, args...) printk("keyb: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif

#define DRIVER_DESC	"Sartorius combix keyboard driver"

#define SERIAL_MAGIC 0x5301


struct kb_private v_tty_data[MAX_NR_VTTY];

static char *combix_name = "Sartorius virtual Keyboard";
//static void init_keyboard_timer(void);
// if key_mode == 1 all keys except 0xC0ff are ignored
static short key_mode=0;
////////////////////////////////////////////////////////////////////

/**
*   PIT2 timer interrupt function. This function call the frontkeyboard scan function
*   in the bios. The keycodes will than be inserted into the tty0 or tty1 buffer,
*   depending from the keycode.
*/
#if defined(CONFIG_SAG_PR5800_CORE)\
 || defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)\
 || defined(CONFIG_SAG_MMA70_CORE)

irqreturn_t pit2_intr(int irq, void *dummy, struct pt_regs *fp)
{
	unsigned long keycode;
	volatile struct mcfpit *tp;
#if defined(CONFIG_SAG_PR5800_CORE)
	struct tty_struct *tty;
#endif
	/* Reset the ColdFire timer */
	tp = (volatile struct mcfpit *) (MCF_IPSBAR + MCFPIT_BASE2);
	tp->pcsr |= MCFPIT_PCSR_PIF;

	keycode=((int(*)(void))0xF0000418)();	// call bios function
#if defined(CONFIG_SAG_PR5800_CORE)
	if(keycode)
	{
		if(key_mode && (keycode&0xFFFFC000) != 0xC0ff)
			return IRQ_HANDLED;
		if((keycode&0xFFFFFF00) == 0x8000)		/* w&m key ? */
			tty = v_tty_data[0].tty;
		else if((keycode&0xFFFFFF00) == 0xC000)		/* w&m key ? */
			tty = v_tty_data[3].tty;
		else
			tty = v_tty_data[1].tty;
		if(tty)
		{
			*(long*)(tty->flip.flag_buf_ptr) = 0;
			*(long*)(tty->flip.char_buf_ptr) = keycode;
			tty->flip.flag_buf_ptr += 4;
			tty->flip.char_buf_ptr += 4;
			tty->flip.count += 4;
			schedule_work(&tty->flip.work);
		}
	}
#endif
	return IRQ_HANDLED;
}
#endif

static struct tty_driver *vt_serial_driver;

/*
 * -------------------------------------------------------------------
 * Here ends the serial interrupt routines.
 * -------------------------------------------------------------------
 */
#if defined(CONFIG_SAG_PR5800_CORE)
#define i (tmp.kb_index)
#define s (tmp.kb_table)
#define v (tmp.kb_value)
static inline int
do_kdsk_ioctl(int cmd, struct kbentry __user *user_kbe, int perm, struct kbd_struct *kbd)
{
	struct kbentry tmp;
	ushort *key_map, val, ov;

	if (copy_from_user(&tmp, user_kbe, sizeof(struct kbentry)))
		return -EFAULT;

	switch (cmd) {
	case KDGKBENT:
		key_map = key_maps[s];
		if (key_map) {
		    val = U(key_map[i]);
		    if (kbd->kbdmode != VC_UNICODE && KTYP(val) >= NR_TYPES)
			val = K_HOLE;
		} else
		    val = (i ? K_HOLE : K_NOSUCHMAP);
		return put_user(val, &user_kbe->kb_value);
	case KDSKBENT:
		if (!perm)
			return -EPERM;
		if (!i && v == K_NOSUCHMAP) {
			/* disallocate map */
			key_map = key_maps[s];
			if (s && key_map) {
			    key_maps[s] = NULL;
			    if (key_map[0] == U(K_ALLOCATED)) {
					kfree(key_map);
					keymap_count--;
			    }
			}
			break;
		}

		if (KTYP(v) < NR_TYPES) {
		    if (KVAL(v) > max_vals[KTYP(v)])
				return -EINVAL;
		}
#if 0
		else
		    if (kbd->kbdmode != VC_UNICODE)
				return -EINVAL;
#endif

		if (!(key_map = key_maps[s])) {
			int j;

			if (keymap_count >= MAX_NR_OF_USER_KEYMAPS &&
			    !capable(CAP_SYS_RESOURCE))
				return -EPERM;

			key_map = (ushort *) kmalloc(sizeof(plain_map),
						     GFP_KERNEL);
			if (!key_map)
				return -ENOMEM;
			key_maps[s] = key_map;
			key_map[0] = U(K_ALLOCATED);
			for (j = 1; j < NR_KEYS; j++)
				key_map[j] = U(K_HOLE);
			keymap_count++;
		}
		ov = U(key_map[i]);
		if (v == ov)
			break;	/* nothing to do */
		/*
		 * Attention Key.
		 */
		if (((ov == K_SAK) || (v == K_SAK)) && !capable(CAP_SYS_ADMIN))
			return -EPERM;
		key_map[i] = U(v);
		if (!s && (KTYP(ov) == KT_SHIFT || KTYP(v) == KT_SHIFT))
			compute_shiftstate();
		break;
	}
	return 0;
}
#undef i
#undef s
#undef v
#endif
static void vt_offintr(void *private)
{
	struct kb_private	*info = (struct kb_private *) private;
	struct tty_struct	*tty;

	tty = info->tty;
	if (!tty)
		return;
	tty_wakeup(tty);
}

static void vt_close(struct tty_struct *tty, struct file * filp)
{
	int line;
	struct kb_private	*info;
	info = tty->driver_data;

	line = tty->index;
	if ((line < 0) || (line >= MAX_NR_VTTY))
		return ;
	info = &v_tty_data[line];

	info->open_count--;
	if(info->open_count<0) info->open_count=0;
	if(info->tty == tty && !info->open_count)
	{
		tty->closing = 0;
		info->tty = 0;
	}
}

int vt_open(struct tty_struct *tty, struct file * filp)
{
	int line;
	struct kb_private	*info;

	line = tty->index;
	if ((line < 0) || (line >= MAX_NR_VTTY))
		return -ENODEV;
	info = &v_tty_data[line];

	/* don't open the line a second time */
	info->open_count++;
	if(info->tty)
		return -EBUSY;
	tty->driver_data = info;
	info->tty = tty;

	return 0;
}

static int vt_write_room(struct tty_struct *tty)
{
	return 4;
}

static int vt_chars_in_buffer(struct tty_struct *tty)
{
	return 0;
}
/**
 * dummy write function, because our virtual tty's are only input devices
*/
static int vt_write(struct tty_struct * tty,
		    const unsigned char *buf, int count)
{
	return 0;
}

/// ioctl: allow only setting of new keymap
int vt_ioctl(struct tty_struct *tty, struct file * file,
	     unsigned int cmd, unsigned long arg)
{
#if defined(CONFIG_SAG_PR5800_CORE)
	void __user *up = (void __user *)arg;
	int perm=1;
#endif
	struct kbd_struct * kbd;

	kbd = kbd_table;
	switch (cmd) {
#if defined(CONFIG_SAG_PR5800_CORE)
		case KDGKBENT:
		case KDSKBENT:
			return do_kdsk_ioctl(cmd, up, perm, kbd);
#endif
		case KDADDIO: key_mode=0; return 0;		/* add i/o port as valid 0x4B34*/
		case KDDELIO: key_mode=1; return 0;		/* del i/o port as valid 0x4B35*/
	}
	return -ENOIOCTLCMD;
}


static struct tty_operations vt_ops = {
	.open = vt_open,
	.close = vt_close,
	.ioctl = vt_ioctl,
	.write_room = vt_write_room,
	.write = vt_write,
	.chars_in_buffer = vt_chars_in_buffer,
};

/*-------------------------------------------------------------
** special line discipline
*/
/*
 * flags definition
*/


static void on_timeout(unsigned long priv);
static int  keyb_open(struct tty_struct *tty);
static void keyb_close(struct tty_struct *tty);
static int keyb_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void keyb_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void keyb_single_char_received(struct tty_struct *tty, const unsigned long cp);
static int  keyb_receive_room(struct tty_struct *tty);
static ssize_t	keyb_write(struct tty_struct * tty, struct file * file,
			 const unsigned char * buf, size_t nr);
static int keyb_chars_in_buffer(struct tty_struct *tty);
static void keyb_write_char(struct keyb_info *pInfo, const unsigned long keycode,int is_sendkey);

static struct tty_ldisc tty_ldisc_N_SAGKEYB =
    {
    .owner	 = THIS_MODULE,
    .magic	= TTY_LDISC_MAGIC,
    .name	= "SAGKEYB",
    .open	= keyb_open,
    .close	= keyb_close,
    .write	= keyb_write,
    .ioctl	= keyb_ioctl,
    .receive_buf = keyb_receive_buf,
    .receive_room = keyb_receive_room,
    .chars_in_buffer = keyb_chars_in_buffer,
    };

static int keyb_chars_in_buffer(struct tty_struct *tty)
{
    	struct keyb_info *pInfo=(struct keyb_info*)tty->disc_data;
	if(!pInfo) return 1;

	/* check for buffer full */
	if(pInfo->count == RX_BUF_SIZE)
		return 1;
	return 0;		/* we're not buffering */
}


static int  keyb_open(struct tty_struct *tty)
{
	struct keyb_info * pInfo;
	pInfo=kmalloc(sizeof(struct keyb_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "keyb: failed to alloc info structure\n");
		return -ENOMEM;
	}
	memset(pInfo,0,sizeof(struct keyb_info));
	pInfo->tty = tty;
	init_timer(&pInfo->tmr);
	pInfo->tmr.data = (unsigned long)pInfo;
	pInfo->tmr.function = on_timeout;
	init_waitqueue_head(&tty->read_wait);
	sema_init(&pInfo->sem,1);
	tty->disc_data = pInfo;

	return 0;
}

static void keyb_close(struct tty_struct *tty)
{
	struct keyb_info *pInfo=(struct keyb_info*)tty->disc_data;
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	del_timer_sync(&pInfo->tmr);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	tty->disc_data =0;
	kfree(pInfo);
}

/**
  * keyb_write: called only from tty2 which is the input from VNC to
  * our internal keyboard or tty3 which is input from IEC function send_key().
  * The keycode are put into the inputbuffer
  * from tty0 and tty1 depending from the keycode 0x8000.
*/
static ssize_t	keyb_write(struct tty_struct * tty, struct file * file,
			 const unsigned char * buf, size_t nr)
{
	unsigned long keycode;
	const unsigned char *p;
	int ret=0;
	int is_sendkey=0;
	int i;
	struct keyb_info *pInfo=0;
	struct tty_struct * ttyuse=0;
	if(tty==v_tty_data[3].tty)
		is_sendkey=1;
	if(nr%4) return -EINVAL;

	// if special mode set all key are ignored
	if(key_mode)
		return -EINVAL;

	for(i=0,p=buf;i< nr;)
	{
		keycode = *p++<<24;
		keycode |= *p++<<16;
		keycode |= *p++<<8;
		keycode |= *p++;
		i +=4;
		if((keycode&0xFFFFFF00) == 0x8000)		/* w&m key ? */
		{
			ttyuse = v_tty_data[0].tty;
			if(ttyuse)
				pInfo=(struct keyb_info*) v_tty_data[0].tty->disc_data;
		}
		else
		{
			ttyuse = v_tty_data[1].tty;
			if(ttyuse)
				pInfo=(struct keyb_info*) v_tty_data[1].tty->disc_data;
		}
		if(!pInfo || !ttyuse) return 0;
		/* ignore if buffer full */
		if(pInfo->count == RX_BUF_SIZE) break;

		keyb_write_char(pInfo,keycode,is_sendkey);
		ret++;
	}
	if(ret)
		wake_up_interruptible(&ttyuse->read_wait);
	return ret;;
}

static void on_timeout(unsigned long priv)
{
	struct keyb_info *pInfo = (void *)priv;
	pInfo->flags = TIMEOUT;
	wake_up_interruptible(&pInfo->tty->read_wait);
}

struct key_io{
	unsigned long rx_key;
	unsigned long timeout;
};

/**
 *   keyb_ioctl: implements the IEC-61131 DEV_KEYHIT, DEV_GETKEY, DEV_INKEY functions.
 *   return 0 if timeout or no charcater available, else 1
 *   @param buf  pointer to structure
                struct key_io{
	           unsigned long rx_key;
	           unsigned long timeout;
                   };

*/
static int keyb_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	int ret =0;
	long n;
	struct key_io *parg=(struct key_io *)arg;
	struct key_io para;
	struct keyb_info *pInfo=(struct keyb_info*)tty->disc_data;

	if(!pInfo) return -EPERM;

	if(cmd < TIOCKEYHIT || cmd > TIOCINKEY) return -ENOIOCTLCMD;

	copy_from_user((void*)&para,(void*)arg,sizeof(struct key_io));
	pInfo->flags = 0;
	if(!pInfo->count)
	{
		if(parg->timeout)
		{
			n=(parg->timeout+(1000/HZ)-1)/(1000/HZ);
			if(n==0) n=1;
			pInfo->tmr.expires = jiffies +n;
			add_timer(&pInfo->tmr);
		}
		switch(cmd)
		{
			case TIOCINKEY:
			case TIOCKEYHIT:
				/* no char and no timeout return 0 */
				if(!parg->timeout)
				{
					put_user(0,&parg->rx_key);
					return 0;
				}
				interruptible_sleep_on(&tty->read_wait);
				break;
			case TIOCGETKEY:
				interruptible_sleep_on(&tty->read_wait);
				/* if waked up by powerfail */
				if(!pInfo->flags)
					pInfo->flags = TIMEOUT;
				break;
		}
		if(parg->timeout)
			del_timer_sync(&pInfo->tmr);
	}
	down(&pInfo->sem);
	if(pInfo->flags == TIMEOUT)
		put_user(0,&parg->rx_key);
	else
	{
		switch(cmd)
		{
		case TIOCINKEY:
		case TIOCGETKEY:
			put_user(pInfo->rx_buf[pInfo->rd_pos++],&parg->rx_key);
			pInfo->rd_pos &= (RX_BUF_SIZE-1);
			pInfo->count--;
			ret = 1;
			break;
		case TIOCKEYHIT:
			put_user(pInfo->rx_buf[pInfo->rd_pos],&parg->rx_key);
			ret = 1;
			break;
		}
	}
	up(&pInfo->sem);
	return ret;
}

//------------------------------------------
// Some Attributes
//------------------------------------------
#define b_on	0x01
#define b_off	0x00
#define b_track 0x80
#define b_raw	0x40

#define is_click(k) (((k)&0xF0000000)==0x40000000)
#define click_x(k)  ((k>>0)&0x3FF)
#define click_y(k)  ((k>>10)&0x3FF)
#define click_b(k)  ((k>>20)&0xFF)
#define make_click(b,x,y) (0x40000000|((b&0xFF)<<20)|((y&0x3FF)<<10)|((x&0x3FF)<<0))

#define IsClickTrack(k)	(((k)&0xFFF00000)==make_click(b_track,0,0))
#define IsClickOn(k)	(((k)&0xFFF00000)==make_click(b_on,0,0))
#define IsClickOff(k)	(((k)&0xFFF00000)==make_click(b_off,0,0))

static void keyb_write_char(struct keyb_info *pInfo, const unsigned long keycode,int is_sendkey)
{
	unsigned wr_pos,last_wr,next_wr;
	if(pInfo->count==RX_BUF_SIZE) return;	// dont write if buffer full
	down(&pInfo->sem);
	wr_pos=pInfo->wr_pos;
	last_wr=(wr_pos-1)&(RX_BUF_SIZE-1);
	next_wr=(wr_pos+1)&(RX_BUF_SIZE-1);
	//=====================================
	// some special handling for touch
	//=====================================
	
	//-------------------------------------
	// tracking events can happen very fast
	// much faster than any application will handle
	// what we do:
	//   if new event is tracking event
	//   and last event was also a tracking event
	//   then replace last event with new event
	//-------------------------------------
	if(IsClickTrack(keycode))	// special for tracking
	{
		// if last code in buffer is also tracking
		if(pInfo->count && IsClickTrack(pInfo->rx_buf[last_wr]))	// also tracking
		{
			pInfo->rx_buf[last_wr]=keycode;	// then replace
		}
		else
		{
			goto write;	// else normal insert
		}
	}
	
	//-------------------------------------
	// this is to get slow applications in sync with touch
	//   if new event is touch-release
	//   and last event was touch-pressed
	//   then remove touch-pressed, because
	//        appl was not ready to recognize it
	//-------------------------------------
	else if(IsClickOff(keycode))	// if is touch-release
	{
		// if last code in buffer is touch-pressed
		if(pInfo->count && IsClickOn(pInfo->rx_buf[last_wr]))	// is click ON
		{
			// drop the on-key
			pInfo->wr_pos=last_wr;
			pInfo->count--;
			goto done;	// quick exit
		}
		else
		{
			goto write;	// else normal insert
		}
	}
	else if(is_sendkey)
	{
		unsigned rd_pos=(pInfo->rd_pos-1)&(RX_BUF_SIZE-1);
		pInfo->rx_buf[rd_pos] = keycode;
		pInfo->rd_pos = rd_pos;
		pInfo->count++;
	}
	else
	{
write:		pInfo->rx_buf[wr_pos] = keycode;
		pInfo->wr_pos=next_wr;
		pInfo->count++;
	}
done:
	up(&pInfo->sem);
	pInfo->flags = SUCCESS;
}

/* entry for receiver interrupt */
static void keyb_single_char_received(struct tty_struct *tty, const unsigned long keycode)
{
	struct keyb_info *pInfo=(struct keyb_info*)tty->disc_data;

	// if special key mode set, ignore all chars
	if(key_mode) return;

	if(!pInfo || pInfo->count == RX_BUF_SIZE) return;

	keyb_write_char(pInfo,keycode,0);
	wake_up_interruptible(&tty->read_wait);
}

static void keyb_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
int i;
const unsigned char *p;
unsigned long keycode;
	for (i=count, p = cp; i; i-=4)
	{
		keycode = *p++<<24;
		keycode |= *p++<<16;
		keycode |= *p++<<8;
		keycode |= *p++;
		keyb_single_char_received(tty,keycode);
	}
}


static int keyb_receive_room(struct tty_struct *tty)
{
	return -1;
}

#ifdef CONFIG_PM
static struct pm_dev *keyb_pm;
static int keyb_pm_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	volatile unsigned long *imrp;

	struct keyb_info *pInfo=0;

	switch (request) {
	case PM_SUSPEND:
		/// disable interrupt
		imrp = (volatile unsigned long *) (MCF_IPSBAR + MCFICM_INTC0 + MCFINTC_IMRH);
		*imrp |= (1 << (MCFINT_PIT2 - 32));
		break;

	case PM_RESUME:
		/// reprogramm timer after powerfail
		/* init PIT2 for keyboard scan */
		init_keyboard_timer();
		if(v_tty_data[1].tty) {
			pInfo=(struct keyb_info*) v_tty_data[1].tty->disc_data;
			keyb_write_char(pInfo,SKEY_POWERFAIL,0);
		}
		break;
	}
	return 0;
}
#endif

#if defined(CONFIG_SAG_PR5800_CORE)\
 || defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)\
 || defined(CONFIG_SAG_MMA70_CORE)
static void init_keyboard_timer(void)
{
	volatile unsigned char *icrp;
	volatile unsigned long *imrp;
	volatile struct mcfpit *tp;

	/* init PIT2 for keyboard scan */
	/* Set up PIT timer 1 as poll clock */
	tp = (volatile struct mcfpit *) (MCF_IPSBAR + MCFPIT_BASE2);
	tp->pcsr = MCFPIT_PCSR_DISABLE;

#if defined(CONFIG_M527x)
	request_irq(64+MCFINT_PIT2, pit2_intr, SA_INTERRUPT, "ColdFire Timer2", NULL);
#endif
#if defined(CONFIG_M528x)
	request_irq(127, pit2_intr, SA_INTERRUPT, "ColdFire Timer2", NULL);
#endif
	icrp = (volatile unsigned char *) (MCF_IPSBAR + MCFICM_INTC0 +
		MCFINTC_ICR0 + MCFINT_PIT2);
	*icrp = 053; /* (octal) PIT1 with level 5, priority 3 */

	tp->pmr = ((MCF_CLK / 2) / 64) / (1000);	/* 1 msec interrupt */
	tp->pcsr = MCFPIT_PCSR_EN | MCFPIT_PCSR_PIE | MCFPIT_PCSR_OVW |
		MCFPIT_PCSR_RLD | MCFPIT_PCSR_CLK64;

	imrp = (volatile unsigned long *) (MCF_IPSBAR + MCFICM_INTC0 + MCFINTC_IMRH);
	*imrp &= ~(1 << (MCFINT_PIT2 - 32));

}
#endif

/**
*   create the virtual tty (0..3) driver with the special line discipline N_SAGKEYB
*/
static int __init combix_keyb_init(void)
{
	struct kb_private *info;
	int status;
	int i;
	memset(&v_tty_data,0,sizeof(v_tty_data));
	vt_serial_driver = alloc_tty_driver(MAX_NR_VTTY);
	if (!vt_serial_driver)
		return -ENOMEM;
	/* Initialize the tty_driver structure */
	vt_serial_driver->owner = THIS_MODULE;
	vt_serial_driver->name = "tty";
	vt_serial_driver->devfs_name = "tty/";
	vt_serial_driver->driver_name = combix_name;
	vt_serial_driver->major = TTY_MAJOR;
	vt_serial_driver->minor_start = 0;
	vt_serial_driver->type = TTY_DRIVER_TYPE_SERIAL;
	vt_serial_driver->subtype = SERIAL_TYPE_NORMAL;
	vt_serial_driver->init_termios = tty_std_termios;

	vt_serial_driver->init_termios.c_cflag =
		B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	vt_serial_driver->flags = TTY_DRIVER_REAL_RAW;

	tty_set_operations(vt_serial_driver, &vt_ops);

	if (tty_register_driver(vt_serial_driver)) {
		printk("vtty: Couldn't register serial driver\n");
		put_tty_driver(vt_serial_driver);
		return(-EBUSY);
	}
	/*
	 *	Configure all the attached virtual serial ports.
	 */
	for (i = 0, info = v_tty_data; (i < MAX_NR_VTTY); i++, info++) {
		info = &v_tty_data[i];
		info->open_count=0;
		info->magic = SERIAL_MAGIC;
		INIT_WORK(&info->tqueue, vt_offintr, info);
	}
	printk("%s registered\n",combix_name);

#if defined(CONFIG_SAG_PR5800_CORE)
	/* init keyboard event handler */
	kbd_init();
#endif
	/* init PIT2 for keyboard scan */
#if defined(CONFIG_SAG_PR5800_CORE)\
 || defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)\
 || defined(CONFIG_SAG_MMA70_CORE)
	init_keyboard_timer();
#endif
	/*
	 * Register the special tty line discipline
	 */

	status = tty_register_ldisc (N_SAGKEYB, &tty_ldisc_N_SAGKEYB);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_SAGLKEYB);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_SAGKEYB.flags,
		        tty_ldisc_N_SAGKEYB.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_SAGKEYB.open);
		TRACE_L("tty_ldisc_N_SAGKEYB = %x", (int)&tty_ldisc_N_SAGKEYB);
	}
	else
	{
		printk (KERN_ERR "SAG keyboard: error registering line discipline: %d\n", status);
	}

#ifdef CONFIG_PM
     	keyb_pm = pm_register(PM_SYS_DEV, PM_SYS_COM, keyb_pm_callback);
	    if (keyb_pm)
		    keyb_pm->data = &v_tty_data[1];
#endif
	return 0;
}


module_init(combix_keyb_init);
MODULE_AUTHOR("Andreas.Horn@sartorius.com");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
