/***************************************************************************
                          sag_modbus.c  -  description
                             -------------------
    begin                : Mo Jan 10 2005
    copyright            : (C) 2005 by Sartorius Hamburg GmbH
    email                : andreas.horn@sartorius.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
** $Log: sag_modbus.c,v $
** Revision 1.11  2006/12/04 10:46:14  dec3ban
** minor fixes in modbus-driver: receive function 7, removed obsolete vars in private struct
**
** Revision 1.10  2005/10/20 11:42:58  dec3hon
** use a unsigned char in receive_single_char
**
** Revision 1.9  2005/10/04 13:13:39  dec3ban
** now making a source-bz2 from the diffs
**
** Revision 1.8  2005/09/21 10:49:28  dec3hon
** ICR reg now set in request timer
**
** Revision 1.7  2005/09/20 13:05:17  dec3hon
** return IRQ_HANDLED and check for wait queue is active
**
** Revision 1.6  2005/08/26 09:19:48  dec3hon
** add n_tty_ioctl call in ioctl
**
** Revision 1.5  2005/07/04 06:14:36  dec3hon
** allow id=0
**
** Revision 1.4  2005/06/23 07:52:08  dec3hon
** correction for timeout
**
** Revision 1.3  2005/02/24 08:47:29  dec3hon
** saglinux.kdevelop
**
** Revision 1.2  2005/02/21 06:53:00  dec3hon
** *** empty log message ***
**
** Revision 1.1  2005/02/17 06:22:49  dec3hon
** new driver
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/string.h>   /* used in new tty drivers */
#include <linux/ioctl.h>
#include <linux/sag_modbus.h>
#include <asm/mcf528x_dma_timer.h>

//#define DEBUG

#ifdef DEBUG
#define TRACE_L(format, args...) printk("xbpi: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif


static irqreturn_t dma_intr(int irq, void *dummy, struct pt_regs *fp);
static unsigned short make_crc(unsigned char c, unsigned last_crc);
static void on_timeout(unsigned long priv);
static void receive_char(struct modbus_info *pInfo, const unsigned char c);
static int  modbus_open(struct tty_struct *tty);
static void modbus_close(struct tty_struct *tty);
static ssize_t modbus_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr);
static ssize_t modbus_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr);
static int modbus_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void modbus_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void modbus_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char fp);
static int  modbus_receive_room(struct tty_struct *tty);
static void modbus_tty_wakeup(struct tty_struct *tty);

int __init modbus_init(void);
void __exit modbus_exit(void);

static struct tty_ldisc tty_ldisc_N_MODBUS =
    {
    .owner	 = THIS_MODULE,
    .magic	= TTY_LDISC_MAGIC,
    .name	= "MODBUS",
    .open	= modbus_open,
    .close	= modbus_close,
    .read	= modbus_read,
    .write	= modbus_write,
    .ioctl	= modbus_ioctl,
    .receive_buf = modbus_receive_buf,
    .receive_room = modbus_receive_room,
    .single_char_received=modbus_single_char_received,
    .write_wakeup= modbus_tty_wakeup

};

static void modbus_start_wait_receive(struct modbus_info *pInfo)
{
	pInfo->flags = 0;
	pInfo->rx_position = 0;
	if(pInfo->rx_timeout)
	{
		init_timer(&pInfo->tmr);
		pInfo->tmr.expires = jiffies + (pInfo->rx_timeout+(1000/HZ)-1)/(1000/HZ);
		add_timer(&pInfo->tmr);
	}
	/* now receive the next command */
	pInfo->state = RX_WAIT_SILENCE;
	/* start timer for 3,5 character wait */
	start_hrt_timer(pInfo->timer,pInfo->delay_35);
}


static void modbus_tty_wakeup(struct tty_struct *tty)
{
	struct modbus_info *pInfo=(struct modbus_info*)tty->disc_data;

	if (!pInfo)
		return;
	if(pInfo->tty->stopped)
		return;

	if(pInfo->state==TX_TRANSMITT)
	{
		pInfo->tty->flags &= ~(1<<TTY_DO_WRITE_WAKEUP);
		modbus_start_wait_receive(pInfo);
	}
}

static int function_lenght[]=
{
	0,		/* not used */
	4,	/* F01 bit  read	ADR_ADR NUM_NUM */
	4,	/* F02 bit  read	ADR_ADR NUM_NUM*/
	4,	/* F03 word read	ADR_ADR NUM_NUM*/
	4,	/* F04 word read	ADR_ADR NUM_NUM*/
	4,	/* F05 byte write	ADR_ADR VAL 000*/
	4,	/* F06 word write	ADR_ADR VAL_VAL*/
	0,	/* F07 high speed read*/
	4,	/* F08 diagnostic exchange*/
	-1,	/* F09 ---*/
	-1,	/* F10 ---*/
	0,	/* F11 read event counter*/
	0,	/* F12 read event history*/
	-1,	/* F13 ---*/
	-1,	/* F14 ---*/
	5+0x40,	/* F15 bit  write ADR_ADR NUM_NUM NUM bits*/
	5+0x40	/* F16 word write ADR_ADR NUM_NUM NUM words*/
};

// note Table is intel order!!
static const unsigned short crc_tab[]=
{
0x00000,0x0c1c0,0x081c1,0x04001,0x001c3,0x0c003,0x08002,0x041c2,
0x001c6,0x0c006,0x08007,0x041c7,0x00005,0x0c1c5,0x081c4,0x04004,
0x001cc,0x0c00c,0x0800d,0x041cd,0x0000F,0x0c1cF,0x081ce,0x0400e,
0x0000a,0x0c1ca,0x081cb,0x0400b,0x001c9,0x0c009,0x08008,0x041c8,
0x001d8,0x0c018,0x08019,0x041d9,0x0001b,0x0c1db,0x081da,0x0401a,
0x0001e,0x0c1de,0x081dF,0x0401F,0x001dd,0x0c01d,0x0801c,0x041dc,
0x00014,0x0c1d4,0x081d5,0x04015,0x001d7,0x0c017,0x08016,0x041d6,
0x001d2,0x0c012,0x08013,0x041d3,0x00011,0x0c1d1,0x081d0,0x04010,
0x001F0,0x0c030,0x08031,0x041F1,0x00033,0x0c1F3,0x081F2,0x04032,
0x00036,0x0c1F6,0x081F7,0x04037,0x001F5,0x0c035,0x08034,0x041F4,
0x0003c,0x0c1Fc,0x081Fd,0x0403d,0x001FF,0x0c03F,0x0803e,0x041Fe,
0x001Fa,0x0c03a,0x0803b,0x041Fb,0x00039,0x0c1F9,0x081F8,0x04038,
0x00028,0x0c1e8,0x081e9,0x04029,0x001eb,0x0c02b,0x0802a,0x041ea,
0x001ee,0x0c02e,0x0802F,0x041eF,0x0002d,0x0c1ed,0x081ec,0x0402c,
0x001e4,0x0c024,0x08025,0x041e5,0x00027,0x0c1e7,0x081e6,0x04026,
0x00022,0x0c1e2,0x081e3,0x04023,0x001e1,0x0c021,0x08020,0x041e0,
0x001a0,0x0c060,0x08061,0x041a1,0x00063,0x0c1a3,0x081a2,0x04062,
0x00066,0x0c1a6,0x081a7,0x04067,0x001a5,0x0c065,0x08064,0x041a4,
0x0006c,0x0c1ac,0x081ad,0x0406d,0x001aF,0x0c06F,0x0806e,0x041ae,
0x001aa,0x0c06a,0x0806b,0x041ab,0x00069,0x0c1a9,0x081a8,0x04068,
0x00078,0x0c1b8,0x081b9,0x04079,0x001bb,0x0c07b,0x0807a,0x041ba,
0x001be,0x0c07e,0x0807F,0x041bF,0x0007d,0x0c1bd,0x081bc,0x0407c,
0x001b4,0x0c074,0x08075,0x041b5,0x00077,0x0c1b7,0x081b6,0x04076,
0x00072,0x0c1b2,0x081b3,0x04073,0x001b1,0x0c071,0x08070,0x041b0,
0x00050,0x0c190,0x08191,0x04051,0x00193,0x0c053,0x08052,0x04192,
0x00196,0x0c056,0x08057,0x04197,0x00055,0x0c195,0x08194,0x04054,
0x0019c,0x0c05c,0x0805d,0x0419d,0x0005F,0x0c19F,0x0819e,0x0405e,
0x0005a,0x0c19a,0x0819b,0x0405b,0x00199,0x0c059,0x08058,0x04198,
0x00188,0x0c048,0x08049,0x04189,0x0004b,0x0c18b,0x0818a,0x0404a,
0x0004e,0x0c18e,0x0818F,0x0404F,0x0018d,0x0c04d,0x0804c,0x0418c,
0x00044,0x0c184,0x08185,0x04045,0x00187,0x0c047,0x08046,0x04186,
0x00182,0x0c042,0x08043,0x04183,0x00041,0x0c181,0x08180,0x04040
};


static int  modbus_open(struct tty_struct *tty)
{
	struct modbus_info * pInfo;
	pInfo=kmalloc(sizeof(struct modbus_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "modbus: failed to alloc info structure\n");
		return -ENOMEM;
	}
	/* request a dma timer */
	if(!(pInfo->timer=request_hrt_timer(&pInfo->intr_request_vector)))
	{
		printk("modbus: unable to get DMA timer\n");
		kfree(pInfo);
		return -EBUSY;
	}

	pInfo->rx_buf = kmalloc(RX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->rx_buf)
	{
		printk(KERN_ERR "modbus: failed to alloc receive buffer\n");
		kfree(pInfo);
		release_hrt_timer(pInfo->timer);
		return -ENOMEM;
	}

	pInfo->tx_buf = kmalloc(TX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->tx_buf)
	{
		printk(KERN_ERR "modbus: failed to alloc transmit buffer\n");
		kfree(pInfo->rx_buf);
		kfree(pInfo);
		release_hrt_timer(pInfo->timer);
		return -ENOMEM;
	}
	tty->disc_data = pInfo;
	pInfo->tty = tty;
	pInfo->state = UNREQUESTED;	// not ready to receive char yet
	init_timer(&pInfo->tmr);
	pInfo->tmr.data = (unsigned long)pInfo;
	pInfo->tmr.function = on_timeout;
	init_waitqueue_head(&pInfo->wait_for_state);
	/* request the interrupt for the dma timer */
	if(request_irq(pInfo->intr_request_vector+64, dma_intr, SA_INTERRUPT,
	               "dma interrupt (modbus)", tty) < 0 )
		printk( "%s(%d):Cant request interrupt\n",
		        __FILE__,__LINE__);

	return 0;
}


static void modbus_close(struct tty_struct *tty)
{
	struct modbus_info *pInfo=(struct r3964_info*)tty->disc_data;
	/* disable dms timer */
	pInfo->timer->dtxmr=0;	// no dma request on reference match
	release_hrt_timer(pInfo->timer);
	free_irq(pInfo->intr_request_vector+64, tty);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	del_timer_sync(&pInfo->tmr);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	kfree(pInfo->rx_buf);
	kfree(pInfo->tx_buf);
	kfree(pInfo);
}

static ssize_t modbus_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr)
{
	return 0;
}

static ssize_t modbus_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr)
{
	return 0;
}

/* 2 char timeout values in usec */
static int baud_delay[] =
    {
        0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
        9600, 19200, 38400, 57600, 115200, 230400, 460800, 0
    };

static int modbus_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	int i;
	int ret =0;
	char c;
	struct MODBUS_PARA *parg=(struct modbus_PARA *)arg;
	struct modbus_info *pInfo=(struct modbus_info*)tty->disc_data;
	struct MODBUS_PARA para;
	struct S_MCF_DMA_TIMER *dmatimer;
	if(cmd!=MODBUS_IO_TELEGRAMM)
		return  n_tty_ioctl (tty, file, cmd, arg);

	copy_from_user((void*)&para,(void*)arg,sizeof(struct MODBUS_PARA));


	if( para.isiz>255
	 || para.osiz>255
	 || (para.ibuf && para.isiz==0)	// fehler: input buffer ohne groesse
	 || (para.obuf && para.osiz==0)	// fehler: output buffer ohne groesse
	)
	{
		put_user(IO_PARA_ERR,&parg->status);
		return -EINVAL;
	}

	if (!tty->driver->tiocmset)
		return -EINVAL;

	put_user(IO_PROGRESS,&parg->status);	// indicate status io_progress

	pInfo->slv_adr = para.id ; 		/* set the slave id */

	i = tty->termios->c_cflag & CBAUD;
	if (i & CBAUDEX)
	{
		i &= ~CBAUDEX;
		if (i < 1 || i > 4)
			;
		else
			i += 15;
	}
	dmatimer = pInfo->timer;
	/* set 3,5 character timeout value in usec == 39 bits*/
	pInfo->delay_35 = (1000*1000*39)/baud_delay[i];
	pInfo->delay_15 = (1000*1000*28)/baud_delay[i];  /* this is the correct timeout */
	/* for baudrates > 19200 fix timeout values used */
	if(baud_delay[i]> 19200)
	{
		pInfo->delay_35 = 1750;
		pInfo->delay_15 = 750;
	}
	//set_hrt_timeout(dmatimer,pInfo->delay_35);
	stop_hrt_timer(dmatimer);
	switch(cmd)
	{
		case MODBUS_IO_TELEGRAMM:
			{
				pInfo->rx_buf_len=para.isiz;
				pInfo->rx_timeout=para.timo;
				/* if have a answer send first */
				if(para.obuf)
				{
					copy_from_user(pInfo->tx_buf,para.obuf,para.osiz);
					//
					// prepare data to send
					//
					pInfo->crc_calculated = 0xffff;
					for(i=0;i<para.osiz;i++)
					{
						c=pInfo->tx_buf[i];
						pInfo->crc_calculated = make_crc(c,pInfo->crc_calculated);
					}
					pInfo->tx_buf[i++]=pInfo->crc_calculated>>8;
					pInfo->tx_buf[i++]=pInfo->crc_calculated>>0;
					//--------------------------
					// wait for stand
					//--------------------------
					pInfo->flags = SUCCESS;
					pInfo->state = TX_WAIT_SILENCE;
					/* start timer for 3,5 character wait */
					start_hrt_timer(dmatimer,pInfo->delay_35);
					/* wait for dmatimer int */
					wait_event_interruptible(pInfo->wait_for_state,pInfo->state==TX_READY_TO_TRANSMITT);

					if (signal_pending(current)) {
						goto error;
					}
					pInfo->state=TX_TRANSMITT;
					pInfo->tty->flags |= (1<<TTY_DO_WRITE_WAKEUP);
					pInfo->tty->driver->write(pInfo->tty,pInfo->tx_buf,para.osiz+2);
				}
				else
				{
					modbus_start_wait_receive(pInfo);
				}
				/* wait for complete receive */
				wait_event_interruptible(pInfo->wait_for_state,pInfo->state==RX_RECEIVE_DONE);
				/* stop timer */
				stop_hrt_timer(dmatimer);
				if (signal_pending(current)) {
					goto error;
				}

				copy_to_user (para.ibuf,pInfo->rx_buf,pInfo->rx_position);
				if(pInfo->flags & SUCCESS)
					para.status = IO_SUCCESS;
				if(pInfo->flags & TIMEOUT)
				{
					para.status = IO_TIMEOUT;
					ret = -ENODATA;
				}
				if(pInfo->flags & RXCERR)
				{
error:
					pInfo->state=UNREQUESTED;
					para.status = IO_PHYSICAL_ERR;
					ret = -ENODATA;
				}
				put_user(pInfo->rx_position,&parg->icnt);
				put_user(para.status,&parg->status);
				break;
			}
		default:
			ret = -ENOIOCTLCMD;
			break;
	}
	/* zero our IPL */
	del_timer_sync(&pInfo->tmr);
	stop_hrt_timer(dmatimer);
	return ret;
}
static void on_timeout(unsigned long priv)
{
	struct modbus_info *pInfo = (void *)priv;
	pInfo->state = UNREQUESTED;
	pInfo->flags = TIMEOUT;
	stop_hrt_timer(pInfo->timer);
	if (waitqueue_active(&pInfo->wait_for_state))
		wake_up_interruptible(&pInfo->wait_for_state);
}

static unsigned short make_crc(unsigned char c, unsigned last_crc)
{
	unsigned index = ((last_crc>>8)^c)&0xff;
	return crc_tab[index]^(last_crc<<8);
}

static void receive_char(struct modbus_info *pInfo, const unsigned char c)
{
	struct S_MCF_DMA_TIMER *dmatimer;
	dmatimer = pInfo->timer;
	stop_hrt_timer(dmatimer);

	switch(pInfo->state)
	{
		case TX_READY_TO_TRANSMITT:
		case TX_TRANSMITT:
			// note: should we cancel?
			break;
		case TX_TRANSMITT_DONE:
		case RX_RECEIVE_DONE:
		case UNREQUESTED:
			break;			/* ignore character */
		// dma interrupt will switch to WAIT_ADDRESS state
		case TX_WAIT_SILENCE:
		case RX_WAIT_SILENCE:
			start_hrt_timer(dmatimer,pInfo->delay_35);
			break;
			/* receive the slave address */
		case RX_WAIT_ADDRESS:
			if(pInfo->slv_adr == c || c==0x00)
			{
				pInfo->rx_buf[0] = c;
				pInfo->rx_position=1;
				pInfo->add_length=0;
				pInfo->crc_calculated = 0xffff;
				pInfo->crc_calculated = make_crc(c,pInfo->crc_calculated);
				pInfo->state = RX_WAIT_FUNCTION;
				start_hrt_timer(dmatimer,pInfo->delay_15);
			}
			else
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
			}
			break;
		case RX_WAIT_FUNCTION:
			if( c > 16 || c < 1)
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
				return;
			}
			pInfo->toreceive = function_lenght[c];
			if(pInfo->toreceive == -1 )
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
				return;
			}
			if(pInfo->toreceive&0x40)
				pInfo->add_length=1;
			pInfo->toreceive = function_lenght[c]&0x3f;
			pInfo->rx_buf[pInfo->rx_position++] = c;
			if(pInfo->rx_position >= pInfo->rx_buf_len)
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
				printk("%s: too much chars received\n",__FILE__);
				return;
			}
			pInfo->crc_calculated = make_crc(c,pInfo->crc_calculated);
			if(pInfo->toreceive==0)
				pInfo->state = RX_CRC_LOW;
			else
				pInfo->state = RX_NEXT_CHAR;
			start_hrt_timer(dmatimer,pInfo->delay_35);	/* retrigger 3 char timeout */
			break;
		case RX_NEXT_CHAR:
			pInfo->rx_buf[pInfo->rx_position++] = c;
			if(pInfo->rx_position >= pInfo->rx_buf_len)
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
				printk("%s: too much chars received\n",__FILE__);
				return;
			}
			pInfo->crc_calculated = make_crc(c,pInfo->crc_calculated);
			pInfo->toreceive--;
			if(pInfo->toreceive==0)
			{
				/* any additonal bytes to read ? */
				if(pInfo->add_length)
				{
					pInfo->toreceive = c;
					pInfo->add_length=0;
				}
				else
					pInfo->state = RX_CRC_LOW;
			}
			start_hrt_timer(dmatimer,pInfo->delay_35);	/* retrigger 3 char timeout */
			break;
		case RX_CRC_LOW:
			pInfo->crc_received = 0;
			pInfo->crc_received = c<<8;
			pInfo->state = RX_CRC_HIGH;
			start_hrt_timer(dmatimer,pInfo->delay_35);	/* retrigger 3 char timeout */
			break;
		case RX_CRC_HIGH:
			pInfo->crc_received |= c;
			if(pInfo->crc_received != pInfo->crc_calculated)
			{
				pInfo->state = RX_WAIT_SILENCE;
				start_hrt_timer(dmatimer,pInfo->delay_35);
				return;
			}
			pInfo->flags = SUCCESS;
			if (waitqueue_active(&pInfo->wait_for_state))
			{
				pInfo->state = RX_RECEIVE_DONE;
				wake_up_interruptible(&pInfo->wait_for_state);	/* transfer finished */
			}
			else
			{
				pInfo->state = UNREQUESTED;
			}
			break;
	}
}

/* entry for receiver interrupt */
static void modbus_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char flags)
{
	struct modbus_info *pInfo=(struct modbus_info*)tty->disc_data;

	if(flags==TTY_NORMAL)
		receive_char(pInfo,cp);
	else if(cp ==0xff && flags == 0xff){	// we have had a powerfail
		reinit_hrt_timer(pInfo->timer);
		start_hrt_timer(pInfo->timer,pInfo->delay_35);
	}
	else	// char with error received
	{
		start_hrt_timer(pInfo->timer,pInfo->delay_35);
		pInfo->state = RX_WAIT_SILENCE;
	}
}
static void modbus_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
int i;
char *f;
const unsigned char *p;
char flags=0;
	for (i=count, p = cp, f = fp; i; i--, p++)
	{
		if (f)
			flags = *f++;
		modbus_single_char_received(tty,*cp++,flags);
	}
}




static int dma_intr(int irq, void *dev_id, struct pt_regs *fp)
{
	struct S_MCF_DMA_TIMER *dmatimer;
	struct modbus_info *pInfo=((struct tty_struct *)dev_id)->disc_data;
	dmatimer = pInfo->timer;
	stop_hrt_timer(dmatimer);
	switch(pInfo->state)
	{
		case UNREQUESTED:
			break;
		case TX_READY_TO_TRANSMITT:
		case TX_TRANSMITT:
		case TX_TRANSMITT_DONE:
		case RX_RECEIVE_DONE:
			break;

		case RX_WAIT_SILENCE:
			pInfo->tty->driver->tiocmset(pInfo->tty,0/*file*/,0/*set*/,TIOCM_RTS/*clear*/);
			pInfo->state = RX_WAIT_ADDRESS;
			break;
		case TX_WAIT_SILENCE:
			if (waitqueue_active(&pInfo->wait_for_state))
			{
				pInfo->state=TX_READY_TO_TRANSMITT;
				wake_up_interruptible(&pInfo->wait_for_state);
			}
			break;
			// timeout while receiving telegram
			// reset start with new timeout
		case RX_WAIT_ADDRESS:
		case RX_WAIT_FUNCTION:
		case RX_NEXT_CHAR:
		case RX_CRC_LOW:
		case RX_CRC_HIGH:
			pInfo->state = RX_WAIT_SILENCE;
			start_hrt_timer(dmatimer,pInfo->delay_35);
			break;
		//default:
			break;
	}
	return IRQ_HANDLED;
}


static int modbus_receive_room(struct tty_struct *tty)
{
	return -1;
}




/* init for compiled-in driver */
int __init modbus_init(void)
{
	int status;
	printk ("Sartorius Modbus driver $Revision: 1.11 $ installed\n");
	/*
	 * Register the tty line discipline
	 */

	status = tty_register_ldisc (N_MODBUS, &tty_ldisc_N_MODBUS);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_MODBUS);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_MODBUS.flags,
		        tty_ldisc_N_MODBUS.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_MODBUS.open);
		TRACE_L("tty_ldisc_N_MODBUS = %x", (int)&tty_ldisc_N_MODBUS);
	}
	else
	{
		printk (KERN_ERR "modbus: error registering line discipline: %d\n", status);
	}
	return status;
}
/* Cleanup - undid whatever init_module did */
void __exit modbus_exit(void)
{
	int status;


	status=tty_register_ldisc(N_MODBUS, NULL);

	if(status!=0)
	{
		printk(KERN_ERR "modbus: error unregistering linediscipline: %d\n", status);
	}
}

module_init(modbus_init);
module_exit(modbus_exit);

MODULE_LICENSE("GPL");
MODULE_ALIAS_LDISC(N_MODBUS);
