/************************************************************************/
/*                                                                      */
/*  mcf_qspi.c - QSPI driver for MCF5272                                */
/*                                                                      */
/*  (C) Copyright 2001, Wayne Roberts (wroberts1@home.com)              */
/*                                                                      */
/*  Driver has an 8bit mode, and a 16bit mode.                          */
/*  Transfer size QMR[BITS] is set thru QSPIIOCS_BITS.                  */
/*  When size is 8, driver works normally:                              */
/*        a char is sent for every transfer                             */
/*  When size is 9 to 16bits, driver reads & writes the QDRs with       */
/*  the buffer cast to unsigned shorts.  The QTR & QRR registers can    */
/*  be filled with up to 16bits.  The length passed to read/write must  */
/*  be of the number of chars (2x number of shorts). This has been      */
/*  tested with 10bit a/d and d/a converters.                           */
/*                                                                      */
/*  * QSPIIOCS_READDATA:                                                */
/*    data to send out during read                                      */
/*  * all other ioctls are global                                       */
/* -------------------------------------------------------------------- */
/*  Ported to linux-2.4.x by Ron Fial (ron@fial.com) August 26,2002     */
/*                                                                      */
/*   Added new include files                                            */
/*   Added module_init(),exit(),                                        */
/*   qspi_read(),qspi_write():  Revised qspi_read & write argument      */
/*     processing to handle new *filep argument. Changed i_rdev access  */
/*     to use filep->f_dentry->d_inode->i_rdev  Changed memcpy_fromfs() */
/*     to memcpy().                                                     */
/*   Added '__init' to compiled-in init routine for memory recovery     */
/*   Added '__exit' for loadable-driver module cleanup routine          */
/*   changed  register_chrdev to  devfs_register_chrdev                 */
/*   changed  unregister_chrdev to  devfs_unregister_chrdev             */
/*   Changed various declarations from int to ssize_t or loff_t         */
/* -------------------------------------------------------------------- */
/*   Changed interruptible_sleep_on to sleep_on so the driver has       */
/*           chance to finish the current transfer before application   */
/*           quits when typing '^C'. Otherwise a write collision will   */
/*           most likely occur.                                         */
/*   Added   safe_flags(); cli; and restore_flags() according to        */
/*           gerg@snapgear.com. Otherwise in some cases (higher clock   */
/*           rates) the transfer is finished before the current process */
/*           is put to sleep and therefore never wakes up again.        */
/*           09/12/2002 richard@opentcp.org                             */
/* -------------------------------------------------------------------- */
/*   02/06/2003 josef.baumgartner@telex.de                              */
/*                                                                      */
/*   Renamed cleanup_module() to qspi_exit() to be able to              */
/*     compile as module.                                               */
/*   Removed init_module() because module_init(qspi_init) does all      */
/*     we need.                                                         */
/*   Changed                                                            */
/*     SPI register settings will be saved for each instance to be able */
/*     to use different communication settings for different tasks.     */
/*     An ioctl() does not longer write directly to the SPI registers.  */
/*     It saves the settings which will be copied into the SPI          */
/*     registers on every read()/write().                               */
/*   Added MODULE_LICENSE("GPL") to avoid tainted kernel message.       */
/*     I think it is GPL?? There's no comment about this??              */
/*   Added polling mode                                                 */
/*     Increases performance for small data transfers.                  */
/*   Added odd mode                                                     */
/*     If an odd number of bytes is transfered and 16bit transfers are  */
/*     used, the last byte is transfered in byte mode.                  */
/*   Added dsp mode                                                     */
/*     If dsp mode is set, transfers will be limited to 15 bytes        */
/*     instead of 16. This ensures that DSPs with 24bit words get       */
/*     whole words within one transfer.                                 */
/* -------------------------------------------------------------------- */
/*   16/09/2003 ivan.zanin@bluewin.ch                                   */
/*                                                                      */
/*   Changed init and exit code to support the MCF5249                  */
/* -------------------------------------------------------------------- */
/************************************************************************/

/* **********************************************************************
Chapter 14. (excerpt) Queued Serial Peripheral Interface (QSPI) Module
   From:  http://e-www.motorola.com/brdata/PDFDB/docs/MCF5272UM.pdf

The following steps are necessary to set up the QSPI 12-bit data transfers
and a QSPI_CLK of 4.125 MHz. The QSPI RAM is set up for a queue of 16
transfers. All four QSPI_CS signals are used in this example.

1. Enable all QSPI_CS pins on the MCF5272. Write PACNT with 0x0080_4000 to
enable QSPI_CS1 and QSPI_CS3.Write PDCNT with 0x0000_0030 to enable QSPI_CS2.

2. Write the QMR with 0xB308 to set up 12-bit data words with the data
shifted on the falling clock edge, and a clock frequency of 4.125 MHz
(assuming a 66-MHz CLKIN).

3. Write QDLYR with the desired delays.

4. Write QIR with 0xD00F to enable write collision, abort bus errors, and
clear any interrupts.

5. Write QAR with 0x0020 to select the first command RAM entry.

6. Write QDR with 0x7E00, 0x7E00, 0x7E00, 0x7E00, 0x7D00, 0x7D00, 0x7D00,
0x7D00, 0x7B00, 0x7B00, 0x7B00, 0x7B00, 0x7700, 0x7700, 0x7700, and 0x7700
to set up four transfers for each chip select. The chip selects are active
low in this example.  NOTE: QDR value auto-increments after each write.

7. Write QAR with 0x0000 to select the first transmit RAM entry.

8. Write QDR with sixteen 12-bit words of data.

9. Write QWR with 0x0F00 to set up a queue beginning at entry 0 and ending
at entry 15.

10. Set QDLYR[SPE] to enable the transfers.

11.Wait until the transfers are complete. QIR[SPIF] is set when the
transfers are complete.

12. Write QAR with 0x0010 to select the first receive RAM entry.

13. Read QDR to get the received data for each transfer.  NOTE: QDR
auto-increments.

14. Repeat steps 5 through 13 to do another transfer.
---------------------------------------------------------------------
    02/06/2003 andreas.horn@sartorius.com
    adapted to kernel 2.6 and 5282

************************************************************************* */


#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/pm.h>


#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

/* for use with Sartorius boards
** some spi devices generates an interrupt for new data available
*/

#if defined(CONFIG_SAG_PR5800_CORE)
#define	MCFINT_ADC_IRQ5		5
#define	MCFINT_ADC_IRQ6		6
#endif
#if defined(CONFIG_SAG_COBRA_CORE)\
 || defined(CONFIG_SAG_LABPRO_CORE)
// interrupts from touch controller
#define	MCFINT_DAV_IRQ		4	// (pen) data available
//#define MCFINT_PENIRQ 	5	// PENIRQ (not used)
#define	MCFINT_KBD_IRQ		6	// keyboard
#endif

#if defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)
#define	MCFINT_ADC_IRQ6		6
#endif

#if defined(CONFIG_M5271_EVB)
#define	MCFINT_ADC_IRQ1		1
#endif

#define MCFINTC0_BASEVEC       64


/*   #if LINUX_VERSION_CODE > KERNEL_VERSION(2,2,0)  */
#include <asm/uaccess.h> /* for put_user */
/*    #endif   */


#include <linux/poll.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <linux/init.h>   /* added by Ron so driver builds directly into kernel */


#include <asm/io.h>
#include <asm/irq.h>

#include <linux/signal.h>
#include <linux/sched.h>
#include <asm/coldfire.h>   /* gets us MCF_MBAR value */
#if defined(CONFIG_M5249)
#include <asm/m5249sim.h>   /* gets us MCFSIM_ICR4, MCFSIM_PACNT */
#endif
#if defined(CONFIG_M528x)||defined(CONFIG_527x)
#include <asm/m528xsim.h>   /* gets us MCFSIM_ICR4, MCFSIM_PACNT */
#endif
#include <asm/mcfsim.h>
#include <asm/mcfedgeport.h>
#include "sag_qspi.h"

#define DEVICE_NAME "qspi"

int __init qspi_init(void);
int init(void);
void __exit qspi_exit(void);

enum E_CR
{
	CR_BITSE_8= 0x00,
	CR_BITSE_16=0x40,

	CR_CONT_NEXT=0x80,
	CR_CONT_STOP=0x00,

	CR_DT_1	 = 0x20,
	CR_DT_0  = 0x00,
	CR_DSCK_1= 0x10,
	CR_DSCK_0= 0x00
};

/* struct wait_queue *wqueue;   */
static DECLARE_WAIT_QUEUE_HEAD(wqueue);   /* use ver 2.4 static declaration - ron */
static int wqueue_int;
// static int cs;

#if defined(CONFIG_SAG_PR5800_CORE)
static DECLARE_WAIT_QUEUE_HEAD(iqueue5);
static DECLARE_WAIT_QUEUE_HEAD(iqueue6);
#endif
#if defined(CONFIG_SAG_MMA70_CORE)
static DECLARE_WAIT_QUEUE_HEAD(iqueue_touch);
#endif
#if defined(CONFIG_SAG_COBRA_CORE) || defined(CONFIG_SAG_LABPRO_CORE)
static DECLARE_WAIT_QUEUE_HEAD(iqueue_touch);
#endif
#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
static DECLARE_WAIT_QUEUE_HEAD(iqueue6);   /* use ver 2.4 static declaration - ron */
#endif
#if defined(CONFIG_M5271_EVB)
static DECLARE_WAIT_QUEUE_HEAD(iqueue1);   /* use ver 2.4 static declaration - ron */
#endif

static DECLARE_MUTEX(sem0);
static DECLARE_MUTEX(sem1);
static DECLARE_MUTEX(sem2);
static DECLARE_MUTEX(sem3);
static DECLARE_MUTEX(sem4);
static DECLARE_MUTEX(sem5);
static DECLARE_MUTEX(sem6);
static DECLARE_MUTEX(sem7);
static DECLARE_MUTEX(sem8);
static DECLARE_MUTEX(sem9);
static DECLARE_MUTEX(semA);
static DECLARE_MUTEX(semB);
static DECLARE_MUTEX(semC);
static DECLARE_MUTEX(semD);
static DECLARE_MUTEX(semE);
static DECLARE_MUTEX(semF);

#if LINUX_VERSION_CODE < 0x020100
static struct semaphore sem = MUTEX;
#else
static DECLARE_MUTEX(sem);
#endif

static DECLARE_MUTEX(ioctl_sem);

#if defined(CONFIG_M527x)
#define MCF_EPORT_EPPAR  *(volatile unsigned short*)(MCF_MBAR+MCF_EPPAR)
#define MCF_EPORT_EPFR   *(volatile unsigned char *)(MCF_MBAR+MCF_EPFR)
//#define MCF_EPORT_EPFR   *(volatile unsigned char *)(MCF_MBAR+MCF_EPFR)
#define MCF_EPORT_EPPDR	*(volatile unsigned char *)(MCF_MBAR+MCF_EPPDR)
#define MCF_EPORT_EPIER  *(volatile unsigned char *)(MCF_MBAR+MCF_EPIER)
#define MCF_INTC0_IMRL *(volatile unsigned long *)(MCF_MBAR+MCFICM_INTC0+MCFINTC_IMRL)
#define MCF_INTC0_ICR0 *(volatile unsigned char *)(MCF_MBAR + MCFICM_INTC0+ MCFINTC_ICR0)
#define MCF_INTC0_ICR(x) *(volatile unsigned char *)(MCF_MBAR + MCFICM_INTC0+ MCFINTC_ICR0 + x)
#endif

#if defined(CONFIG_SAG_MMA70_CORE) \
 || defined(CONFIG_SAG_COBRA_CORE) \
 || defined(CONFIG_SAG_LABPRO_CORE)
//------------------------------------
// any touch interrupt
//------------------------------------
unsigned touch_pending=0;
irqreturn_t touch_intr(int irq, void *data, struct pt_regs *fp)
{
	MCF_EPORT_EPFR = 1<<(irq-64); /* reset interrupt */
	touch_pending|=1<<(irq-64);
	if (waitqueue_active(&iqueue_touch))
		wake_up_interruptible(&iqueue_touch);
	return IRQ_HANDLED;
}
#endif

#if defined(CONFIG_SAG_PR5800_CORE)\
 || defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)
irqreturn_t irqx_intr(int irq, void *data, struct pt_regs *fp)
{
	wait_queue_head_t *iqueue = (wait_queue_head_t *)data;
	MCF_EPORT_EPFR = 1<<(irq-64); /* reset interrupt */
	MCF_INTC0_IMRL|=1<<(irq-64); // disable interrupt
	if(iqueue)
		wake_up(iqueue);	/* transfer finished */
	return IRQ_HANDLED;
}
#endif

static void qspi_timer(unsigned long data)
{
	qspi_dev *dev = (qspi_dev *)data;
	int irq=dev->irq;
#if defined(MCFINT_DAV_IRQ)
	if(irq==MCFINT_DAV_IRQ)		// if waiting for touch
		touch_pending|=0x8000;	// set bit to exit wait_event
#endif
#if defined(CONFIG_SAG_PR5800_CORE)\
 || defined(CONFIG_SAG_PR5410_CORE)\
 || defined(CONFIG_SAG_PR5220_CORE)
	if(irq)
	{
		MCF_EPORT_EPFR = 1<<irq; /* reset interrupt */
		MCF_INTC0_IMRL|=1<<irq;  /* disable interrupt */
	}
#endif
	if(dev->iqueue)
		wake_up(dev->iqueue);	/* transfer finished */

}

int qspi_dev_internal_io_write_read(qspi_dev *dev,
	SPI_PARA *apara, mem_type_t memtype);

irqreturn_t qspi_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned short qir = QIR;
	if (qir & QIR_WCEF)
		printk("WCEF ");	/* write collison */
	if (qir & QIR_ABRT)
		printk("ABRT ");	/* aborted by clearing of QDLYR[SPE] */
	if (qir & QIR_SPIF)
	{
		wqueue_int=1;
		wake_up(&wqueue);	/* transfer finished */
	}
	QIR |= QIR_WCEF | QIR_ABRT | QIR_SPIF;	/* clear all bits */
	return IRQ_HANDLED;
}


void qspi_ioctl_lock(void)
{
	down(&ioctl_sem);
}
void qspi_ioctl_unlock(void)
{
	up(&ioctl_sem);
}

int qspi_start_wait(qspi_dev * dev, SPI_PARA *apara, mem_type_t memtype)
{
	UBYTE sync;

	if(memtype == MEM_TYPE_USER)
		get_user(sync, &apara->sync);
	else
		sync = apara->sync;

	if(sync && dev->irq)
	{
		int timeout_param;

		if(memtype == MEM_TYPE_USER)
			get_user(timeout_param,&apara->timeout);
		else
			timeout_param = apara->timeout;

		if(timeout_param)
		{
			// make it abs
			int timeout=timeout_param;
			if(timeout<0) timeout=-timeout;
			init_timer(&dev->timer);
			timeout=(timeout+1000/HZ-1)/(1000/HZ);
			dev->timer.expires = jiffies + timeout;	// + timeout
			dev->timer.data = (unsigned long)dev;
			dev->timer.function = &qspi_timer;	/* timer handler */
			add_timer(&dev->timer);
		}
		up (&sem);
		//save_flags(flag); cli();		// added according to gerg@snapgear.com
#if defined(MCFINT_DAV_IRQ)
		{
		// we wait for  (input is low) or (input went low)
#define TOUCH_BITS ((1<<MCFINT_DAV_IRQ)|(1<<MCFINT_KBD_IRQ))
#define TOUCH_BITS_LOW  ((MCF_EPORT_EPPDR & TOUCH_BITS)!=TOUCH_BITS)
			set_task_state(current,TASK_INTERRUPTIBLE);
			wait_event_interruptible(*(dev->iqueue), (TOUCH_BITS_LOW || touch_pending!=0));
			touch_pending=0;
		}
#else
		{
			int irq=dev->irq;
			MCF_EPORT_EPPAR &=~(0x03<<(irq*2));	// set level sensitive
			MCF_EPORT_EPFR   = 1<<(irq); /* reset interrupt */
			MCF_EPORT_EPIER |=(1<<irq);	// enable interrupt
			MCF_INTC0_IMRL  &=~((1<<irq)|1); // enable interrupt
			//interruptible_sleep_on(dev->iqueue);			// changed richard@opentcp.org
			//restore_flags(flag);			// added according to gerg@snapgear.com

			wait_event( *(dev->iqueue),(((MCF_EPORT_EPPDR)&(1<<irq))==0));

			//save_flags(flag); cli();		// added according to gerg@snapgear.com
			MCF_EPORT_EPFR  = 1<<(irq); /* reset interrupt */
			MCF_INTC0_IMRL |=1<<(irq); // disable interrupt
			//restore_flags(flag);			// added according to gerg@snapgear.com
		}
#endif

		down(&sem);	// will be upped by caller
		if(timeout_param)
		{
			if(timeout_param>0 && ! timer_pending(&dev->timer))
				timeout_param=0;
			del_timer(&dev->timer);
		}
		if(signal_pending(current))
			return -EINTR;
		if(timeout_param>0)	// timeout occured
			return -ETIMEDOUT;
	}
	return 1;
}
/**
 * spi_dev_ioctl
 * Set configuration bits in the device structure or create an array
 * of data to transfer during the next tranceive operation.
 * This function does not actually interface with any hardware, it is used
 * only for modifying the device structure. From kernel space this could
 * be done more efficiently by directly modifying the structure.
 * This function is not thread safe, qspi_mutex_up/down should be used.
 * @param dev device node containing transfer mode information
 * @param cmd IOCTL command see mcf_qspi.h
 * @param arg IOCTL arg
 * @return 0 on success
 */
int qspi_dev_ioctl(qspi_dev *dev, unsigned int cmd, unsigned long arg, mem_type_t memtype)
{
	int ret = 0;

#if CONFIG_ALL_QSPI
	int error;
	int devno;
	struct qspi_read_data *read_data;
#endif

#if CONFIG_SAG_QSPI
	if (cmd == QSPIIOCS_WRRD) {
		down(&sem);
		ret = qspi_start_wait(dev, (SPI_PARA*)arg, memtype);
		up(&sem);
		if (ret != 1)
			return ret;
	}
#endif
	qspi_ioctl_lock();
	if(dev->sem) {
		down(dev->sem);
	}
	down(&sem);
	switch (cmd) {
#if CONFIG_ALL_QSPI
		case QSPIIOCS_DOUT_HIZ:
			dev->dohie = arg ? 1 : 0;
			break;
		case QSPIIOCS_BITS:	/* set QMR[BITS] */
			if (((arg > 0) && (arg < 8)) || (arg > 16)){
				ret = -EINVAL;
				break;
			}
			dev->bits = arg;
			break;
		case QSPIIOCG_BITS:	/* get QMR[BITS] */
			*((int *)arg) = dev->bits;
			break;
		case QSPIIOCS_CPOL:	/* set SCK inactive state */
			dev->cpol = arg ? 1 : 0;
			break;
		case QSPIIOCS_CPHA:	/* set SCK phase, 1=rising edge */
			dev->cpha = arg ? 1 : 0;
			break;
		case QSPIIOCS_BAUD:	/* set SCK baud rate divider */
			if (arg > 255){
				ret = -EINVAL;
				break;
				}
			dev->baud = arg;
			break;
		case QSPIIOCS_QCD:	/* set start delay */
			if (arg > 127){
				ret = -EINVAL;
				break;
				}
			dev->qcd = arg;
			break;
		case QSPIIOCS_DTL:	/* set after delay */
			if (arg > 255){
				ret = -EINVAL;
				break;
				}
			dev->dtl = arg;
			break;
		case QSPIIOCS_CONT:	/* continuous CS asserted during transfer */
			dev->qcr_cont = arg ? 1 : 0;
			break;
		/* set dsp mode, used to limit transfer to 15 byte for 24bit DSPs */
		case QSPIIOCS_DSP_MOD:
			dev->dsp_mod  = arg ? 1 : 0;
			break;
		/* if an odd count of bytes is transfered, force the transfer of the last byte
		 * to byte mode, even if word mode is used */
		case QSPIIOCS_ODD_MOD:
			dev->odd_mod = arg ? 1 : 0;
			break;
		case QSPIIOCS_READDATA:	/* set data to be sent during reads */
			read_data = (qspi_read_data *)arg;
			error = verify_area(VERIFY_READ, read_data,
							sizeof(qspi_read_data));
			if (error){
				ret = error;
				break;
			}
			if (read_data->length > sizeof(read_data->buf)){
				ret = -EINVAL;
				break;
				}
			if (memtype == MEM_TYPE_USER){
				copy_from_user(&dev->read_data, read_data, sizeof(qspi_read_data));
			} else {
				memcpy(&dev->read_data, read_data, sizeof(qspi_read_data));
			}
			break;
		/* use driver in polling mode, increases performance for small transfers */
		case QSPIIOCS_POLL_MOD:
			dev->poll_mod = arg ? 1 : 0;
			break;
		case QSPIIOCS_PID:
			devno = dev->num;// ??? and than ??? //MINOR(file->f_dentry->d_inode->i_rdev);
			// if(devno==0) process_id[devno] = arg;
			break;
#endif
#if CONFIG_SAG_QSPI
		case QSPIIOCS_WRRD:
			ret = qspi_dev_internal_io_write_read(dev, (SPI_PARA*)arg, memtype);
			break;
#endif
		default:
			ret = -EINVAL;
			break;
	}

	up(&sem);
	if(dev->sem) {
		up(dev->sem);
	}
	qspi_ioctl_unlock();

	return ret;
}

/**
 * qspi_ioctl
 * Standard ioctl system call for qspi character devices
 * calls qspi_control protected by the qspi mutex.
 */
static int qspi_ioctl(struct inode *inode, struct file *file,
				unsigned int cmd, unsigned long arg)
{
	int ret;
	qspi_dev *dev = file->private_data;

	ret = qspi_dev_ioctl(dev, cmd, arg, MEM_TYPE_USER);

	return ret;
}


/**
 * qspi_dev_open
 * Create a QSPI device.
 * Configuration information is stored in this device structure, which
 * is used by the read and write calls, to dynamically change the SPI's
 * configuration(bitrate, CPOL, etc...)
 * The elements of the structure are modifyable via control calls.
 * This function allocates the space for the device with Kmalloc, so if
 * the device is not destroyed(qspi_destroy_device, it will result in a
 * memory leak.
 * @return a newly allocated and initialized qspi device
 */
qspi_dev *qspi_dev_open(int num)
{
	qspi_dev *dev;

	if ((dev = kmalloc(sizeof(qspi_dev), GFP_KERNEL)) == NULL) {
		return(NULL);
	}

	memset(dev, 0, sizeof(qspi_dev));

	/* set default values */
#if CONFIG_ALL_QSPI
	dev->read_data.length = 0;
	dev->read_data.buf[0] = 0;
	dev->read_data.buf[1] = 0;
	dev->read_data.loop = 0;
#endif
	dev->poll_mod = 0;		/* interrupt mode */
	dev->bits = 16;
	dev->baud = 160;		/* 200khz with 64Mhz system clock */
	dev->cpol = 0;
	dev->cpha = 0;
	dev->qcr_cont = 1;
	dev->dsp_mod = 0;               /* no DSP mode */
	dev->odd_mod = 0;               /* no ODD mode */
	dev->qcd = 17;
	dev->dtl = 1;
	/* set default values */
	dev->iqueue=0;
	dev->irq=0;	// 1..7
	dev->sem=0;
	dev->num = num;			/* device number */

#if defined(CONFIG_SAG_PR5800_CORE)
	switch(dev->num)
	{
	case 0:dev->irq=MCFINT_ADC_IRQ5;dev->iqueue=&iqueue5;break;
	case 2:dev->irq=MCFINT_ADC_IRQ6;dev->iqueue=&iqueue6;break;
	}
#endif
#if defined(CONFIG_SAG_MMA70_CORE)\
  ||defined(CONFIG_SAG_LABPRO_CORE)
	switch(dev->num)
	{
	case 1:
		dev->irq=MCFINT_DAV_IRQ;
		dev->iqueue=&iqueue_touch;
		touch_pending=0;
		enable_edgeport(MCFINT_DAV_IRQ,FALLING_MODE,touch_intr);	// touch dev
		enable_edgeport(MCFINT_KBD_IRQ,FALLING_MODE,touch_intr);		// keypad
		break;
	}
#endif
#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
	switch(dev->num)
	{
	//   0111b
	case 7:dev->irq=MCFINT_ADC_IRQ6;dev->iqueue=&iqueue6;break;
	}
#endif
#if defined(CONFIG_M5271_EVB)
	// JJvB : this assumes M5271EVB
	// we have one earom and adc, all other generate error
	switch(dev->num)
	{
		case 13:
		case 14: break;
		default:
			return -ENOENT;
	}
	dev->irq=MCFINT_ADC_IRQ1;dev->iqueue=&iqueue1;
#endif
	switch(dev->num)
	{
	case 0:dev->sem=&sem0;break;
	case 1:dev->sem=&sem1;break;
	case 2:dev->sem=&sem2;break;
	case 3:dev->sem=&sem3;break;
	case 4:dev->sem=&sem4;break;
	case 5:dev->sem=&sem5;break;
	case 6:dev->sem=&sem6;break;
	case 7:dev->sem=&sem7;break;
	case 8:dev->sem=&sem8;break;
	case 9:dev->sem=&sem9;break;
	case 10:dev->sem=&semA;break;
	case 11:dev->sem=&semB;break;
	case 12:dev->sem=&semC;break;
	case 13:dev->sem=&semD;break;
	case 14:dev->sem=&semE;break;
	case 15:dev->sem=&semF;break;
	}

	return dev;
}


static int qspi_open(struct inode *inode, struct file *file)
{
	int devno;
	qspi_dev *dev;

	devno = MINOR(file->f_dentry->d_inode->i_rdev);
	if ((dev = qspi_dev_open(devno)) == NULL) {
		return -ENOMEM;
	}

	file->private_data = dev;

	dev->bits = 16;
	dev->baud = 160;		/* 200khz with 64Mhz system clock */

//	printk("spi-open(%d->i=%d)\n",devno,dev->irq);
	return 0;
}


/**
 * qspi_dev_release
 * free a previously created qspi device
 * @param device the device to destroy
 */
int qspi_dev_release(qspi_dev *dev)
{
#if defined(MCFINT_DAV_IRQ)
	if(dev->num==1)
	{
		disable_edgeport(MCFINT_DAV_IRQ);
		disable_edgeport(MCFINT_KBD_IRQ);
	}
#endif
	kfree(dev);

	return 0;
}


static int qspi_release(struct inode *inode, struct file *file)
{
	int ret;
	qspi_dev *dev = file->private_data;
// 	int devno;
// 	devno = MINOR(file->f_dentry->d_inode->i_rdev);
	ret = qspi_dev_release(dev);

	return ret;
}

//   basic 2.4 kernel function format
// ssize_t qspi_read(struct file* w12f, char * w12c, size_t w12d, loff_t * w12e) { ; }

#if CONFIG_ALL_QSPI

static ssize_t qspi_read(
	struct file * filep,
	char * buffer,
	size_t length,
	loff_t *off)

{
#if  defined(CONFIG_SAG_PR5800_CORE) \
  ||  defined(CONFIG_SAG_PR5410_CORE) \
  || defined(CONFIG_SAG_PR5220_CORE)
	return -ENOSYS;
#else
	int qcr_cs, total = 0, i = 0, max_trans;
        unsigned char bits, word = 0;
	unsigned long	flag;
        qspi_dev *dev;
        int rdi = 0;

        down(&sem);

        dev = filep->private_data;

        /* set the register with default values */
	QMR = QMR_MSTR |
		(dev->dohie << 14) |
		(dev->bits << 10) |
		(dev->cpol << 9) |
		(dev->cpha << 8) |
		(dev->baud);

	QDLYR = (dev->qcd << 8) | dev->dtl;

	if (dev->dsp_mod)
		max_trans = 15;
	else
		max_trans = 16;

        qcr_cs = (~MINOR(filep->f_dentry->d_inode->i_rdev) << 8) & 0xf00;  /* CS for QCR */
//	qcr_cs = 0;  // temporarilly force CS0 only, until can check minor device number processing!!

	bits = dev->bits % 0x10;
	if (bits == 0 || bits > 0x08)
		word = 1; /* 9 to 16bit transfers */

//		printk("\n READ driver -- ioctl xmit data fm dev->read_data.buf array  %x %x %x %x \n",dev->read_data.buf[0],dev->read_data.buf[1],dev->read_data.buf[2],dev->read_data.buf[3]);

        while (i < length) {
                unsigned short *sp = (unsigned short *)&buffer[i];
                unsigned char *cp = &buffer[i];
                unsigned short *rd_sp = (unsigned short *)dev->read_data.buf;
                int x, n;

                QAR = TX_RAM_START;             /* address first QTR */
                for (n=0; n<max_trans; n++) {
                        if (rdi != -1) {
                                if (word) {
                                        QDR = rd_sp[rdi++];
                                        if (rdi == dev->read_data.length>>1)
                                                rdi = dev->read_data.loop ? 0 : -1;
                                } else {
                                        QDR = dev->read_data.buf[rdi++];
                                        if (rdi == dev->read_data.length)
                                                rdi = dev->read_data.loop ? 0 : -1;
                                }
                        } else
                                QDR = 0;

                        i++;
                        if (word)
                                i++;
                        if (i > length) break;
                }
                QAR = COMMAND_RAM_START;        /* address first QCR */
                for (x=0; x<n; x++) {
                        /* QCR write */
                        if (dev->qcr_cont) {
                                if (x == n-1 && i == length)
                                        QDR = QCR_SETUP | qcr_cs;       /* last transfer */
                                else
                                        QDR = QCR_CONT | QCR_SETUP | qcr_cs;
                        } else
                                QDR = QCR_SETUP | qcr_cs;
                }

		QWR = QWR_CSIV | ((n-1) << 8);

		/* check if we are using polling mode. Polling increases
		 * performance for samll data transfers but is dangerous
		 * if we stay too long here, locking other tasks!!
		 */
    		if (dev->poll_mod){
			QIR = QIR_SETUP_POLL;
			QDLYR |= QDLYR_SPE;

			while ((QIR & QIR_SPIF) != QIR_SPIF)
				;
			QIR = QIR | QIR_SPIF;
		} else {
        		QIR = QIR_SETUP;
			save_flags(flag); cli();		// like in write function

			QDLYR |= QDLYR_SPE;
//			interruptible_sleep_on(&wqueue);
			sleep_on(&wqueue);			// changed richard@opentcp.org
			restore_flags(flag);			// like in write function

		}

                QAR = RX_RAM_START;     /* address: first QRR */
                if (word) {
                        /* 9 to 16bit transfers */
                        for (x=0; x<n; x++) {
                               put_user(QDR,sp++);
                        }
                } else {
                        /* 8bit transfers */
                        for (x=0; x<n; x++)
                                put_user(QDR,cp++);
                }

                if (word)
                        n <<= 1;

                total += n;
        }

        up(&sem);

        return total;
#endif
}



static int qspi_write(
        struct file * filep,
        const char * buffer,
        size_t length,
        loff_t *off)
{
#if  defined(CONFIG_SAG_PR5800_CORE) \
  ||  defined(CONFIG_SAG_PR5410_CORE)\
  || defined(CONFIG_SAG_PR5220_CORE)
	return -ENOSYS;
#else
	int	qcr_cs, i = 0, total = 0;
	static char dbuf[1024];
	int	z, max_trans;
	unsigned char	bits, word = 0;
	unsigned long	flag;
	qspi_dev *dev;

	down(&sem);

	dev = filep->private_data;

	QMR = QMR_MSTR |
		(dev->dohie << 14) |
		(dev->bits << 10) |
		(dev->cpol << 9) |
		(dev->cpha << 8) |
		(dev->baud);

	QDLYR = (dev->qcd << 8) | dev->dtl;

	bits = (QMR >> 10) % 0x10;
	if (bits == 0 || bits > 0x08)
		word = 1;	/* 9 to 16bit transfers */

	qcr_cs = (~MINOR(filep->f_dentry->d_inode->i_rdev) << 8) & 0xf00;	/* CS for QCR */
                                  /* next line was memcpy_fromfs()  */
//	qcr_cs = 0; // for testing, to force CS0 always
	copy_from_user (dbuf, buffer, length);

//	printk("data to write is %x  %x  %x  %X  \n",dbuf[0],dbuf[1],dbuf[2],dbuf[3]);

	if (dev->odd_mod)
		z = QCR_SETUP8;
	else
		z = QCR_SETUP;

	if (dev->dsp_mod)
		max_trans = 15;
	else
		max_trans = 16;

	while (i < length) {
		int x, n;
		QAR = TX_RAM_START;		/* address: first QTR */
		if (word) {
			for (n=0; n<max_trans; ) {
			    	/* in odd mode last byte will be transfered in byte mode */
				if (dev->odd_mod && (i + 1 == length)) {
					QDR = dbuf[i];  /* tx data: QDR write */
					// printk("0x%X ", dbuf[i]);
					n++;
					i++;
					break;
				} else {
					QDR = (dbuf[i] << 8) + dbuf[i+1]; /* tx data: QDR write */
					//printk("0x%X 0x%X ", dbuf[i], dbuf[i+1]);
					n++;
					i += 2;
					if (i >= length) break;
					}
			}
		} else {
			/* 8bit transfers */
			for (n=0; n<max_trans; ) {
				QDR = dbuf[i];	/* tx data: QTR write */
				n++;
				i++;
				if (i == length) break;
			}
		}
		QAR = COMMAND_RAM_START;	/* address: first QCR */
		for (x=0; x<n; x++) {
			/* QCR write */
			if (dev->qcr_cont) {
				if (x == n-1 && i == length)
					if ((i % 2)!= 0)
						QDR = z | qcr_cs; /* last transfer and odd number of chars */
                                        else
                                            	QDR = QCR_SETUP | qcr_cs;	/* last transfer */
				else
					QDR = QCR_CONT | QCR_SETUP | qcr_cs;
			} else {
				if (x == n-1 && i == length)
					QDR = z | qcr_cs; /* last transfer */
                                else
					QDR = QCR_SETUP | qcr_cs;
			}
		}

		QWR = QWR_CSIV | ((n-1) << 8);	/* QWR[ENDQP] = n << 8 */

		/* check if we are using polling mode. Polling increases
		 * performance for samll data transfers but is dangerous
		 * if we stay too long here, locking other tasks!!
		 */
		if (dev->poll_mod){
			QIR = QIR_SETUP_POLL;
			QDLYR |= QDLYR_SPE;

			while ((QIR & QIR_SPIF) != QIR_SPIF)
				;
			QIR = QIR | QIR_SPIF;
		} else {
			QIR = QIR_SETUP;
			save_flags(flag); cli();		// added according to gerg@snapgear.com
			QDLYR |= QDLYR_SPE;

//			interruptible_sleep_on(&wqueue);
			sleep_on(&wqueue);				// changed richard@opentcp.org

			restore_flags(flag);			// added according to gerg@snapgear.com
		}


		if (word)
			n <<= 1;

		total += n;
	}

	up(&sem);

	return total;
#endif
}
#endif

int qspi_dev_io_write_read(
	qspi_dev * dev,
	SPI_PARA *apara, mem_type_t memtype)
{
	int ret = 0;

	down (&sem);
	ret = qspi_dev_internal_io_write_read(dev, apara, memtype);
	up (&sem);

	return ret;
}

/**
 * qspi_internal_io_write_read
 * write an array of data to the queued SPI bus
 * @param dev device node containing transfer mode information
 * @param apara the data to send and receive
 * @return 1 on success
 */
int qspi_dev_internal_io_write_read(
        qspi_dev * dev,
	SPI_PARA *apara, mem_type_t memtype)
{
	unsigned qrw;
	int n;
	int newqp;
	int endqp;
	SPI_PARA para;
    unsigned long timeout;

//	down(&sem);
asm(" .global stop_here;stop_here:");

// 	dev = filep->private_data;

	if (memtype == MEM_TYPE_USER) {
		copy_from_user(&para, apara, sizeof(SPI_PARA));
	} else {
		memcpy(&para, apara, sizeof(SPI_PARA));
	}

#if defined(CONFIG_SAG_PR5410_CORE)
	// set additional cs
	*(volatile unsigned char *)(MCF_MBAR + MCF_GPIO_PORTB) &= ~0x60;
	*(volatile unsigned char *)(MCF_MBAR + MCF_GPIO_PORTB) |=  (para.sr<<5);
#endif

	QMR = QMR_MSTR |	// must be 1 always
		(dev->dohie << 14) |
		(0/*dev->bits*/ << 10) |	// alsways assume 8/16
//		(dev->cpol << 9) |
		(para.mode << 8) |
		(para.clock);

	QDLYR = (dev->qcd << 8) | dev->dtl;

                                  /* next line was memcpy_fromfs()  */
//	qcr_cs = 0; // for testing, to force CS0 always
//	printk("data to write is %x  %x  %x  %X  \n",dbuf[0],dbuf[1],dbuf[2],dbuf[3]);
	newqp=para.newqp;
	endqp=para.endqp;
	QAR = TX_RAM_START+newqp;		/* address: first QTR */
	for(n=newqp;n<=endqp;++n) {
		QDR = para.wr[n];
	}
	QAR = COMMAND_RAM_START+newqp;		/* address: first QCR */

/*	if (cs != (para.cr[0]&0x07)) {
		cs = (para.cr[0]&0x07);
		printk("%d\n",cs);
	}
*/
	for(n=newqp;n<=endqp;++n)
	{
		short cr;
		cr = para.cr[n];
/*		if (cs != (para.cr[0]&0x07)) {
			cs = (para.cr[0]&0x07);
			printk("%d\n",cs);
		}
*/
		QDR=cr<<8;
	}

	// JJvB: FIX to keep Chipselect low, if last cmd is CONT_NEXT
	//QWR = QWR_CSIV|(endqp<<8)|(newqp<<0);	/* QWR[ENDQP] = n << 8 */

	qrw=QWR_CSIV;
	if((para.cr[para.endqp]&(CR_CONT_NEXT|0x0f))==(CR_CONT_NEXT|0x0))
		qrw=0;
	QWR = qrw|(endqp<<8)|(newqp<<0);

	/* check if we are using polling mode. Polling increases
	* performance for samll data transfers but is dangerous
	* if we stay too long here, locking other tasks!!
	*/
	if (dev->poll_mod){
		QIR = QIR_SETUP_POLL;
		QDLYR |= QDLYR_SPE;

		while ((QIR & QIR_SPIF) != QIR_SPIF)
			;
		QIR = QIR | QIR_SPIF;
	} else {
		wqueue_int=0;
		QIR = QIR_SETUP;
		//save_flags(flag); cli();		// added according to gerg@snapgear.com
		QDLYR |= QDLYR_SPE;

		//interruptible_sleep_on(&wqueue);
//			sleep_on(&wqueue);			// changed richard@opentcp.org

		timeout = wait_event_timeout(wqueue,wqueue_int,HZ);

		if(timeout == 0)
        {
            printk("QSPI Timeout\n");
            // cancel transfer?
			QIR |= QIR_WCEF | QIR_ABRT | QIR_SPIF;	/* clear all bits */
			return -ETIMEDOUT;

        }

		//restore_flags(flag);			// added according to gerg@snapgear.com
		if(signal_pending(current))
		{
		// cancel transfer?
			QIR |= QIR_WCEF | QIR_ABRT | QIR_SPIF;	/* clear all bits */
			return -EINTR;
		}

	}

	QAR = RX_RAM_START+newqp;	/* address: first QCR */
	for(n=newqp;n<=endqp;++n) {
		if (memtype == MEM_TYPE_USER)
			put_user(QDR,&apara->rd[n]);
		else
			apara->rd[n] = QDR;
	}

	return 1;
}


struct file_operations Fops = {
	owner:		THIS_MODULE,
#if CONFIG_ALL_QSPI
	read:		qspi_read,
	write:		qspi_write,
#endif
	ioctl:		qspi_ioctl,
	open:		qspi_open,
	release:	qspi_release  /* a.k.a. close */
};

#ifdef CONFIG_PM
static int qspi_pm_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	volatile unsigned long *lp;
	volatile unsigned char *cp;

	switch (request) {
	case PM_SUSPEND:
		/* zero our IPL */
		*(volatile unsigned long *)(MCF_MBAR + MCFICM_INTC0 + MCFINTC_IMRL) |=
		~((1 << (MCFQSPI_IRQ_INT)));  /* disable qspi interrupt */
		break;

	case PM_RESUME:
		/* restore interrupt settings */
			/* setup interrupt in CPU */
		cp = (volatile unsigned char *)(MCF_MBAR + MCFICM_INTC0+ MCFINTC_ICR0 + MCFQSPI_IRQ_INT);
		*cp = 0x36;         /* autovector on, il=6, ip=6     */

		lp = (volatile unsigned long *)(MCF_MBAR + MCFSIM_PQSPAR);
		*lp &= 0xdc9FFFFF;  /* activate qspi_cs0 .. 3, qspi_dout, qspi_in, qspi_clk  */

		lp = (volatile unsigned long *)(MCF_MBAR + MCFICM_INTC0 + MCFINTC_IMRL);
		*lp &= ~((1 << (MCFQSPI_IRQ_INT)) | 1);  /* enable qspi interrupt */
		break;
	}
	return 0;
}
#endif

int init_mcf_qspi(void)
{
	/* common init: driver or module: */

	/* request interrupt vector from kernel */
	if (request_irq(MCFQSPI_IRQ_VECTOR, qspi_interrupt, SA_INTERRUPT, "ColdFire QSPI", NULL)) {
		printk("QSPI: Unable to attach ColdFire QSPI interrupt "
			"vector=%d\n", MCFQSPI_IRQ_VECTOR);
		return -EINVAL;
	}


#if defined(CONFIG_M528x) || defined(CONFIG_M527x)||defined(CONFIG_M532x)
	/* setup interrupt in CPU */
    MCF_INTC0_ICR(MCFQSPI_IRQ_INT) =066;         /* autovector on, il=6, ip=6 (octal)     */

#if defined(CONFIG_M528x) || defined(CONFIG_M527x)
    lp = (volatile unsigned long *)(MCF_MBAR + MCFSIM_PQSPAR);
    *lp &= 0xdc9FFFFF;  /* activate qspi_cs0 .. 3, qspi_dout, qspi_in, qspi_clk  */
#endif
#if defined(CONFIG_M532x)
	MCF_GPIO_PAR_QSPI = 0xFFF0;
#endif
    MCF_INTC0_IMRL &= ~((1 << (MCFQSPI_IRQ_INT)) | 1);  /* enable qspi interrupt */
#endif

	/* request additional interrupts for ADC-EndOfConversion */

#if defined(CONFIG_SAG_PR5800_CORE)
	if(request_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ5, irqx_intr, SA_INTERRUPT,
		"irq5(adc1) interrupt", &iqueue5) < 0 )
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,MCFINT_ADC_IRQ5);

	if(request_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ6, irqx_intr, SA_INTERRUPT,
		"irq6(adc2) interrupt", &iqueue6) < 0 )
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,MCFINT_ADC_IRQ6);
#endif

#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
	if(request_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ6, irqx_intr, SA_INTERRUPT,
		"irq6(adc) interrupt", &iqueue6) < 0 )
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,MCFINT_ADC_IRQ6);
#endif
#if defined(CONFIG_M5271_EVB)
	if(request_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ1, irqx_intr, SA_INTERRUPT,
		"irq1(adc71) interrupt", &iqueue1) < 0 )
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,MCFINT_ADC_IRQ1);
#endif

/*
 *  These values have to be setup according to the applications
 *  using the qspi driver. Maybe some #defines at the beginning
 *  would be more appropriate. Especially the transfer size
 *  and speed settings
 */

	QMR = 0x81A2;// default mode setup: 16 bits, baud, 160kHz clk.
	QDLYR = 0x0202;	// default start & end delays

	init_waitqueue_head(&wqueue);    /* was init_waitqueue() --Ron */
#if defined(CONFIG_SAG_PR5800_CORE)
	init_waitqueue_head(&iqueue5);    /* was init_waitqueue() --Ron */
	init_waitqueue_head(&iqueue6);    /* was init_waitqueue() --Ron */
#endif
#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
	init_waitqueue_head(&iqueue6);    /* was init_waitqueue() --Ron */
#endif
#if defined(CONFIG_M527x_EVB)
	init_waitqueue_head(&iqueue1);    /* was init_waitqueue() --Ron */
#endif
#if defined(CONFIG_SAG_MMA70_CORE) \
 || defined(CONFIG_SAG_COBRA_CORE) \
 || defined(CONFIG_SAG_LABPRO_CORE)
	init_waitqueue_head(&iqueue_touch);    /* was init_waitqueue() --Ron */
#endif
#ifdef CONFIG_PM
     	pm_register(PM_SYS_DEV, PM_SYS_COM, qspi_pm_callback);
#endif
	return 0;
}

/* init for compiled-in driver: call from mem.c */
int __init qspi_init()                                  /* the __init added by ron  */
{
	if (register_chrdev(QSPI_MAJOR, DEVICE_NAME, &Fops)) {
		printk(KERN_NOTICE "Can't allocate major number %d for SPI devices.\n",
		       QSPI_MAJOR);
		return -EAGAIN;;
	}
        printk ("Sartorius QSPI device driver installed\n");
	return init_mcf_qspi();
}

/* Cleanup - undid whatever init_module did */
void __exit qspi_exit()      /* the __exit added by ron  */
{
	int ret=0;
	free_irq(MCFQSPI_IRQ_VECTOR, NULL);
#if defined(CONFIG_SAG_PR5800_CORE)
        free_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ5,0);
        free_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ6,0);
#endif
#if defined(MCFINT_DAV_IRQ)
        free_irq(MCFINTC0_BASEVEC+MCFINT_DAV_IRQ,0);
        free_irq(MCFINTC0_BASEVEC+MCFINT_KBD_IRQ,0);
#endif
#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
        free_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ6,0);
#endif
#if defined(CONFIG_M527x_EVB)
        free_irq(MCFINTC0_BASEVEC+MCFINT_ADC_IRQ1,0);
#endif
#if  defined(CONFIG_M5249)
        /* autovector on, il=0, ip=0 */
	*(volatile unsigned char *)(MCF_MBAR + MCFSIM_ICR10) = 0x80;
	/* disable qspi interrupt */
	*(volatile unsigned long *)(MCF_MBAR + MCFSIM_IMR) |= 0x00040000;
#endif
#if  defined(CONFIG_M528x)||  defined(CONFIG_M527x)||defined(CONFIG_M532x)
	/* zero our IPL */
	MCF_INTC0_IMRL |= ~((1 << (MCFQSPI_IRQ_INT)));  /* disable qspi interrupt */
#else
	/* zero our IPL */
	*(volatile unsigned long *)(MCF_MBAR + MCFICM_INTC0 + MCFINTC_IMRL) |=
	     ~((1 << (MCFQSPI_IRQ_INT)));  /* disable qspi interrupt */
#endif

	/* Unregister the device */
	if ((unregister_chrdev(QSPI_MAJOR, DEVICE_NAME)) < 0)
		printk("Error in unregister_chrdev: %d\n", ret);
}

#if defined(CONFIG_SAG_PR5800_CORE) \
 || defined(CONFIG_SAG_PR5410_CORE) \
 || defined(CONFIG_SAG_PR5220_CORE) \
 || defined(CONFIG_SAG_COBRA_CORE)  \
 || defined(CONFIG_SAG_LABPRO_CORE)


/*
** this is used for reading hw-address
** len must be even for this simple use
*/
int earom_rd(int cs,unsigned offs,unsigned len,void *data)
{
	unsigned long	flag;
	unsigned newqp=0;
	unsigned endqp=2+(len/2)-1;
	unsigned n;

	enum{DEF_MODE=3};
	enum{DEF_CLOCK=254};

#if defined(CONFIG_SAG_PR5410_CORE)|| defined(CONFIG_SAG_PR5220_CORE)
	offs+=0x400;		/* always reading high rom */
#endif
	QMR = QMR_MSTR|(0<<14)|(0<<10)|(DEF_MODE << 8)|(DEF_CLOCK);

	QDLYR = (17 << 8) | 1;
	//--------------------
	// write data
	//--------------------
	QAR = TX_RAM_START+newqp;		/* address: first QTR */
	QDR = 0x03;	// Read Data from Memory Array
	QDR = offs;	// Set address
	for(n=0;n<len;n+=2)
		QDR=0;	// dummy data
	//--------------------
	// command
	//--------------------
	QAR = COMMAND_RAM_START+newqp;		/* address: first QCR */
	QDR = (CR_CONT_NEXT|CR_BITSE_8|cs)<<8;	/* byte command */
	QDR = (CR_CONT_NEXT|CR_BITSE_16|cs)<<8;	/* word offset */
	for(n=0;n+2<len;n+=2)
		QDR=(CR_CONT_NEXT|CR_BITSE_16|cs)<<8;
	QDR=(CR_CONT_STOP|CR_BITSE_16|cs)<<8;

	//------------------------
	// do the polling
	//-------------------------

	save_flags(flag); cli();		// added according to gerg@snapgear.com

	QWR = QWR_CSIV|(endqp<<8)|(newqp<<0);	/* QWR[ENDQP] = n << 8 */
	QIR = QIR_SETUP_POLL;
	QDLYR |= QDLYR_SPE;
	while ((QIR & QIR_SPIF) != QIR_SPIF)
		;
	//QIR = QIR | QIR_SPIF;
	QIR |= QIR_WCEF | QIR_ABRT | QIR_SPIF;	/* clear all bits */

	restore_flags(flag);			// added according to gerg@snapgear.com

	// read the data
	QAR = RX_RAM_START+newqp;	/* address: first QCR */
	n=QDR;	// dummy read
	n=QDR;	// dummy read
	for(n=0;n<len;n+=2)
		((short*)data)[n/2]=QDR;
	return 1;
}
#endif

module_init(qspi_init);
module_exit(qspi_exit);

MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("mcf_qspi");
MODULE_DESCRIPTION("Direct character-device access to SPI devices");
