/***************************************************************************
                          sag_rawio.c  -  description
                             -------------------
    begin                : Mo Jan 10 2005
    copyright            : (C) 2005 by Sartorius Hamburg GmbH
    email                : andreas.horn@sartorius.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
** $Log: sag_rawio.c,v $
** Revision 1.2  2006/01/26 11:59:35  dec3hon
** check for input request
**
** Revision 1.1  2006/01/25 09:11:24  dec3hon
** new ldisc sag_rawio
**
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/string.h>   /* used in new tty drivers */
#include <linux/ioctl.h>
#include <linux/sag_rawio.h>

//#define DEBUG

#ifdef DEBUG
#define TRACE_L(format, args...) printk("xbpi: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif


static void on_timeout(unsigned long priv);
static void receive_char(struct rawio_info *pInfo, const unsigned char c);
static int  rawio_open(struct tty_struct *tty);
static void rawio_close(struct tty_struct *tty);
static ssize_t rawio_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr);
static ssize_t rawio_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr);
static int rawio_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void rawio_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void rawio_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char fp);
static int  rawio_receive_room(struct tty_struct *tty);
static void put_char(struct rawio_info *pInfo, unsigned char ch);

int __init rawio_init(void);
void __exit rawio_exit(void);

static struct tty_ldisc tty_ldisc_N_RAWIO =
    {
    .owner	 = THIS_MODULE,
    .magic	= TTY_LDISC_MAGIC,
    .name	= "RAWIO",
    .open	= rawio_open,
    .close	= rawio_close,
    .read	= rawio_read,
    .write	= rawio_write,
    .ioctl	= rawio_ioctl,
    .receive_buf = rawio_receive_buf,
    .receive_room = rawio_receive_room,
    .single_char_received=rawio_single_char_received                                                                                                         };


static int  rawio_open(struct tty_struct *tty)
{
	struct rawio_info * pInfo;
	pInfo=kmalloc(sizeof(struct rawio_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "rawio: failed to alloc info structure\n");
		return -ENOMEM;
	}
	
	pInfo->rx_buf = kmalloc(RX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->rx_buf)
	{
		printk(KERN_ERR "rawio: failed to alloc receive buffer\n");
		kfree(pInfo);
		return -ENOMEM;
	}

	pInfo->tx_buf = kmalloc(TX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->tx_buf)
	{
		printk(KERN_ERR "rawio: failed to alloc transmit buffer\n");
		kfree(pInfo->rx_buf);
		kfree(pInfo);
		return -ENOMEM;
	}
	pInfo->timeout=0;
	tty->disc_data = pInfo;
	pInfo->tty = tty;
	pInfo->state = IDLE;
	init_timer(&pInfo->tmr);
	pInfo->tmr.data = (unsigned long)pInfo;
	pInfo->tmr.function = on_timeout;
	init_waitqueue_head(&pInfo->read_wait);
	return 0;
}


static void rawio_close(struct tty_struct *tty)
{
	struct rawio_info *pInfo=(struct r3964_info*)tty->disc_data;
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	del_timer_sync(&pInfo->tmr);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	kfree(pInfo->rx_buf);
	kfree(pInfo->tx_buf);
	kfree(pInfo);
}

static ssize_t rawio_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr)
{
	struct rawio_info *pInfo=(struct rawio_info*)tty->disc_data;
	if(!tty) return -ENODATA;
	pInfo->ilen = 0;
	pInfo->flags = 0;
	
	if(pInfo->timeout)
	{
		init_timer(&pInfo->tmr);
		pInfo->tmr.expires = jiffies +pInfo->timeout+(1000/HZ)-1/(1000/HZ);
		add_timer(&pInfo->tmr);
	}
	/* now receive the from input */
	pInfo->state = RECEIVING;
	interruptible_sleep_on(&pInfo->read_wait);
	del_timer_sync(&pInfo->tmr);
	if (signal_pending(current)) {
		return -EINTR;
	}
	if(pInfo->flags & TIMEOUT)
		return -ETIME;
	
	memcpy(buf,pInfo->rx_buf,pInfo->ilen);
	if(pInfo->flags & SUCCESS)
		return pInfo->ilen;
	return 0;
}

static ssize_t rawio_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr)
{
	int i;
	struct rawio_info *pInfo=(struct rawio_info*)tty->disc_data;
	if(!tty) return -ENODATA;
	for(i=0;i<nr;i++)
	{
		put_char(pInfo,buf[i]);
	}
	return 0;
}

static void put_char(struct rawio_info *pInfo, unsigned char ch)
{
	struct tty_struct *tty = pInfo->tty;

	if(tty==NULL)
		return;

	if(tty->driver->put_char)
	{
		tty->driver->put_char(tty, ch);
	}
}


static int rawio_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	int i;
	int ret =0;
	char c;
	struct RAWIO_PARA *parg=(struct RAWIO_PARA *)arg;
	struct rawio_info *pInfo=(struct rawio_info*)tty->disc_data;
	struct RAWIO_PARA para;
	if(cmd!=RAWIO_WRITEREAD)
		return  n_tty_ioctl (tty, file, cmd, arg);
	
	copy_from_user((void*)&para,(void*)arg,sizeof(struct RAWIO_PARA));
	

	if( para.isiz>255 
	 || para.osiz>255
	 || (para.ibuf && para.isiz==0)	// fehler: input buffer ohne groesse
	 || (para.obuf && para.osiz==0)	// fehler: output buffer ohne groesse
	)
	{
		put_user(IO_PARA_ERR,&parg->status);
		return -EINVAL;
	}
	
	if (!tty->driver->tiocmset)
		return -EINVAL;
	
	switch(cmd)
	{
		case RAWIO_WRITEREAD:
		{	
			para.status = IO_SUCCESS;
			copy_from_user(pInfo->tx_buf,para.obuf,para.osiz);
			copy_from_user(&pInfo->in_len,&para.isiz,2);
			copy_from_user(&pInfo->out_len,&para.osiz,2);
			copy_from_user(&pInfo->timeout,&para.timo,4);
			
			get_user(pInfo->startchar,&para.startchar);
			get_user(pInfo->stopchar,&para.stopchar);
			pInfo->flags = SUCCESS;
			/* if have something to write do it first */
			if(pInfo->out_len)
			{
				for(i=0;i<pInfo->out_len;i++)
				{
					c=pInfo->tx_buf[i];
					put_char(pInfo,c);
				}
			}
			if(pInfo->in_len)
			{
				pInfo->ilen = 0;
				pInfo->flags = 0;
				
				if(pInfo->timeout)
				{
					init_timer(&pInfo->tmr);
					pInfo->tmr.expires = jiffies +(para.timo+(1000/HZ)-1)/(1000/HZ);
					add_timer(&pInfo->tmr);
				}
				/* now receive the from input */
				if(pInfo->startchar)
					pInfo->state = WAIT_FORSTART;
				else
					pInfo->state = RECEIVING;
				interruptible_sleep_on(&pInfo->read_wait);
				if (signal_pending(current)) {
					if (tty->driver->flush_buffer)
						tty->driver->flush_buffer(tty);
					para.status = IO_PHYSICAL_ERR;
					ret = -ENODATA;
				}
				
				copy_to_user (para.ibuf,pInfo->rx_buf,pInfo->ilen);
				if(pInfo->flags & TIMEOUT)
				{
					/* remove not sent chars from buffer */
					if (tty->driver->flush_buffer)
						tty->driver->flush_buffer(tty);
					para.status = IO_TIMEOUT;
					ret = -ENODATA;
				}
				if(pInfo->flags & RXCERR)
				{
					para.status = IO_PHYSICAL_ERR;
					ret = -ENODATA;
				}
				put_user(pInfo->ilen,&parg->icnt);
			}
			put_user(para.status,&parg->status);
			break;
		}
		case RAWIO_SETTIMEOUT:
		{
			copy_from_user(&pInfo->timeout,&para.timo,4);
		}
		default:
			ret = -ENOIOCTLCMD;
			break;
	}
	del_timer_sync(&pInfo->tmr);
	return ret;
}
static void on_timeout(unsigned long priv)
{
	struct rawio_info *pInfo = (void *)priv;
	pInfo->flags = TIMEOUT;
	pInfo->state = IDLE;
	if (waitqueue_active(&pInfo->read_wait))
		wake_up_interruptible(&pInfo->read_wait);
}

static void receive_char(struct rawio_info *pInfo, const unsigned char c)
{
	switch(pInfo->state)
	{
		case IDLE:
			return;
		case WAIT_FORSTART:
			if(c != pInfo->startchar) return;
			pInfo->state = RECEIVING;
			break;
		case RECEIVING:
			pInfo->rx_buf[pInfo->ilen] = c;
			if(++pInfo->ilen == pInfo->in_len ||
			   (pInfo->stopchar && c == pInfo->stopchar) )
			{
				pInfo->flags = SUCCESS;
				pInfo->state = IDLE;
				wake_up_interruptible(&pInfo->read_wait);
			}
			break;
	}
}

/* entry for receiver interrupt */
static void rawio_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char flags)
{
	struct rawio_info *pInfo=(struct rawio_info*)tty->disc_data;
	
	if(flags==TTY_NORMAL)
		receive_char(pInfo,cp);
	else if(cp ==0xff && flags == 0xff){	// we have had a powerfail
		pInfo->flags = TIMEOUT;
		pInfo->state = IDLE;
		if (waitqueue_active(&pInfo->read_wait))
			wake_up_interruptible(&pInfo->read_wait);
	}
	else
		pInfo->flags = RXCERR;
}

static void rawio_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
int i; 
char *f;
const unsigned char *p;
char flags=0;
	for (i=count, p = cp, f = fp; i; i--, p++)
	{
		if (f)
			flags = *f++;
		rawio_single_char_received(tty,*cp++,flags);
	}
}

static int rawio_receive_room(struct tty_struct *tty)
{
	return -1;
}




/* init for compiled-in driver */
int __init rawio_init(void)
{
	int status;
	printk ("Sartorius rawio driver $Revision: 1.2 $ installed\n");
	/*
	 * Register the tty line discipline
	 */

	status = tty_register_ldisc (N_RAWIO, &tty_ldisc_N_RAWIO);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_RAWIO);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_RAWIO.flags,
		        tty_ldisc_N_RAWIO.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_RAWIO.open);
		TRACE_L("tty_ldisc_N_MODBUS = %x", (int)&tty_ldisc_N_RAWIO);
	}
	else
	{
		printk (KERN_ERR "rawio: error registering line discipline: %d\n", status);
	}
	return status;
}
/* Cleanup - undid whatever init_module did */
void __exit rawio_exit(void)
{
	int status;


	status=tty_register_ldisc(N_RAWIO, NULL);

	if(status!=0)
	{
		printk(KERN_ERR "rawio: error unregistering linediscipline: %d\n", status);
	}
}

module_init(rawio_init);
module_exit(rawio_exit);

MODULE_LICENSE("GPL");
MODULE_ALIAS_LDISC(N_RAWIO);
