/***************************************************************************
                          sag_wmprint.c  -  description
                             -------------------
    begin                : Mo Jan 10 2005
    copyright            : (C) 2005 by Sartorius Hamburg GmbH
    email                : andreas.horn@sartorius.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
** $Log: sag_wmprint.c,v $
** Revision 1.4  2005/10/20 11:42:58  dec3hon
** use a unsigned char in receive_single_char
**
** Revision 1.3  2005/10/19 09:06:35  dec3hon
** check for pending signals after timeout occurs
**
** Revision 1.2  2005/08/26 09:19:48  dec3hon
** add n_tty_ioctl call in ioctl
**
** Revision 1.1  2005/02/24 08:47:29  dec3hon
** saglinux.kdevelop
**
**
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <asm/dma.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/string.h>   /* used in new tty drivers */
#include <linux/ioctl.h>
#include <linux/sag_wmprint.h>

//#define DEBUG

#ifdef DEBUG
#define TRACE_L(format, args...) printk("xbpi: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif


static int print_buffer(struct wmprint_info *pInfo);
static void on_timeout(unsigned long priv);
static void receive_char(struct wmprint_info *pInfo, const unsigned char c);
static int  wmprint_open(struct tty_struct *tty);
static void wmprint_close(struct tty_struct *tty);
static ssize_t wmprint_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr);
static ssize_t wmprint_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr);
static int wmprint_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void wmprint_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void wmprint_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char fp);
static int  wmprint_receive_room(struct tty_struct *tty);

int __init wmprint_init(void);
void __exit wmprint_exit(void);

static struct tty_ldisc tty_ldisc_N_WMPRINT =
    {
    .owner	 = THIS_MODULE,
    .magic	= TTY_LDISC_MAGIC,
    .name	= "W&MPRINT",
    .open	= wmprint_open,
    .close	= wmprint_close,
    .read	= wmprint_read,
    .write	= wmprint_write,
    .ioctl	= wmprint_ioctl,
    .receive_buf = wmprint_receive_buf,
    .receive_room = wmprint_receive_room,
    .single_char_received=wmprint_single_char_received                                                                                                         
    };


static int  wmprint_open(struct tty_struct *tty)
{
	struct wmprint_info * pInfo;
	pInfo=kmalloc(sizeof(struct wmprint_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "W&M printer: failed to alloc info structure\n");
		return -ENOMEM;
	}

	pInfo->tx_buf = kmalloc(TX_BUF_SIZE, GFP_KERNEL);

	if(!pInfo->tx_buf)
	{
		printk(KERN_ERR "W&M printer: failed to alloc transmit buffer\n");
		kfree(pInfo->tx_buf);
		kfree(pInfo);
		return -ENOMEM;
	}
	tty->disc_data = pInfo;
	pInfo->tty = tty;
	pInfo->state = IDLE;
	pInfo->spaceleft = TX_BUF_SIZE;
	pInfo->olen = 0;
	init_timer(&pInfo->tmr);
	pInfo->tmr.data = (unsigned long)pInfo;
	pInfo->tmr.function = on_timeout;
	init_waitqueue_head(&pInfo->read_wait);
	/* request the interrupt for the dma timer */
	return 0;
}


static void wmprint_close(struct tty_struct *tty)
{
	struct wmprint_info *pInfo=(struct r3964_info*)tty->disc_data;
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	del_timer_sync(&pInfo->tmr);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	kfree(pInfo->tx_buf);
	kfree(pInfo);
}

static ssize_t wmprint_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr)
{
	return 0;
}

unsigned char cnvtab[]=
{
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,		/*00..07 */
0x20,0x09,0x20,0x20,0x0C,0x20,0x20,0x20,		/*08..0F	; allow TAB and FormFeed*/
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,		/*10..17*/
0x20,0x20,0x20,0x1B,0x20,0x20,0x20,0x20,		/*18..1F*/
};

static ssize_t wmprint_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr)

{
	int i;
	int ret =0;
	unsigned char *pBuf;
	char c;
	struct wmprint_info *pInfo=(struct wmprint_info*)tty->disc_data;

	if( !nr)
	{
		return -EINVAL;
	}

	if (!tty->driver->tiocmset)
		return -EINVAL;

	pBuf = pInfo->tx_buf;
	for(i=0;i<nr;i++)
	{
		/* if buffer is full print them */
		if(pInfo->spaceleft==0)
		{
			ret =print_buffer(pInfo);
			if(pInfo->flags != SUCCESS)
				return -EIO;
			pInfo->spaceleft=TX_BUF_SIZE;
			pInfo->olen=0;
		}
		get_user(c,buf+i);
		switch(c)
		{
			case 0x0d:
				break;			/* CR will be ignored */

			case 0x0a:
				pBuf[pInfo->olen++]=c;
				pInfo->spaceleft=0;
				break;			/* force print buffer */

			case 0x0c:
				if(pInfo->olen==0)
				{
					pBuf[pInfo->olen++]=c;
					pInfo->spaceleft=0;
					break;			/* force print buffer */
				}
			default:
				if(c < 0x20)
					c = cnvtab[(int)c];
				pBuf[pInfo->olen++]=c;
				pInfo->spaceleft--;
				break;
		}
	}
	/* if buffer is full print them */
	if(pInfo->spaceleft==0)
	{
		ret =print_buffer(pInfo);
		if(ret < 0 || pInfo->flags != SUCCESS)
			return -EIO;
		pInfo->spaceleft=TX_BUF_SIZE;
		pInfo->olen=0;
	}
	return nr;
}
static int print_buffer(struct wmprint_info *pInfo)


{
	struct tty_struct *tty = pInfo->tty;
	int retry,i;
	int sretry=0;
	if(tty==NULL)
		return -1;
	if(!tty->driver->put_char)
		return -1;
	pInfo->flags = SUCCESS;
	retry=0;
	pInfo->bcc=0;
	/* calculate bcc */
	for(i=0;i<pInfo->olen;i++)
		pInfo->bcc ^= pInfo->tx_buf[i];
	pInfo->bcc ^= 0x03;;

	do
	{
		retry=0;
		/* try to connect to printer */
		do
		{
			pInfo->state = WAIT_FOR_CONNECT;
			pInfo->flags = SUCCESS;
			tty->driver->put_char(tty,0x05);		/* send ENQ */
			/* 5 sec timeout */
			pInfo->tmr.expires = jiffies +5000/(1000/HZ);
			add_timer(&pInfo->tmr);
			interruptible_sleep_on(&pInfo->read_wait);
			del_timer_sync(&pInfo->tmr);
			if (signal_pending(current)) {
				pInfo->flags = TIMEOUT;
				return -1;
			}
			if(pInfo->flags == TIMEOUT)
				retry++;
			if(retry == 3)
			{
				/* can't connect to printer don't try again */
				pInfo->flags = TIMEOUT;
				return -1;
			}
		}
		while(pInfo->flags != SUCCESS);
		pInfo->state = WAIT_FOR_ACK;
		tty->driver->put_char(tty,0x02);			/* send STX */
		tty->driver->write(tty,pInfo->tx_buf,pInfo->olen);	/* send string */
		tty->driver->put_char(tty,0x03);			/* send ETX */
		tty->driver->put_char(tty,pInfo->bcc);			/* send bcc */

		add_timer(&pInfo->tmr);
		interruptible_sleep_on(&pInfo->read_wait);
		del_timer_sync(&pInfo->tmr);
		if (signal_pending(current)) {
			pInfo->flags = TIMEOUT;
			return -1;
		}
		if(pInfo->flags == TIMEOUT)
			sretry++;
		if(sretry == 3)
		{
			pInfo->flags = TIMEOUT;
			return -1;
		}
	}
	while(pInfo->flags != SUCCESS);
	return 0;
}


static int wmprint_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	return  n_tty_ioctl (tty, file, cmd, arg);
}
static void on_timeout(unsigned long priv)
{
	struct wmprint_info *pInfo = (void *)priv;
	pInfo->state = IDLE;
	pInfo->flags = TIMEOUT;
	wake_up_interruptible(&pInfo->read_wait);
}

static void receive_char(struct wmprint_info *pInfo, const unsigned char c)
{
	struct tty_struct *tty = pInfo->tty;
	
	switch(pInfo->state)
	{
		case IDLE:
			break;
		
		case WAIT_FOR_CONNECT:
			if(c == 0x06)
			{
				wake_up_interruptible(&pInfo->read_wait);
				pInfo->state = IDLE;
			}
			break;
		/* wait for ack or nak */
		case WAIT_FOR_ACK:
			if(c == 0x06)
			{
				pInfo->flags = SUCCESS;
				pInfo->state = IDLE;
				wake_up_interruptible(&pInfo->read_wait);
			}
			else
			{
				pInfo->flags = SNDERR;
				pInfo->state = ACK_NAK;
				tty->driver->put_char(tty,0x06);	/* ENQ */
			}
			return;
			
		/* wait for ACK */
		case ACK_NAK:
			if( c == 0x06)
			{
				pInfo->state = WAIT_FOR_CONNECT;
				wake_up_interruptible(&pInfo->read_wait);
			}
			return;
	}
}

/* entry for receiver interrupt */
static void wmprint_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char flags)
{
	struct wmprint_info *pInfo=(struct wmprint_info*)tty->disc_data;
	if(flags==TTY_NORMAL)
	{
		receive_char(pInfo,cp);
	}
	/* in case of parity error ignore char, driver will be waked up 
	 using the timeout feature */
}
static void wmprint_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
int i; 
char *f;
const unsigned char *p;
char flags=0;
	for (i=count, p = cp, f = fp; i; i--, p++)
	{
		if (f)
			flags = *f++;
		wmprint_single_char_received(tty,*cp++,flags);
	}
}


static int wmprint_receive_room(struct tty_struct *tty)
{
	return -1;
}




/* init for compiled-in driver */
int __init wmprint_init(void)
{
	int status;
	printk ("Sartorius W&M printer driver $Revision: 1.4 $ installed\n");
	/*
	 * Register the tty line discipline
	 */

	status = tty_register_ldisc (N_WMPRINT, &tty_ldisc_N_WMPRINT);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_WMPRINT);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_WMPRINT.flags,
		        tty_ldisc_N_WMPRINT.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_WMPRINT.open);
		TRACE_L("tty_ldisc_N_WMPRINT = %x", (int)&tty_ldisc_N_WMPRINT);
	}
	else
	{
		printk (KERN_ERR "W&M printer: error registering line discipline: %d\n", status);
	}
	return status;
}
/* Cleanup - undid whatever init_module did */
void __exit wmprint_exit(void)
{
	int status;


	status=tty_register_ldisc(N_WMPRINT, NULL);

	if(status!=0)
	{
		printk(KERN_ERR "W&M printer: error unregistering linediscipline: %d\n", status);
	}
}

module_init(wmprint_init);
module_exit(wmprint_exit);

MODULE_LICENSE("GPL");
MODULE_ALIAS_LDISC(N_WMPRINT);
