/***************************************************************************
                          sag_xbpi.c  -  description
                             -------------------
    begin                : Mo Jan 10 2005
    copyright            : (C) 2005 by Sartorius Hamburg GmbH
    email                : andreas.horn@sartorius.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
** $Log: sag_xbpi.c,v $
** Revision 1.11  2008/08/19 13:53:00  kf 
** add CM handling
**
** Revision 1.10  2005/10/24 09:27:52  dec3ban
** must set RTS before break to enable RS485 output
**
** Revision 1.9  2005/10/20 11:42:58  dec3hon
** use a unsigned char in receive_single_char
**
** Revision 1.8  2005/10/04 12:23:47  dec3ban
** added xbpi to dma interrupt text
**
** Revision 1.7  2005/09/21 10:49:28  dec3hon
** ICR reg now set in request timer
**
** Revision 1.6  2005/09/20 13:05:17  dec3hon
** return IRQ_HANDLED and check for wait queue is active
**
** Revision 1.5  2005/08/26 09:19:48  dec3hon
** add n_tty_ioctl call in ioctl
**
** Revision 1.4  2005/06/22 12:13:30  dec3ban
** use tty->termios_sem as semaphore for tty in ioctl
**
** Revision 1.3  2005/02/15 10:42:49  dec3ban
** added per device semaphore
**
** Revision 1.2  2005/01/24 11:55:26  dec3hon
** use dma timer routines from dma_timer
**
** Revision 1.1  2005/01/18 12:35:41  dec3hon
** *** empty log message ***
**
** Revision 1.1  2005/01/10 11:45:38  dec3hon
** new driver
**
*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <asm/dma.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>
#include <linux/in.h>
#include <linux/tty.h>
#include <linux/errno.h>
#include <linux/string.h>   /* used in new tty drivers */
#include <linux/ioctl.h>
//#include <linux/poll.h>
#include <linux/sag_xbpi.h>
#include <asm/mcf528x_dma_timer.h>

#define DEBUG

#define SUPPORT_PTY 1

#ifdef DEBUG
#define TRACE_L(format, args...) printk("xbpi: " format "\n" , ## args);
#else
#define TRACE_L(fmt, arg...) /**/
#endif


static void xbpi_tty_wakeup(struct tty_struct *tty);
static irqreturn_t on_chr_timeout(int irq, void *dummy, struct pt_regs *fp);
static void xbpi_start_receive(struct xbpi_info *pInfo);
static void on_overall_timeout(unsigned long priv);
static void CM_on_overall_timeout(unsigned long priv);
static void xbpi_receive_char(struct xbpi_info *pInfo, const unsigned char c);
static void receive_error(struct xbpi_info *pInfo, const char flag);
static int  xbpi_open(struct tty_struct *tty);
static void xbpi_close(struct tty_struct *tty);
static ssize_t xbpi_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr);
static ssize_t xbpi_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr);
//static unsigned int xbpi_poll(struct tty_struct * tty, struct file * file, poll_table *wait);
static int xbpi_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg);
static void xbpi_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count);
static void xbpi_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char fp);
static int  xbpi_receive_room(struct tty_struct *tty);

int __init xbpi_init(void);
void __exit xbpi_exit(void);


static void xbpi_tty_wakeup(struct tty_struct *tty)
{
	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;

	if (!pInfo)
		return;
	if(pInfo->tty->stopped)
		return;

	if(pInfo->state_tx==TX_TRANSMIT)
	{
		pInfo->tty->flags &= ~(1<<TTY_DO_WRITE_WAKEUP);
		if(pInfo->para.ibuf)
			xbpi_start_receive(pInfo);
		else
		{
			pInfo->state_tx=TX_TRANSMIT_DONE;
			if (waitqueue_active(&pInfo->wait_for_state))
				wake_up_interruptible(&pInfo->wait_for_state);
		}
	}
}

static struct tty_ldisc tty_ldisc_N_XBPI = {
    .owner	 = THIS_MODULE,
    .magic	= TTY_LDISC_MAGIC,
    .name	= "XBPI",
    .open	= xbpi_open,
    .close	= xbpi_close,
    .read	= xbpi_read,
    .write	= xbpi_write,
    .ioctl	= xbpi_ioctl,
    .receive_buf = xbpi_receive_buf,
    .receive_room = xbpi_receive_room,
    .single_char_received = xbpi_single_char_received,
    .write_wakeup = xbpi_tty_wakeup
};


static int xbpi_open(struct tty_struct *tty)
{
	struct xbpi_info * pInfo;
	pInfo=kmalloc(sizeof(struct xbpi_info), GFP_KERNEL);

	if(!pInfo)
	{
		printk(KERN_ERR "xbpi:open failed to alloc info structure\n");
		return -ENOMEM;
	}
	
	memset (pInfo, 0, sizeof(struct xbpi_info));
	
	TRACE_L("open(%s)",tty->name);

	if(tty->driver->type == TTY_DRIVER_TYPE_PTY)
	{
		/* init to zero */
		pInfo->intr_request_vector=0;
		pInfo->chr_timeout_timer=0;
	}
	else
	{
		/* request a dma timer */
		if(!(pInfo->chr_timeout_timer=request_hrt_timer(&pInfo->intr_request_vector)))
		{
			printk("xbpi: unable to get DMA timer\n");
			kfree(pInfo);
			return -EBUSY;
		}
	}
	tty->disc_data = pInfo;
	pInfo->tty = tty;
	pInfo->state_tx = IDLE;
	pInfo->state_rx = RX_WAIT_LEN;
	pInfo->rx_buf = &pInfo->rx_buf_cs;
	pInfo->rx_buf_cm.rx_ready=0;
	pInfo->rx_buf_cs.rx_ready=0;
	init_timer(&pInfo->overall_timeout_timer);
	pInfo->overall_timeout_timer.data = (unsigned long)pInfo;
	pInfo->overall_timeout_timer.function = on_overall_timeout;
	
	init_timer(&pInfo->CM_overall_timeout_timer);
	pInfo->CM_overall_timeout_timer.data = (unsigned long)pInfo;
	pInfo->CM_overall_timeout_timer.function = CM_on_overall_timeout;
	
	init_waitqueue_head(&pInfo->wait_for_state);

	init_MUTEX(&pInfo->CMsem);
	
	/* request the interrupt for the dma timer */
	
	if(pInfo->intr_request_vector
	&& request_irq(pInfo->intr_request_vector+64, on_chr_timeout, SA_INTERRUPT,
	               "dma interrupt (xbpi)", tty) < 0 )
		printk( "%s(%d):Cant request interrupt\n",
		        __FILE__,__LINE__);

	return 0;
}


static void xbpi_close(struct tty_struct *tty)
{
	TRACE_L("close(%s)",tty->name);

	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;
	/* disable dms timer */
	if(pInfo->chr_timeout_timer)
	{
		pInfo->chr_timeout_timer->dtxmr=0;	// no dma request on reference match
		release_hrt_timer(pInfo->chr_timeout_timer);
	}
	if(pInfo->intr_request_vector)
		free_irq(pInfo->intr_request_vector+64, tty);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	del_timer_sync(&pInfo->overall_timeout_timer);
	del_timer_sync(&pInfo->CM_overall_timeout_timer);
	/*
	 * Make sure that our task queue isn't activated.  If it
	 * is, take it out of the linked list.
	 */
	kfree(pInfo);
}

static ssize_t xbpi_read(struct tty_struct *tty, struct file *file,
                         unsigned char *buf, size_t nr)
{
	return 0;
}


static ssize_t xbpi_write(struct tty_struct * tty, struct file * file,
                          const unsigned char * buf, size_t nr)
{
	return 0;
}

/* 2 char timeout values in usec */
static unsigned baud_delay[] =
{
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
	9600, 19200, 38400, 57600, 115200, 230400, 460800, 0
};


static void xbpi_start_receive(struct xbpi_info *pInfo)
{
	if(pInfo->para.timo)
	{
		init_timer(&pInfo->overall_timeout_timer);
		pInfo->overall_timeout_timer.expires = jiffies +(pInfo->para.timo+1000/HZ-1)/(1000/HZ);
		add_timer(&pInfo->overall_timeout_timer);
	}
/*	if(pInfo->chr_timeout_timer)
	{
		pInfo->state_rx = RX_WAIT_SILENCE;
		start_hrt_timer(pInfo->chr_timeout_timer,pInfo->delay_15);
	}*/
//	else
//		pInfo->state=RX_WAIT_LEN;
}

static inline void set_delay (tcflag_t c_cflag, struct xbpi_info *pInfo)
{
	struct S_MCF_DMA_TIMER *dmatimer;
	int i;
	
	i = c_cflag & CBAUD;
	if (i & CBAUDEX)
	{
		i &= ~CBAUDEX;
		if (i < 1 || i > 4)
			;
		else
			i += 15;
	}
	/* mayby NULL for PTYs */
	dmatimer = pInfo->chr_timeout_timer;

	/* set 3 character timeout value in usec*/
	/* note: values slightly modified to ensure timing*/
	/* 35->44 */
	/* 15->11 */
	if(dmatimer)
	{
		pInfo->delay_35 = (1000*1000*44)/baud_delay[i];
		pInfo->delay_15 = (1000*1000*11)/baud_delay[i];
	}
	else
	{
		pInfo->delay_35 = 0;
		pInfo->delay_15 = 0;

	}
	//dmatimer->dtmr&=~DTMR_ORRI_1;
	//set_task_state(current,TASK_INTERRUPTIBLE);
}

static int xbpi_ioctl(struct tty_struct * tty, struct file * file,
                      unsigned int cmd, unsigned long arg)
{
	int i;
	int ret =0;
	struct XBPI_PARA *parg=(struct XBPI_PARA *)arg;
	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;
	struct XBPI_PARA para;
	struct S_MCF_DMA_TIMER *dmatimer;
	unsigned char bcc;
	unsigned char len;
	
	if(cmd!=XBPI_IO_TELEGRAMM)
	{
		return  n_tty_ioctl (tty, file, cmd, arg);
	}

	copy_from_user((void*)&para,(void*)arg,sizeof(struct XBPI_PARA));

	if( para.isiz>255
	 || para.osiz>255
	 || (para.ibuf && para.isiz==0)	// fehler: input buffer ohne groesse
	 || (para.obuf && para.osiz==0)	// fehler: output buffer ohne groesse
	 || (!para.ibuf && para.isiz!=0)// fehler: kein inbut, aber groesse
	 || (!para.obuf && para.osiz!=0)// fehler: kein output, aber groesse
	)
	{
		put_user(IO_PARA_ERR,&parg->status);
		return -EINVAL;
	}
	if(para.obuf)
	{
		UBYTE len;
		get_user(len,(UBYTE*)para.obuf);
		if(len!=para.osiz)
		{
			put_user(IO_PARA_ERR,&parg->status);
			return -EINVAL;
		}
	}
#if ! SUPPORT_PTY	
	if (!tty->driver->tiocmset)
		return -EINVAL;
#endif

	// CM receive if no output buffer available
	if(!para.obuf && para.ibuf)
	{											// **** CM read
		down(&pInfo->CMsem);
		put_user(IO_PROGRESS,&parg->status);	// indicate status io_progress
		set_delay(tty->termios->c_cflag, pInfo);	// FIXME: this is obsolete
		if(!(pInfo->rx_buf_cm.rx_ready))
		{
			if(para.timo)
			{
				init_timer(&pInfo->CM_overall_timeout_timer);
				pInfo->CM_overall_timeout_timer.expires = jiffies +(para.timo+1000/HZ-1)/(1000/HZ);
				add_timer(&pInfo->CM_overall_timeout_timer);
			}
			//********** waiting point ******
			wait_event_interruptible (pInfo->wait_for_state, pInfo->rx_buf_cm.rx_ready);
			if(para.timo)
				del_timer_sync(&pInfo->CM_overall_timeout_timer);
		}
		
		if(pInfo->rx_buf_cm.flags & RX_SUCCESS)
			para.status=IO_SUCCESS;
		if(pInfo->rx_buf_cm.flags & RX_TIMEOUT)
		{
			TRACE_L("CM:TIMEOUT");
			para.status = IO_TIMEOUT;
			ret = -ENODATA;
		}
		if(pInfo->rx_buf_cm.flags & RX_CERR)
		{
			TRACE_L("CM:IO_PHYSICAL_ERR");
			para.status = IO_PHYSICAL_ERR;
			ret = -ENODATA;
		}
		len = pInfo->rx_buf_cm.position;
		if (len > para.isiz) 
			len = para.isiz;					// dont overload input buffer
		if (len)
			copy_to_user (para.ibuf, pInfo->rx_buf_cm.buf, len);
		put_user (len, &parg->icnt);
		put_user (para.status, &parg->status);
		pInfo->rx_buf_cm.rx_ready =0;
		/* zero our IPL */
		up(&pInfo->CMsem);
	}
	else
	{											// ***** CS command (write / read)
		down(&tty->termios_sem);
		pInfo->para=para;						// save for later use
		put_user(IO_PROGRESS,&parg->status);	// indicate status io_progress
		set_delay (tty->termios->c_cflag, pInfo);
		dmatimer = pInfo->chr_timeout_timer;	// mayby NULL for PTYs 

		if(para.brk && tty->driver->tiocmset)
		{
			//                         set , reset
			tty->driver->tiocmset(tty, file, TIOCM_RTS, 0);
			tty->driver->ioctl(tty,file,TCSBRKP,1); /* 1 dezisecond */
			//	                set , reset
			tty->driver->tiocmset(tty, file,0 , TIOCM_RTS);
	
			// wait for 30 msec without receiving a char
			pInfo->state_tx = WAIT_AFTER_BREAK;
			init_timer(&pInfo->overall_timeout_timer);
			pInfo->overall_timeout_timer.expires = jiffies + WAIT_AFTER_BREAK;
			add_timer(&pInfo->overall_timeout_timer);
			wait_event_interruptible(pInfo->wait_for_state,pInfo->state_tx==WAIT_AFTER_BREAK_DONE);
		}
		else
		{
			if(dmatimer)
			{
				pInfo->state_tx = TX_WAIT_SILENCE;
				/* start timer for 3 character wait */
				start_hrt_timer(dmatimer,pInfo->delay_35);
				/* wait for dmatimer int */
				wait_event_interruptible(pInfo->wait_for_state,pInfo->state_tx==TX_READY_TO_TRANSMIT);
			}
			else
			{
				pInfo->state_tx=TX_READY_TO_TRANSMIT;
			}
		}
		pInfo->rx_buf_cs.rx_ready=0;
		if(para.obuf)
		{
			/* set RTS */
			if(tty->driver->tiocmset)
				tty->driver->tiocmset(tty, file, TIOCM_RTS, 0);
	
			copy_from_user(pInfo->tx_buf,para.obuf,para.osiz);
			// send question to device
			for(i=0, bcc=0; i<para.osiz; i++)
				bcc += pInfo->tx_buf[i];
//			if (pInfo->tx_buf[3] == 0x1F)		// debug to stimulate a timeout!
//				bcc++;
			pInfo->tx_buf[para.osiz]=bcc;
			pInfo->state_tx=TX_TRANSMIT;
			pInfo->tty->flags |= (1<<TTY_DO_WRITE_WAKEUP);
			pInfo->tty->driver->write(pInfo->tty,pInfo->tx_buf,para.osiz+1);
			if(para.ibuf)
				wait_event_interruptible (pInfo->wait_for_state, pInfo->rx_buf_cs.rx_ready);
			else
				wait_event_interruptible (pInfo->wait_for_state, pInfo->state_tx==TX_TRANSMIT_DONE);
		}
		else
		{
			xbpi_start_receive(pInfo);
			wait_event_interruptible (pInfo->wait_for_state, pInfo->rx_buf_cs.rx_ready);
		}
		
		if(para.timo)
			del_timer_sync(&pInfo->overall_timeout_timer);
			
		if(pInfo->rx_buf_cs.flags & RX_SUCCESS)
			para.status=IO_SUCCESS;
		if(pInfo->rx_buf_cs.flags & RX_TIMEOUT)
		{
			TRACE_L("CS:RX_TIMEOUT");
			para.status = IO_TIMEOUT;
			ret = -ENODATA;
		}
		if(pInfo->rx_buf_cs.flags & RX_CERR)
		{
			TRACE_L("CS:IO_PHYSICAL_ERR");
			para.status = IO_PHYSICAL_ERR;
			ret = -ENODATA;
		}
		if(para.ibuf)
		{
			len = pInfo->rx_buf_cs.position;
			if (len > para.isiz) 
				len = para.isiz;					// dont overload input buffer
			if (len)
				copy_to_user (para.ibuf, pInfo->rx_buf_cs.buf, len);
			put_user (len, &parg->icnt);
		}
		put_user(para.status,&parg->status);
		/* zero our IPL */
		up(&tty->termios_sem);
	}
	return ret;
}


//------------ CS timeout ----------------------
static void on_overall_timeout(unsigned long priv)
{
	struct xbpi_info *pInfo = (void *)priv;

	if (pInfo->state_tx == WAIT_AFTER_BREAK)
	{
		pInfo->state_tx = WAIT_AFTER_BREAK_DONE;
		wake_up_interruptible (&pInfo->wait_for_state);
	}
	else
	{
		TRACE_L("CS:on_overall_timeout");
		pInfo->rx_buf_cs.flags |= RX_TIMEOUT;
		pInfo->rx_buf_cs.rx_ready = 1;
		if (waitqueue_active(&pInfo->wait_for_state))
			wake_up_interruptible (&pInfo->wait_for_state);
	}
}

//------------ CM read timeout ----------------------
static void CM_on_overall_timeout(unsigned long priv)
{
	struct xbpi_info *pInfo = (void *)priv;

	{
		TRACE_L("CM_on_overall_timeout");
		pInfo->rx_buf_cm.flags |= RX_TIMEOUT;
		pInfo->rx_buf_cm.rx_ready = 1;
		if (waitqueue_active(&pInfo->wait_for_state))
			wake_up_interruptible (&pInfo->wait_for_state);
	}
}

static void xbpi_receive_char(struct xbpi_info *pInfo, const unsigned char c)
{
	struct S_MCF_DMA_TIMER *dmatimer;
	dmatimer = pInfo->chr_timeout_timer;

	// FIXME::
//	if (is_timeout())					// state machine timeout mechanism if no dmatimer is used
//		pInfo->state_rx = RX_WAIT_LEN;			// reset state machine if last call has been longer than 20 msec ago
		
	switch (pInfo->state_rx)
	{
		/* retrigger 3 char timeout */
		case RX_WAIT_SILENCE:
			if(dmatimer)
				dmatimer->dtcn=0;	/* restart timer */
			break;
		case RX_WAIT_LEN:
			if (c==0)	// illegal length
			{
				printk("xbpi: RX_WAIT_LEN got len 0\n");
				//if(dmatimer)
				//	stop_hrt_timer(dmatimer);
				break;
			}
			pInfo->state_rx = RX_COMMAND_CHAR;
			pInfo->rx_tel_len = c;
			if(dmatimer)
				start_hrt_timer(dmatimer,pInfo->delay_35);	/* start timer for 3 character timeout */
			break;
		case RX_COMMAND_CHAR:
			if (c & XBPI_CM_BIT)				// the second char decides whether we have a CS or a CM response)
				pInfo->rx_buf = &pInfo->rx_buf_cm;
			else
				pInfo->rx_buf = &pInfo->rx_buf_cs;
			pInfo->rx_buf->flags = 0;
			pInfo->rx_buf->position = 0;
			pInfo->rx_buf->buf[pInfo->rx_buf->position++] = pInfo->rx_tel_len;
			pInfo->bcc = pInfo->rx_tel_len;
			pInfo->state_rx = RX_NEXT_CHAR;
			// ********** NO break;
		case RX_NEXT_CHAR:
			if(dmatimer)
				start_hrt_timer(dmatimer,pInfo->delay_35);	/* retrigger 3 char timeout */
			if(pInfo->rx_buf->position >= RX_BUF_SIZE-1)
				pInfo->rx_buf->flags |= RX_CERR;
			else
				pInfo->rx_buf->buf[pInfo->rx_buf->position++] = c;
			pInfo->bcc += c;
			if(pInfo->rx_buf->position>=pInfo->rx_tel_len)
				pInfo->state_rx = RX_WAIT_CHK;
			break;
		case RX_WAIT_CHK:
			if(pInfo->bcc != c)
			{
				printk("xbpi:RX_WAIT_CHK bcc error %02X<>%02X\n",pInfo->bcc,c);
				pInfo->rx_buf->flags |= RX_CERR;
			}
			else
				pInfo->rx_buf->flags |= RX_SUCCESS;
			/* stop timer */
			//if(dmatimer)
			//	stop_hrt_timer(dmatimer);
			pInfo->state_rx = RX_WAIT_LEN;
			pInfo->rx_buf->rx_ready=1;
			if (waitqueue_active(&pInfo->wait_for_state))
				wake_up_interruptible(&pInfo->wait_for_state);	/* CS/CM receive finished */
			break;
	}
}

static void receive_error(struct xbpi_info *pInfo, const char flag)
{
	pInfo->rx_buf->flags |= RX_CERR;
}

/* entry for receiver interrupt */
static void xbpi_single_char_received(struct tty_struct *tty, const unsigned char cp,
                             unsigned char flags)
{
	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;
	if(flags==TTY_NORMAL)
	{
		xbpi_receive_char(pInfo,cp);
	}
	else if(cp == 0xff && flags == 0xff)
	{	// we have had a powerfail
		if(pInfo->chr_timeout_timer)
		{
			reinit_hrt_timer(pInfo->chr_timeout_timer);
			start_hrt_timer(pInfo->chr_timeout_timer,pInfo->delay_15);
		}
	}
	else
	{
		receive_error(pInfo, flags);
	}
}

const char*xbpi_state_rx_str(e_xbpi_state_rx state)
{
	switch(state)
	{
	case RX_WAIT_SILENCE:return "RX_WAIT_SILENCE";
	case RX_WAIT_LEN:return "RX_WAIT_LEN";
	case RX_COMMAND_CHAR:return "RX_COMMAND_CHAR";
	case RX_NEXT_CHAR:return "RX_NEXT_CHAR";
	case RX_WAIT_CHK:return "RX_WAIT_CHK";
	}
	return "??";
}

static void xbpi_receive_buf(struct tty_struct *tty, const unsigned char *cp,
                             char *fp, int count)
{
	int i;
	char flags=0;
	//----------------------------------------------
	// JJvB: on pty's we assume always a complete frame
	// so rx_state should be RX_WAIT_LEN before
	// and after. we'll to a printk to catch faults
	//----------------------------------------------
	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;
	if(!pInfo->chr_timeout_timer)
	{
		if(pInfo->state_rx != RX_WAIT_LEN)
		printk("xbpi:rxb-a state_rx=%s want %s\n",
			xbpi_state_rx_str(pInfo->state_rx),
			xbpi_state_rx_str(RX_WAIT_LEN));
	//	pInfo->state_rx = RX_WAIT_LEN;			// reset state machine if last call has been longer than 20 msec ago
	}

	for (i=count; i; i--)
	{
		if (fp)
			flags = *fp++;
		xbpi_single_char_received(tty,*cp++,flags);
	}

	if(!pInfo->chr_timeout_timer)
	{
		if(pInfo->state_rx != RX_WAIT_LEN)
			printk("xbpi:rxb-b state_rx=%s want %s\n",
				xbpi_state_rx_str(pInfo->state_rx),
				xbpi_state_rx_str(RX_WAIT_LEN));
	//	pInfo->state_rx = RX_WAIT_LEN;			// reset state machine if last call has been longer than 20 msec ago
	}

}

//-----------------------------------------------------------------
// NOTE: this will only be called on physical serial ports
// PTYs dont use character timeout, so this will NEVER be called on an pty
//-----------------------------------------------------------------
static int on_chr_timeout(int irq, void *dev_id, struct pt_regs *fp)
{
	struct xbpi_info *pInfo=((struct tty_struct *)dev_id)->disc_data;
	if(pInfo->chr_timeout_timer)
		stop_hrt_timer(pInfo->chr_timeout_timer);
	
	if (pInfo->state_tx == TX_WAIT_SILENCE)
	{
		pInfo->state_tx=TX_READY_TO_TRANSMIT;
		if (waitqueue_active(&pInfo->wait_for_state))
			wake_up_interruptible(&pInfo->wait_for_state);	/* transfer finished */
	}
	else if (pInfo->state_rx == RX_WAIT_SILENCE)
	{
		pInfo->state_rx=RX_WAIT_LEN;
	}
	else
	{
		if(pInfo->rx_buf)
			pInfo->rx_buf->flags |= RX_CERR;
		pInfo->state_rx=RX_WAIT_LEN;
	}

	return IRQ_HANDLED;
}

static int xbpi_receive_room(struct tty_struct *tty)
{
	int c;
	struct xbpi_info *pInfo=(struct xbpi_info*)tty->disc_data;
	//FIXME::this should depend on buffer, if state is RX_NEXT_CHAR
	c = RX_BUF_SIZE-pInfo->rx_buf_cs.position;
	return c > 0 ? c : 0;
}




/* init for compiled-in driver */
int __init xbpi_init(void)
{
	int status;
	printk ("Sartorius XBPI driver installed\n");
	/*
	 * Register the tty line discipline
	 */

	status = tty_register_ldisc (N_XBPI, &tty_ldisc_N_XBPI);
	if (status == 0)
	{
		TRACE_L("line discipline %d registered", N_XBPI);
		TRACE_L("flags=%x num=%x", tty_ldisc_N_XBPI.flags,
		        tty_ldisc_N_XBPI.num);
		TRACE_L("open=%x", (int)tty_ldisc_N_XBPI.open);
		TRACE_L("tty_ldisc_N_XBPI = %x", (int)&tty_ldisc_N_XBPI);
	}
	else
	{
		printk (KERN_ERR "xbpi: error registering line discipline: %d\n", status);
	}
	return status;
}
/* Cleanup - undid whatever init_module did */
void __exit xbpi_exit(void)
{
	int status;


	status=tty_register_ldisc(N_XBPI, NULL);

	if(status!=0)
	{
		printk(KERN_ERR "xbpi: error unregistering linediscipline: %d\n", status);
	}
}

module_init(xbpi_init);
module_exit(xbpi_exit);

MODULE_LICENSE("GPL");
MODULE_ALIAS_LDISC(N_XBPI);
