/*
 * 	ColdFire 527x I2C master support
 */
#include <linux/i2c.h>
#include <linux/i2c-algo-mcf.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <asm/coldfire.h>
#include <asm/m5485sim.h>
#include <asm/m527xi2c.h>

#define get_clock(adap) (clock)
#define get_own(adap) 	(own)

static int clock=0x3b;  //50000 / 1024 ~ 49 KHz
static int own=0x78;

static struct i2c_algo_mcf_data i2c_mcf_board_data = {
	timeout:	10000
	};

static struct i2c_adapter i2c_mcf_board_adapter = {
	name:		"MCF527x",
	algo_data:	&i2c_mcf_board_data
	};

/*
 *  static void i2c_start()
 *
 *  Generates START signal
 */
static void
i2c_start(
	struct i2c_algo_mcf_data *adap
) {
	MCF_I2CR |= MCF_I2CR_MSTA;
}


/*
 *  static void i2c_stop()
 *
 *  Generates STOP signal
 */
static void
i2c_stop(
	struct i2c_algo_mcf_data *adap
) {
	MCF_I2CR &= ~MCF_I2CR_MSTA;
}

static int
i2c_getack(
	struct i2c_algo_mcf_data *adap
) {
	return !(MCF_I2SR & MCF_I2SR_RXAK);
}

/*
 *  static void i2c_repstart()
 *
 *  Generates repeated start signal (without STOP while mastering the bus)
 */
static void
i2c_repstart(
	struct i2c_algo_mcf_data *adap
) {
	MCF_I2CR |= MCF_I2CR_RSTA;
	MCF_I2CR |= MCF_I2CR_MTX;
}


/*
 *  static void wait_for_bb()
 *
 *  Wait for bus idle state
 */
static int
wait_for_bb(
	struct i2c_algo_mcf_data *adap
) {
	while (MCF_I2SR & MCF_I2SR_IBB);
	return 0;
}

/*
 *  static void wait_for_not_bb()
 *
 *  Wait for bus busy state
 */
static int
wait_for_not_bb(
	struct i2c_algo_mcf_data *adap
) {
	while (!(MCF_I2SR & MCF_I2SR_IBB));
	return 0;
}

/*
 *  static void wait_xfer_done()
 *
 *  Wait for transfer to complete
 */
static int
wait_xfer_done(
	struct i2c_algo_mcf_data *adap
) {
	int i;

	for (i = 0; i < adap->timeout; i++) {
		if (MCF_I2SR & MCF_I2SR_IIF)
		{
			MCF_I2SR &= ~MCF_I2SR_IIF;
			return 0;
		}
		udelay(1);
	}

	return -ETIMEDOUT;
}


/*
 *  static void i2c_set_addr()
 *
 *  Sets slave address to communicate
 */
static int
i2c_set_addr(
	struct i2c_algo_mcf_data *adap,
        struct i2c_msg *msg,
	int retries
) {
	unsigned short flags = msg->flags;
	unsigned char addr;

	if ( (flags & I2C_M_TEN)  ) {
		/* 10 bit address not supported yet */
		return -EIO;
	} else {
		/* normal 7bit address */
		addr = ( msg->addr << 1 );
		if (flags & I2C_M_RD )
			addr |= 1;
		if (flags & I2C_M_REV_DIR_ADDR )
			addr ^= 1;

		MCF_I2DR = addr;
	}
	return 0;
}


/*
 *  static void mcf_i2c_init()
 *
 *  Perform ColdFire i2c initialization
 */
static void
mcf_i2c_init(struct i2c_algo_mcf_data *adap)
{
        u8 dummy;

	/* Setup GPIO lines */
	MCF_PAR_FECI2CIRQ |= MCF_PAR_SDA;
	MCF_PAR_FECI2CIRQ |= MCF_PAR_SCL;

	/*  Ensure slaves are in idle state */
	if (MCF_I2SR & MCF_I2SR_IBB) {
		MCF_I2ICR = 0x00;
		MCF_I2CR  = 0x00;
		MCF_I2CR  = 0x0A;
		dummy = MCF_I2DR;
		MCF_I2SR  = 0x00;
		MCF_I2CR  = 0x00;
		MCF_I2ICR = 0x01;
	}

	/* setup SCL clock */
	MCF_I2FDR = get_clock(adap);

	/* set slave address */
	MCF_I2AR = get_own(adap);

	/* enable I2C module */
	MCF_I2CR = MCF_I2CR_IEN;
}

static int 
i2c_outb(
	struct i2c_adapter *i2c_adap, 
	char c
) {

	struct i2c_algo_mcf_data *adap = i2c_adap->algo_data;
	int timeout;
    
	/* Put data to be sent */
	MCF_I2DR = c;
    
	/* Wait for xfer completed*/
	timeout = wait_xfer_done(adap);
	if (timeout) {
		i2c_stop(adap);
		wait_for_bb(adap);
		printk(KERN_ERR "i2c-algo-mcf: %s i2c_write: "
	    		"error - timeout.\n", i2c_adap->name);
		return -EREMOTEIO; /* got a better one ?? */
	}

	return 0;
}


/*
 *  static void mcf_sendbytes()
 *
 *  Perform tx data transfer
 */
static int
mcf_sendbytes(
	struct i2c_adapter *i2c_adap,
	const char *buf,
        int count, int last
) {
	struct i2c_algo_mcf_data *adap = i2c_adap->algo_data;
	int ret, i;

	/* Set master TX mode */
	MCF_I2CR |= MCF_I2CR_MTX;

	for (i=0; i<count; ++i) {
		printk(KERN_DEBUG "i2c-algo-mcf: %s i2c_write: writing %2.2X\n",
		      i2c_adap->name, buf[i]&0xff);
		      
		ret = i2c_outb(i2c_adap, buf[i]);
		if (ret < 0)
			return ret;
	}
	if (last) {
		i2c_stop(adap);
		wait_for_bb(adap);
	}
	else {
		i2c_repstart(adap);
	}

	return (i);
}


/*
 *  static void mcf_readbytes()
 *
 *  Perform rx data transfer
 */
static int
mcf_readbytes(
	struct i2c_adapter *i2c_adap,
	char *buf,
        int count, int last
) {
	int i;
	struct i2c_algo_mcf_data *adap = i2c_adap->algo_data;
	u8 dummy;

	/* Set master RX mode */
	MCF_I2CR &= ~MCF_I2CR_MTX;
	MCF_I2CR &= ~MCF_I2CR_TXAK;
	dummy = MCF_I2DR;

	for (i = 0; i < count-1; i++) {
		if (wait_xfer_done(adap)) {
			i2c_stop(adap);
			wait_for_bb(adap);
			printk(KERN_DEBUG "i2c-algo-mcf: mcf_readbytes timed out.\n");
			return (-1);
		}

		/* store next data byte */
		buf[i] = MCF_I2DR;
	}

	if (wait_xfer_done(adap)) {
		i2c_stop(adap);
		wait_for_bb(adap);
		printk(KERN_DEBUG "i2c-algo-mcf: mcf_readbytes timed out.\n");
		return (-1);
	}

	/* Disable acknowlege (set I2CR.TXAK) */
	MCF_I2CR |= MCF_I2CR_TXAK;
	buf[i] = MCF_I2DR;
	if (wait_xfer_done(adap)) {
		i2c_stop(adap);
		wait_for_bb(adap);
		printk(KERN_DEBUG "i2c-algo-mcf: mcf_readbytes timed out.\n");
		return (-1);
	}

	if (last) {
		i2c_stop(adap);
		wait_for_bb(adap);
	} else {
		i2c_repstart(adap);
	}

	return (i+1);
}


/*
 *  static void mcf_xfer()
 *
 *  Perform master data I/O transfer 
 */
static int
mcf_xfer(
	struct i2c_adapter *i2c_adap,
	struct i2c_msg msgs[],
	int num
) {
	struct i2c_algo_mcf_data *adap = i2c_adap->algo_data;
	struct i2c_msg *pmsg;
	int i;
	int ret=0, timeout;

	/* Skip own address */
	if (get_own(adap) == (msgs[0].addr << 1))
	{
		return -EIO;
	}

	/*  Ensure slaves are in idle state */
	if (MCF_I2SR & MCF_I2SR_IBB) {
		MCF_I2ICR = 0x00;
		MCF_I2CR  = 0x00;
		MCF_I2CR  = 0x0A;
		timeout = MCF_I2DR;
		MCF_I2SR  = 0x00;
		MCF_I2CR  = 0x00;
		MCF_I2ICR = 0x01;
	}
	
	/* setup SCL clock */
	MCF_I2FDR = get_clock(adap);
	/* set slave address */
	MCF_I2AR = get_own(adap);
	/* enable I2C module */
	MCF_I2CR = MCF_I2CR_IEN;

	MCF_I2CR |= MCF_I2CR_TXAK;

	/* Check for bus busy */
	wait_for_bb(adap);

	for (i = 0; ret >= 0 && i < num; i++) {
		pmsg = &msgs[i];

		printk(KERN_DEBUG "i2c-algo-mcf: Doing %s %d bytes to 0x%02x - %d of %d messages\n",
		     pmsg->flags & I2C_M_RD ? "read" : "write",
                     pmsg->len, pmsg->addr, i + 1, num);

		/* Send START */
		if (i == 0) {
			i2c_start(adap);
		}

		/* Wait for Bus Busy */
		wait_for_not_bb(adap);

		MCF_I2CR |= MCF_I2CR_MTX;

		ret = i2c_set_addr(adap, pmsg, i2c_adap->retries);
		if (ret < 0)
			return ret;

		/* Wait for address transfer completion */
		wait_xfer_done(adap);

		/* Check for ACK */
		if (!i2c_getack(adap)) {
			i2c_stop(adap);
			wait_for_bb(adap);
			printk(KERN_DEBUG "i2c-algo-mcf: No ack after "
				    "send address in mcf_xfer\n");
			return (-EREMOTEIO);
		}
		
		printk(KERN_DEBUG "i2c-algo-mcf: Msg %d, addr=0x%x, flags=0x%x, len=%d\n",
			    i, msgs[i].addr, msgs[i].flags, msgs[i].len);

		/* Read */
		if (pmsg->flags & I2C_M_RD) {
			/* read bytes into buffer*/
			ret = mcf_readbytes(i2c_adap, pmsg->buf, pmsg->len,
                                            (i + 1 == num));

			if (ret != pmsg->len) {
				printk(KERN_DEBUG "i2c-algo-mcf: fail: "
					    "only read %d bytes.\n",ret);
			} else {
				printk(KERN_DEBUG "i2c-algo-mcf: read %d bytes.\n",ret);
			}
		} else {
			/* write bytes into buffer*/
			ret = mcf_sendbytes(i2c_adap, pmsg->buf, pmsg->len,
                                            (i + 1 == num));
			if (ret != pmsg->len) {
				printk(KERN_DEBUG "i2c-algo-mcf: fail: "
					    "only wrote %d bytes.\n",ret);
			} else {
				printk(KERN_DEBUG "i2c-algo-mcf: wrote %d bytes.\n",ret);
			}
		}
	}

	/* Disable I2C module */
	MCF_I2CR = 0;
	
	return (i);
}


/*
 *  static void mcf_func()
 *
 *  Return algorithm funtionality
 */
static u32
mcf_func(
	struct i2c_adapter *i2c_adap
) {
	return I2C_FUNC_SMBUS_EMUL;
}


/*
 *  static void mcf_ioctl()
 *
 *  Perform I/O control
 */
static int
mcf_ioctl(
	struct i2c_adapter *adapter,
	unsigned int cmd, unsigned long arg)
{
	return 0; /* dummy */
}

/*
 *  ColdFire bus algorithm callbacks
 */
static struct i2c_algorithm mcf_algo = {
	.name		= "ColdFire I2C algorithm",
	.id		= I2C_ALGO_MCF,
	.master_xfer	= mcf_xfer,
	.algo_control	= mcf_ioctl,
	.functionality	= mcf_func,
};

/*
 *  registering functions to load algorithms at runtime
 */
int i2c_mcf_add_bus(struct i2c_adapter *adap)
{
	struct i2c_algo_mcf_data *mcf_adap = adap->algo_data;

	adap->id |= mcf_algo.id;
	adap->algo = &mcf_algo;
	adap->timeout = 100;

	mcf_i2c_init(mcf_adap);
		
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif

	i2c_add_adapter(adap);

	return 0;
}


int i2c_mcf_del_bus(struct i2c_adapter *adap)
{
	return i2c_del_adapter(adap);
}

int __init i2c_algo_mcf_init(void)
{
	if (i2c_mcf_add_bus(&i2c_mcf_board_adapter))
		return -ENODEV;
	printk("i2c-algo-mcf.o: I2C ColdFire algorithm module is loaded.\n");
	return(0);
}

/*
 * 	Driver public symbols
 */
EXPORT_SYMBOL(i2c_mcf_add_bus);
EXPORT_SYMBOL(i2c_mcf_del_bus);


#if defined(MODULE)
MODULE_DESCRIPTION("I2C-Bus driver for Coldfire M5485");

MODULE_PARM(mcf_scan, "i");
MODULE_PARM(clock, "i");
MODULE_PARM(own, "i");

int init_module(void)
{
	return i2c_algo_mcf_init();
}

void cleanup_module(void)
{
}

#endif  /* MODULE */

