/*
 *  linux/drivers/mmc/sag_mmcspi.c
 *
 *  Driver for MMC Cards
 *
 *  Copyright (C) 2007-2008 Sartorius Mechatronics. (www.sartorius.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  The following cards were successfully tested.
 *
 *    > SanDisk 1.0 GB SD.
 *    > SanDisk Ultra II, 1GB SD.
 *    > SanDisk Extreme III, 1GB SD.
 *    > SanDisk 4.0 GB SDHC.
 *    > AGFA 1GB SD.
 *    > AGFA High Speed 1GB SD.
 *    > Lexar 1GB SD.
 *    > Lexar 4GB SDHC.
 *    > Kingston 2GB SD.
 *    > CnMemory First Class Silver 1GB SD.
 *    > Toshiba 256MB SD.
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/dma-mapping.h>
#include <linux/mmc/card.h>
#include <linux/mmc/host.h>
#include <linux/mmc/protocol.h>
#include <asm/scatterlist.h>

#include <asm/mcfsim.h>

#include <sag_qspi.h>

#include "mmc_crc16.h"

#define DRIVER_VERSION "0.2"
#define DRIVER_NAME	"mmcqspi"
#define SPIMMC_DETECT_TIMEOUT HZ /* 1 sec */

#define HOST_IDLE    	0
#define HOST_ACTIVE  	(1<<0)
#define HOST_NEEDINIT	(1<<1)

#define SPI_PARA_DEF_MODE 3


#define QSPI_CR_BITSE_8  0x00    // 8 bits per transfer
#define QSPI_CR_BITSE_16 0x40    // QMR[BITS] bits per transfer

#define QSPI_CR_CONT_NEXT 0x80	// keep CS asserted between transfer
#define QSPI_CR_CONT_STOP 0x00	// drop CS once between transfer


#define DEBUG_MMC_SPI 0
#define DEBUG_FNT 0

#if (DEBUG_MMC_SPI==1)
	#define mmc_dbg(fmt...) printk(fmt)
#else
	#define mmc_dbg(fmt...) do { } while (0)
#endif

#if (DEBUG_FNT==1)
	#define PR_FUNC mmc_dbg("MMC SPI: %s\n",__FUNCTION__)
#else
	#define PR_FUNC do { } while (0)
#endif


#define MMC_TOCKEN_START_MBLK 0xFC
#define MMC_TOCKEN_STOP_TRAN  0xFD
#define MMC_TOCKEN_START_SBLK 0xFE

#define MMC_RW_CMD_FAILED  1
#define MMC_RW_DATA_FAILED 2

enum E_MMC_STATUS
{
	MMC_STATUS_OK=0,
	/* R1 */
	MMC_STATUS_IN_IDLE_STATE   = 1<<0,
	MMC_STATUS_ERASE_RESET     = 1<<1,
	MMC_STATUS_ILLEGAL_COMMAND = 1<<2,
	MMC_STATUS_COM_CRC_ERROR   = 1<<3,
	MMC_STATUS_ERASE_SEQ_ERROR = 1<<4,
	MMC_STATUS_ADDRESS_ERROR   = 1<<5,
	MMC_STATUS_PARAMETER_ERROR = 1<<6,
	/* R2 */
	MMC_STATUS_CARD_LOCKED     = 1<<8,
	MMC_STATUS_WP_ERASE_SKIP   = 1<<9,
	MMC_STATUS_ERROR           = 1<<10,
	MMC_STATUS_CC_ERROR        = 1<<11,
	MMC_STATUS_ECC_FAILED      = 1<<12,
	MMC_STATUS_WP_VIOLATION    = 1<<13,
	MMC_STATUS_ERASE_PARAM     = 1<<14,
	MMC_STATUS_OUT_OF_RANGE    = 1<<15,

	MMC_STATUS_WR_PROT=0x102,
	MMC_STATUS_CRC_ERROR=0x101,

	MMC_STATUS_BUSY=0xFF,
};


struct sag_mmc_host
{
	struct mmc_host*   	mmc;		/* MMC structure */
	spinlock_t         	lock;		/* Mutex */
	struct mmc_request*	mrq;		/* Current request */
	u32 flags;
	struct timer_list  	timer;		/* Check for card updates in a defined interval */
	qspi_dev*          	spi_dev;
};

/**
 * Perform baud rate calculation for QMR register
 * @param desired desired baud rate in hz
 * @returns divider setting for QMR, -1 for invalid settings
 */
static u16 qspi_dev_BAUD(u32 desired)
{
	u16 baud;
	//JJvB: avoid div-by-zero exception
	if(desired==0) return -1;
#if defined(CONFIG_M532x) || defined(CONFIG_M5249) || defined(CONFIG_M527x)
	baud =  MCF_BUSCLK/(2*desired);
#else
	baud =  MCF_CLK/(2*desired);
#endif
	if((baud>0x00FF)||(baud<2))
		return -1;//invalid setting

	return baud;
}


static unsigned char crc7(const u8*data,unsigned len)
{
	unsigned a,i;
	unsigned crc=0;
	for(a=0;a<len;a++)
	{
		unsigned byte=*data++;
		for(i=0;i<8;i++)
		{
			crc<<=1;
			if((byte & 0x80)^(crc & 0x80))
				crc ^=0x09;
			byte <<= 1;
		}
	}
	return crc&127;
}

static void sag_mmc_power_on(void)
{
	MCF_GPIO_PODR_SSI &= ~(MCF_GPIO_PODR_SSI_PODR_SSI2);
}

static void sag_mmc_power_off(void)
{
	MCF_GPIO_PODR_SSI |= (MCF_GPIO_PODR_SSI_PODR_SSI2);
}

static int sag_mmc_card_inserted(void)
{
	// The signal is low active, so return the inverse
	return (MCF_GPIO_PPDSDR_SSI & MCF_GPIO_PCLRR_SSI_PCLRR_SSI3 ? 0 : 1);
}

static int sag_mmc_card_ro(void)
{
#ifdef CONFIG_SAG_LABPRO_CORE
	// JJvB: Really the input is a Write Enable
	return (MCF_GPIO_PPDSDR_SSI & MCF_GPIO_PCLRR_SSI_PCLRR_SSI4 ? 1 : 0);
#else
	// The signal is low active, so return the inverse
	return (MCF_GPIO_PPDSDR_SSI & MCF_GPIO_PCLRR_SSI_PCLRR_SSI4 ? 0 : 1);
#endif
}

static void sag_mmc_poll_event(unsigned long arg)
{
	struct sag_mmc_host *host = (struct mmcspi_host *) arg;

	int card = sag_mmc_card_inserted();
	int controller;

	spin_lock(&(host->lock));
	controller = (host->flags & HOST_ACTIVE) ? 1 : 0;

	if (card != controller) {
		if (host->flags & HOST_NEEDINIT) // Don't hock up twice
			goto out;

		host->flags &= ~HOST_ACTIVE;

		if (card) {
			host->flags |= HOST_ACTIVE|HOST_NEEDINIT;
		}

		printk(KERN_INFO DRIVER_NAME ": card %s\n", 
			card?"inserted":"removed");

		mmc_detect_change(host->mmc, 0);
	}
out:
	spin_unlock(&(host->lock));
	mod_timer(&host->timer, jiffies + SPIMMC_DETECT_TIMEOUT);
}



/**=========================================================================
	Helper functions
*/

#define TRANS_BITS 16

static void mmc_spi_dummy_clocks(qspi_dev *dev)
{

#define NUM_PULSE 80
#define NUM_PULSE_BITS (NUM_PULSE/TRANS_BITS)
	int n;
	SPI_PARA para;

	PR_FUNC;

	for (n=0 ; n<NUM_PULSE_BITS ; ++n)
	{
		para.wr[n] = 0xFFFF;
		para.cr[n] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|0x0F;
	}
	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = NUM_PULSE_BITS-1;
	para.sync = 0;
	qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);
}


static void mmc_spi_do_dummy_clocks(struct sag_mmc_host *host/*qspi_dev *dev*/)
{
	PR_FUNC;
	spin_lock(&(host->lock));
	if (host->flags & HOST_NEEDINIT) {
		spin_unlock(&(host->lock));
#if defined(CONFIG_M532x)
		/*
		 * We need to generate 74 CLK Pulses with CS HIGH!
		 * The controller doesn't do this, so we need to set
		 * CS0 to GPIO, push 74 dummy bits out on the line and
		 * reset the CS0 pin to QSPI_PCS0, yes that's ugly but it should
		 * work for now
		 */
		mmc_spi_dummy_clocks(host->spi_dev);
#else
		#error I don't know how to init the card with CS HIGH on your Board.
#endif
		spin_lock(&(host->lock));
		host->flags &= ~HOST_NEEDINIT;
	}
	spin_unlock(&(host->lock));

}

inline void print_resp(u16 data)
{
#if (DEBUG_MMC_SPI==1)

	if ((data&MMC_STATUS_IN_IDLE_STATE) == MMC_STATUS_IN_IDLE_STATE)
		printk("MMC_STATUS_IN_IDLE_STATE ");

	if ((data&MMC_STATUS_ERASE_RESET) == MMC_STATUS_ERASE_RESET)
		printk("MMC_STATUS_ERASE_RESET ");

	if ((data&MMC_STATUS_ILLEGAL_COMMAND) == MMC_STATUS_ILLEGAL_COMMAND)
		printk("MMC_STATUS_ILLEGAL_COMMAND ");

	if ((data&MMC_STATUS_CRC_ERROR) == MMC_STATUS_COM_CRC_ERROR)
		printk("MMC_STATUS_COM_CRC_ERROR ");

	if ((data&MMC_STATUS_ERASE_SEQ_ERROR) == MMC_STATUS_ERASE_SEQ_ERROR)
		printk("MMC_STATUS_ERASE_SEQ_ERROR ");

	if ((data&MMC_STATUS_ADDRESS_ERROR) == MMC_STATUS_ADDRESS_ERROR)
		printk("MMC_STATUS_ADDRESS_ERROR ");

	if ((data&MMC_STATUS_PARAMETER_ERROR) == MMC_STATUS_PARAMETER_ERROR)
		printk("MMC_STATUS_PARAMETER_ERROR ");

	if (data == MMC_STATUS_BUSY)
		printk("MMC_STATUS_BUSY ");

	if (data == MMC_STATUS_OK)
		printk("MMC_STATUS_OK ");

	printk("\n");
#endif

	return;
}


static void mmc_cs_off(qspi_dev *dev)
{
	SPI_PARA para;

	PR_FUNC;

	para.wr[0] = 0xFFFF;
	para.cr[0] = QSPI_CR_CONT_STOP|QSPI_CR_BITSE_8|0x0F;
	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = 0;
	para.sync = 0;

	qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);
}


static u8 mmc_spi_read_byte(qspi_dev *dev)
{
	SPI_PARA para;

	para.wr[0] = 0xFF;
	para.cr[0] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_8|dev->num;
	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = 0;
	para.sync = 0;

	qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);

	return (u8)para.rd[0];
}

static u8 mmc_spi_wait_for_r1(qspi_dev *dev)
{
	u8 byte;
	int n;

	PR_FUNC;

	for (n=0 ; n<0xFF ; n++)
	{
		byte = mmc_spi_read_byte(dev);
		if (byte & 0x80) {
			udelay(100);
			continue; // if first bit is set
		}
		break;
	}

	return byte;
}
static u8 mmc_spi_wait_for_start_token(qspi_dev *dev)
{
	u8 byte;
	int n;

	PR_FUNC;

	for (n=0 ; n<0xFF ; n++)
	{
		byte = mmc_spi_read_byte(dev);
		if (byte == MMC_TOCKEN_START_SBLK)
			break;
		udelay(100);
	}

	return byte;
}
/*
static u8 mmc_spi_wait_until_busy(qspi_dev *dev)
{
	u8 byte;
	int n;

	PR_FUNC;

	for (n=0 ; n<0xFF ; n++)
	{
		byte = mmc_spi_read_byte(dev);
		if (byte != MMC_STATUS_BUSY)
			break;
		udelay(100);
	}

	return byte;
}
*/
static u8 mmc_spi_wait_for_data_resp_token(qspi_dev *dev)
{
	u8 byte;
	int n;

	PR_FUNC;

	for (n=0 ; n<0xFF ; n++)
	{
		byte = mmc_spi_read_byte(dev);
		if (!(byte&0x10) && (byte&0x01))
			break;
		udelay(100);
	}

	return byte;
}

static u16 mmc_spi_read_n(qspi_dev *dev, void *vdata, unsigned len, unsigned calc_crc)
{
	SPI_PARA para;
	u16 sync;
	unsigned have_crc=0;
	unsigned want_crc;
	int sts;
	u16* wdata = (u16*)vdata;

	PR_FUNC;

	if (len & 1)
		return MMC_STATUS_PARAMETER_ERROR;

	sync = mmc_spi_wait_for_start_token(dev);
	if (sync != MMC_TOCKEN_START_SBLK)
	{
		printk("read block error: %02X ", sync);
		/* check for data error token */
		if ((sync&0xF0) == 0) {
			sync = 0;
			printk("'data error token: ");
			if (sync&0x01) {
				printk("error");
				sync |= MMC_STATUS_ERROR;
			}
			if (sync&0x02) {
				printk("CC error");
				sync |= MMC_STATUS_CC_ERROR;
			}
			if (sync&0x04) {
				printk("Card ECC Failed");
				sync |= MMC_STATUS_ECC_FAILED;
			}
			if (sync&0x08) {
				printk("out of range");
				sync |= MMC_STATUS_OUT_OF_RANGE;
			}
			printk("'");
		}
		printk("\n");
		return sync;
	}

	while (len)
	{
		unsigned put = 0;
		unsigned blk = len;
		u8 *mrk;
		unsigned get=0;
		unsigned l;

		if (blk>32)
			blk = 32;
		for (l=blk ; l ; l-=2)
		{
			para.wr[put] = 0xFFFF;
			para.cr[put++] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num;
		}
		para.mode = SPI_PARA_DEF_MODE;
		para.clock = dev->baud;
		para.newqp = 0;
		para.endqp = put-1;
		para.sync = 0;

		sts = qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);

		mrk = (u8*)wdata;
		for (l=blk ; l ; l-=2)
			*wdata ++= para.rd[get++];

		if (calc_crc)
			have_crc = mmc_crc16(have_crc, mrk, blk);
		len-=blk;
	}

	if (calc_crc) {
		para.wr[0] = 0xFFFF;
		para.cr[0] = QSPI_CR_CONT_STOP|QSPI_CR_BITSE_16|dev->num;
		para.mode = SPI_PARA_DEF_MODE;
		para.clock = dev->baud;
		para.newqp = 0;
		para.endqp = 0;
		para.sync = 0;

		sts = qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);

		want_crc = para.rd[0];
		if (want_crc != have_crc)
		{
			mmc_dbg("CRC16 mismatch want=%04X have=%04X\n", want_crc, have_crc);
			printk("read block CRC error\n");
			return MMC_STATUS_CRC_ERROR;
		}
	}

	return MMC_STATUS_OK;
}


static u16 mmc_spi_write_n(qspi_dev *dev, const void*vdata, unsigned len)
{
	SPI_PARA para;
	u16 resp;
	int sts;
	unsigned have_crc = 0;
	const u16 *wdata;
	unsigned l;

	if (len&1)
		return MMC_STATUS_PARAMETER_ERROR;

	// write sync
	para.wr[0] = 0xFE;
	para.cr[0] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_8|dev->num;
	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = 0;
	para.sync = 0;

	sts = qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);

	have_crc = mmc_crc16(have_crc, (u8*)vdata, len);
	wdata = (const u16*)vdata;

	while (len)
	{
		unsigned put = 0;
		unsigned blk = len;

		if (blk>32)
			blk = 32;

		for (l=blk ; l ; l-=2)
		{
			para.wr[put] = *wdata++;
			para.cr[put++] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num;
		}

		para.mode = SPI_PARA_DEF_MODE;
		para.clock = dev->baud;
		para.newqp = 0;
		para.endqp = put-1;
		para.sync = 0;

		sts = qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);
		len -= blk;
	}

	para.wr[0] = have_crc;
	para.cr[0] = QSPI_CR_CONT_STOP|QSPI_CR_BITSE_16|dev->num;
	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = 0;
	para.sync = 0;

	sts = qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);

	resp = mmc_spi_wait_for_data_resp_token(dev);

	if ((resp&0x1F)==0x0D) {
		printk("write: data write error\n");
		return MMC_STATUS_ERROR;
	}

	if ((resp&0x1F)==0x0B) {
		printk("write: data CRC error\n");
		return MMC_STATUS_CRC_ERROR;
	}

	if ((resp&0x1F)!=0x05) {
		printk("write: data not accepted\n");
		return MMC_STATUS_ERROR;// ?? should not happen
	}

	// wait while busy
	for (l=0 ; l<0xFFFF ; ++l)
	{
		resp = mmc_spi_read_byte(dev);
		if (resp==0xFF)
			break;	// not busy any more

		udelay(100);
	}

	return MMC_STATUS_OK;
}


/**=========================================================================
	MMC SPI
*/



static u8 mmc_spi_doR1(qspi_dev *dev, u32 cmd, u32 arg)
{
	SPI_PARA para;
	unsigned crc=0xFF;
	u8 cmd_byte = 0x40 | (cmd & 0x3F);
	u8 resp = 0;

	PR_FUNC;

	if (cmd == MMC_GO_IDLE_STATE)
		crc=0x95;

	para.wr[0]=0xFF;              // 0x00|FF
	para.wr[1]=0xFF00|cmd_byte;   // 0xFF|cmd
	para.wr[2]=arg>>16;           // ARG1|ARG2
	para.wr[3]=arg>>0;            // ARG3|ARG4

	//JJvB: this is the right thing,
	// but then kernel thinks, it might by a SDHC card
	if(cmd == SD_SEND_IF_COND)
		crc=(crc7(((u8*)para.wr)+3, 5)<<1)|1;

	para.wr[4]=(crc<<8)|0x00FF;   // CRC|0xFF
	para.wr[5]=0xFFFF;            // 0xFF|FF

	para.cr[0] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_8|0x0F;      // 0x00
	para.cr[1] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num; // 0xFF|FF
	para.cr[2] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num; // CMD|ARG1
	para.cr[3] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num; // ARG2|ARG3
	para.cr[4] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_16|dev->num; // ARG4|CRC
	para.cr[5] = QSPI_CR_CONT_NEXT|QSPI_CR_BITSE_8|dev->num;  // 0xFF

	para.mode = SPI_PARA_DEF_MODE;
	para.clock = dev->baud;
	para.newqp = 0;
	para.endqp = 5;
	para.sync = 0;

	qspi_dev_io_write_read(dev, &para, MEM_TYPE_KERNEL);
	resp = para.rd[5];

	if (resp == MMC_STATUS_BUSY)
		resp = mmc_spi_wait_for_r1(dev);

	mmc_dbg("R1(CMD%d) arg=%08X: %02X => ", cmd, arg, resp);

	print_resp(resp);

	return resp;
}

static u16 mmc_spi_doR2(qspi_dev *dev, u32 cmd, u32 arg)
{
	u16 resp;

	PR_FUNC;

	qspi_ioctl_lock();
	resp = mmc_spi_doR1(dev, cmd, arg);
	if (resp == MMC_STATUS_OK) {
		resp |= mmc_spi_read_byte(dev)<<8;
	}
	qspi_ioctl_unlock();

	mmc_dbg("R2: %04X\n", resp);
	return resp;
}

static u32 _mmc_spi_doR3(qspi_dev *dev, u32 cmd, u32 arg, u32 *rresp)
{
	u32 resp = 0;
	u8 byte = 0;
	PR_FUNC;

	qspi_ioctl_lock();
	resp = mmc_spi_doR1(dev, cmd, arg);
	if (((resp&~MMC_STATUS_IN_IDLE_STATE) == MMC_STATUS_OK) && rresp) {
		// read 4 bytes
		byte = mmc_spi_read_byte(dev);
		mmc_dbg("%02X ",byte);
		*rresp = 0|byte<<24;

		byte = mmc_spi_read_byte(dev);
		mmc_dbg("%02X ",byte);
		*rresp |= byte<<16;

		byte = mmc_spi_read_byte(dev);
		mmc_dbg("%02X ",byte);
		*rresp |= byte<<8;

		byte = mmc_spi_read_byte(dev);
		mmc_dbg("%02X ",byte);
		*rresp |= byte;
	}
	qspi_ioctl_unlock();

	mmc_dbg("R3: %08X\n", *rresp);
	return resp;
}

static u32 mmc_spi_doR3(qspi_dev *dev, u32 cmd, u32 arg, u32 *resp)
{
	PR_FUNC;

	return _mmc_spi_doR3(dev, cmd, arg, resp);
}

static u32 mmc_spi_doR7(qspi_dev *dev, u32 cmd, u32 arg, u32 *resp)
{
	return _mmc_spi_doR3(dev, cmd, arg, resp);
}

static u32 mmc_spi_readSingleBlock(qspi_dev *dev, char *buffer, int arg, int len)
{
	u32 err = 0;

	qspi_ioctl_lock();

	err = mmc_spi_doR1(dev, MMC_READ_SINGLE_BLOCK, arg);
	if (err != MMC_STATUS_OK) {
		err = MMC_RW_CMD_FAILED;
	}
	else {
		err = mmc_spi_read_n(dev, buffer, len, 1);
		if (err != MMC_STATUS_OK) {
			err = MMC_RW_DATA_FAILED;
		}
	}

	qspi_ioctl_unlock();

	return err;
}

static u32 mmc_spi_writeSingleBlock(qspi_dev *dev, const char *buffer, int arg, int len)
{
	u32 err = 0;

	qspi_ioctl_lock();

	err = mmc_spi_doR1(dev, MMC_WRITE_BLOCK, arg);
	if (err != MMC_STATUS_OK) {
		err = MMC_RW_CMD_FAILED;
	}
	else {
		err = mmc_spi_write_n(dev, buffer, len);
		if (err != MMC_STATUS_OK) {
			err = MMC_RW_DATA_FAILED;
		}
	}

	qspi_ioctl_unlock();

	return err;
}

/**=========================================================================

*/


static u32 sag_mmc_get_register(struct sag_mmc_host *host, u32 opcode, u32 count)
{
	struct mmc_command *cmd = host->mrq->cmd;
	int counter=0;
	u32 u32data = 0;
	u8 data[20];

	PR_FUNC;

	qspi_ioctl_lock();

	cmd->error = mmc_spi_doR1(host->spi_dev, opcode, 0);
	if (cmd->error & MMC_STATUS_ILLEGAL_COMMAND) {
		cmd->error = MMC_ERR_INVALID;
	}
	else {
		cmd->error = mmc_spi_read_n(host->spi_dev, data, count, 0);
		if (cmd->error != MMC_STATUS_OK) {
			cmd->error = MMC_ERR_FAILED;
		}
	}

	qspi_ioctl_unlock();

	if (cmd->error == MMC_STATUS_OK) {
		for(counter=0; counter < count; counter++)
		{
			u32data <<= 8;
			u32data |= data[counter];
			if(!((counter+1) % 4))
			{
				cmd->resp[(counter/4)] = u32data;
				u32data = 0;
			}
		}
	}

	return cmd->error;
}


static void sag_mmc_request(struct mmc_host *mmc, struct mmc_request *mrq)
{
	struct mmc_command	*cmd;
	struct sag_mmc_host	*host;
	struct mmc_data *datablk;

	u32 err = 0;
	char* buffer;
	u32 opcode = 0;

	PR_FUNC;

#define command_pr(fmt...) mmc_dbg(fmt)
// #define command_pr(fmt...) printk(fmt)
// #define command_pr(fmt...) do { } while (0)

	host = mmc_priv(mmc);
	cmd = mrq->cmd;
	datablk = cmd->data;
	host->mrq = mrq;

	/*
	 * Put the command and the data on the bus ...
	 * reread everything to a local buffer and copy this back
	 */
	spin_lock(&(host->lock));
	err = (host->flags & HOST_ACTIVE);
	spin_unlock(&(host->lock));

	command_pr("cmd opcode: %d => ",cmd->opcode);

	if(!err) {
		mmc_dbg("Request with no card insered\n");
		cmd->error = MMC_ERR_FAILED;
		goto out;
	}

	err = 0;

	mmc_spi_do_dummy_clocks(host);

	switch(cmd->opcode) {

		case MMC_GO_IDLE_STATE:	/* CMD0 */
		/* SD Mode : bc 	| don't care	| none	*/
		/* SPI Mode: yes	| none      	| R1  	*/
			command_pr("MMC_GO_IDLE_STATE\n");
			err = mmc_spi_doR1(host->spi_dev, cmd->opcode, 0);

			if (!(err&MMC_STATUS_IN_IDLE_STATE))
				goto out;

			cmd->resp[0] = 0xFFFFFF00 | err;

			break;

		case MMC_READ_MULTIPLE_BLOCK: /* CMD17 */
		/* SD Mode : adtc	| [31:0] data adress	| R1	*/
		/* SPI Mode: yes	| [31:0] data adress	| R1	*/
			command_pr("MMC_READ_MULTIPLE_BLOCK maped to ");

		case MMC_READ_SINGLE_BLOCK: /* CMD18 */ {
		/* SD Mode : adtc	| [31:0] data adress	| R1	*/
		/* SPI Mode: yes	| [31:0] data adress	| R1	*/

			int i = 0;
			u32 addr = 0;
			unsigned sectno = 0;
			u32 arg = cmd->arg;
			int retries = 3;

			command_pr("MMC_READ_SINGLE_BLOCK\n");

			buffer = page_address(datablk->sg->page) + datablk->sg->offset;

			while(i < datablk->blocks) {

				if (mmc_host_mode_sdhc(host->mmc)) {
					sectno = arg;
					addr = arg*datablk->blksz;
				}
				else {
					addr = arg;
					sectno = (addr==0)?0:addr/datablk->blksz;
				}

				mmc_dbg("Reading block %d (%d bytes) addr: 0x%08X\n",
					sectno, datablk->blksz, addr);

				err = mmc_spi_readSingleBlock(host->spi_dev, buffer, arg, datablk->blksz);
				if (err) {
					printk(DRIVER_NAME ": read block %d (%d bytes) fails\n", sectno, datablk->blksz);
					if (retries--)
						continue;

					if (err == MMC_RW_CMD_FAILED)
						cmd->error = MMC_ERR_FAILED;
					else
						datablk->error = MMC_ERR_FAILED;

					goto out;
				}

				if (mmc_host_mode_sdhc(host->mmc))
					arg++;
				else
					arg += datablk->blksz;
				buffer += datablk->blksz;
				i++;
				retries = 3;
				datablk->bytes_xfered += datablk->blksz;
			}

			break;
		}

		case MMC_SEND_STATUS: { /* CMD13 */
		/* SD Mode : ac 	| [31:16] RCA [15:0] don't care	| R1	*/
		/* SPI Mode: yes	| none                         	| R2	*/
			command_pr("MMC_SEND_STATUS\n");

			cmd->resp[0] = mmc_spi_doR2(host->spi_dev, cmd->opcode, 0);
			if (cmd->resp[0] != MMC_STATUS_OK) {
				cmd->error = MMC_ERR_FAILED;
				goto out;
			}

			cmd->resp[0] = R1_READY_FOR_DATA;

			break;
		}
		case MMC_SET_BLOCKLEN: /* CMD16 */
		/* SD Mode : ac 	| [31:0] block length	| R1	*/
		/* SPI Mode: yes	| [31:0] block length	| R1	*/
			command_pr("MMC_SET_BLOCKLEN\n");

			err = mmc_spi_doR1(host->spi_dev, cmd->opcode, cmd->arg);
			if (err&MMC_STATUS_ILLEGAL_COMMAND) {
				cmd->error = MMC_ERR_INVALID;
				goto out;
			}

			if (err != MMC_STATUS_OK) {
				cmd->error = MMC_ERR_FAILED;
				goto out;
			}

			cmd->resp[0] = 0xFFFFFF00 | err;

			break;

		case MMC_WRITE_MULTIPLE_BLOCK: /* CMD25 */
		/* SD Mode : adtc 	| [31:0] data address	| R1	*/
		/* SPI Mode: yes	| [31:0] data address	| R1	*/
			command_pr("MMC_WRITE_MULTIPLE_BLOCK\n");
			break;

		case MMC_WRITE_BLOCK: /* CMD24 */ {
		/* SD Mode : adtc 	| [31:0] data address	| R1	*/
		/* SPI Mode: yes	| [31:0] data address	| R1	*/

			u32 addr = 0;
			unsigned sectno = 0;

			command_pr("MMC_WRITE_BLOCK\n");

			if (mmc_host_mode_sdhc(host->mmc)) {
				sectno = cmd->arg;
				addr = sectno*datablk->blksz;
			}
			else {
				addr = cmd->arg;
				sectno = (addr==0)?0:addr/datablk->blksz;
			}

			mmc_dbg("Writing block %d (%d bytes) addr: 0x%08X\n",
				sectno, datablk->blksz, addr);

			buffer = page_address(datablk->sg->page) + datablk->sg->offset;

			err = mmc_spi_writeSingleBlock(host->spi_dev, buffer, cmd->arg, datablk->blksz);
			if (err) {
				printk(DRIVER_NAME ": write block %d (%d bytes) fails\n", sectno, datablk->blksz);

				if (err == MMC_RW_CMD_FAILED)
					cmd->error = MMC_ERR_FAILED;
				else
					datablk->error = MMC_ERR_FAILED;
				goto out;
			}

			datablk->bytes_xfered = datablk->blksz;
			cmd->resp[0] = 0xFFFFF00;

			break;
		}

		/* Half FAKE */
		case MMC_ALL_SEND_CID: /* CMD2 */
		/* SD Mode : bcr 	| [31:0] don't care	| R2	*/
		/* SPI Mode: no */
			command_pr("MMC_ALL_SEND_CID\n");
			if (mmc_host_is_spi(host->mmc))
				printk(DRIVER_NAME ": CMD%d (MMC_ALL_SEND_CID) isn't supported in SPI mode\n", cmd->opcode);

			cmd->error = MMC_ERR_FAILED;
			goto out;

		case MMC_SEND_CID: /* CMD10 */
		/* SD Mode : ac 	| [31:16] RCA [15:0] don't care	| R2	*/
		/* SPI Mode: yes	| none                         	| R1	*/
			command_pr("MMC_SEND_CID\n");
			opcode = MMC_SEND_CID;

		case MMC_SEND_CSD: {/* CMD9 */
		/* SD Mode : ac 	| [31:16] RCA [15:0] don't care	| R2	*/
		/* SPI Mode: yes	| none                         	| R1	*/
			command_pr("MMC_SEND_CSD\n");
			if (opcode == 0)
				opcode = MMC_SEND_CSD;

			err = sag_mmc_get_register(host, opcode, 16);
			if (err != MMC_STATUS_OK) {
				goto out;
			}

			break;
		}

		case MMC_APP_CMD: /* CMD55 */
		/* SD Mode : ac 	| [31:16] RCA [15:0] stuff bits	| R1	*/
		/* SPI Mode: yes	| stuff bits                   	| R1	*/
			command_pr("MMC_APP_CMD\n");
			err =  mmc_spi_doR1(host->spi_dev, cmd->opcode, cmd->arg);

			if (err&MMC_STATUS_ILLEGAL_COMMAND) {
				cmd->error = MMC_ERR_INVALID;
				goto out;
			}

			if (!(err == MMC_STATUS_OK || (err&MMC_STATUS_IN_IDLE_STATE))) {
				cmd->error = MMC_ERR_TIMEOUT;
				goto out;
			}

			cmd->resp[0] = R1_APP_CMD;

			break;

/// ---------------------------------------------------------------

		case SD_SEND_IF_COND: { // CMD8
			command_pr("SD_SEND_IF_COND\n");
			err = mmc_spi_doR7(host->spi_dev, cmd->opcode, cmd->arg, &cmd->resp[0]);

			if (err&MMC_STATUS_ILLEGAL_COMMAND) {
				cmd->error = MMC_ERR_INVALID;
				goto out;
			}

			if (err&MMC_STATUS_COM_CRC_ERROR) {
				cmd->error = MMC_ERR_BADCRC;
				goto out;
			}

			break;
		}

		case MMC_SPI_READ_OCR:
			command_pr("MMC_SPI_READ_OCR\n");

			err = mmc_spi_doR3(host->spi_dev, cmd->opcode, cmd->arg, &cmd->resp[0]);

			if (((err&~MMC_STATUS_IN_IDLE_STATE) != MMC_STATUS_OK)) {
				cmd->error = MMC_ERR_FAILED;
				goto out;
			}

			break;

		case SD_APP_OP_COND: /* ACMD41 */
		/* SD Mode : bcr	| [31:0] OCR without busy	| R3	*/
		/* SPI Mode: yes	| none                   	| R1	*/
			command_pr("SD_APP_OP_COND\n");

		/* Do the init process for 10 times */
		case MMC_SEND_OP_COND: { /* CMD1 */
		/* SD Mode : reserved	*/
		/* SPI Mode: yes	| none      	| R1  	*/
			command_pr("MMC_SEND_OP_COND\n");

			err = mmc_spi_doR1(host->spi_dev, cmd->opcode, cmd->arg);

			if (err & MMC_STATUS_IN_IDLE_STATE) {
				cmd->error = MMC_ERR_FAILED;
				goto out;
			}

			if (err & MMC_STATUS_ILLEGAL_COMMAND) {
				cmd->error = MMC_ERR_INVALID;
				goto out;
			}

			mmc_dbg("%s: Card initialisation completted.\n", mmc_hostname(mmc));
			break;
		}
		/// ---------------------------------------------------------------

		case SD_APP_SEND_SCR: {
			command_pr("SD_APP_SEND_SCR\n");

			err = sag_mmc_get_register(host, cmd->opcode, 8);
			if (err != MMC_STATUS_OK) {
				goto out;
			}

			mmc_dbg("buffer : %04X %04X", cmd->resp[0], cmd->resp[1]);
			break;
		}

		/** *************************************** */
		case SD_SWITCH: /* CMD6 */ /* MMC_SWITCH */ /* SD_APP_SET_BUS_WIDTH */ {

			if (mmc_host_mode_mmc(host->mmc)) {
				command_pr("MMC_SWITCH\n");
			}

			if (mmc_host_mode_sd(host->mmc)) {
				if((cmd->flags & MMC_CMD_ADTC) && mmc_host_is_spi(host->mmc)/*host->mmc->bus_mode == MMC_BUS_SPI*/) {
					/* SD_SWITCH */
					/* only in SPI mode */
					/* SPI Mode: yes	| [31] mode [7:4] function group 2 [3:0]function group 1	| R1 (512 block)	*/
					command_pr("SD_SWITCH\n");
					err = mmc_spi_doR1(host->spi_dev, cmd->opcode, cmd->arg);

					if (err & MMC_STATUS_ILLEGAL_COMMAND) {
						cmd->error = MMC_ERR_INVALID;
						goto out;
					}

					if (err != MMC_STATUS_OK) {
						cmd->error = MMC_ERR_FAILED;
						goto out;
					}

					// read block
					if (cmd->flags == MMC_DATA_READ) {
						buffer = page_address(datablk->sg->page) + datablk->sg->offset;
						err = mmc_spi_read_n(host->spi_dev, buffer, mrq->data->sg->length, 1); /* len == 512 */
					}
				}
				if((cmd->flags & MMC_CMD_AC) && !mmc_host_is_spi(host->mmc)/*host->mmc->bus_mode != MMC_BUS_SPI*/) {
					/* SD_APP_SET_BUS_WIDTH */
					command_pr("SD_APP_SET_BUS_WIDTH\n");
				}
			}

			break;
		}

		/** *************************************** */
		/* FAKE */ /* not suppurted in SPI mode */

		case MMC_SELECT_CARD: /* CMD7; R1*/
		/* SD Mode : ac 	| [31:16] RCA [15:0] stuff bits	| R1	*/
		/* SPI Mode: no */
			command_pr("MMC_SELECT_CARD\n");
			if (mmc_host_is_spi(host->mmc))
				printk(DRIVER_NAME ": CMD%d (MMC_SELECT_CARD) isn't supported in SPI mode\n", cmd->opcode);

			cmd->error = MMC_ERR_FAILED;
			goto out;

		case MMC_SET_RELATIVE_ADDR: /*(SD_SEND_RELATIVE_ADDR)*//* CMD3; R6*/
		/* SD Mode : ac 	| [31:16] RCA [15:0] stuff bits	| R1	*/
		/* SPI Mode: no */
			if (mmc_host_mode_sd(host->mmc))
				command_pr("MMC_SET_RELATIVE_ADDR\n");
			else
				command_pr("SD_SEND_RELATIVE_ADDR\n");

			if (mmc_host_is_spi(host->mmc))
				printk(DRIVER_NAME ": CMD%d (%s) isn't supported in SPI mode\n", 
					cmd->opcode, mmc_host_mode_sd(host->mmc)?
					"SD_SEND_RELATIVE_ADDR":"MMC_SET_RELATIVE_ADDR");

			cmd->error = MMC_ERR_FAILED;
			goto out;

		default:
			mmc_dbg(" Unknown Command [%s MODE]:\nOpCode: %d | ARG:0x%X\n",
				(mmc->mode == MMC_MODE_MMC) ? "MMC" : "SD",
				mrq->cmd->opcode, mrq->cmd->arg);
	}

	mmc_cs_off(host->spi_dev);

	cmd->error = MMC_ERR_NONE;
out:
	mmc_request_done(mmc, mrq);
}

static int sag_mmc_get_ro(struct mmc_host *mmc)
{
	return sag_mmc_card_ro();
}


inline void print_ios(struct mmc_ios *ios)
{
#if 0
#if (DEBUG_MMC_SPI==1)

	printk("Clock: %d (%03d)\n", ios->clock,qspi_dev_BAUD(ios->clock));

	mmc_dbg("VDD: ");
	switch(ios->vdd) {
	case MMC_VDD_150: printk("150"); break;
	case MMC_VDD_155: printk("155"); break;
	case MMC_VDD_160: printk("160"); break;
	case MMC_VDD_165: printk("165"); break;
	case MMC_VDD_170: printk("170"); break;
	case MMC_VDD_180: printk("180"); break;
	case MMC_VDD_190: printk("190"); break;
	case MMC_VDD_200: printk("200"); break;
	case MMC_VDD_210: printk("210"); break;
	case MMC_VDD_220: printk("220"); break;
	case MMC_VDD_230: printk("230"); break;
	case MMC_VDD_240: printk("240"); break;
	case MMC_VDD_250: printk("250"); break;
	case MMC_VDD_260: printk("260"); break;
	case MMC_VDD_270: printk("270"); break;
	case MMC_VDD_280: printk("280"); break;
	case MMC_VDD_290: printk("290"); break;
	case MMC_VDD_300: printk("300"); break;
	case MMC_VDD_310: printk("310"); break;
	case MMC_VDD_320: printk("320"); break;
	case MMC_VDD_330: printk("330"); break;
	case MMC_VDD_340: printk("340"); break;
	case MMC_VDD_350: printk("350"); break;
	case MMC_VDD_360: printk("360"); break;
	}
	printk("\n");

	printk("Bus mode: ");
	switch (ios->bus_mode) {
	case MMC_BUSMODE_OPENDRAIN: printk("OPENDRAIN"); break;
	case MMC_BUSMODE_PUSHPULL: printk("PUSHPULL"); break;
	default: printk("Unknown"); break;
	}
	printk("\n");

	printk("Chip select: ");
	switch (ios->chip_select){
	case MMC_CS_DONTCARE: printk("DONTCARE"); break;
	case MMC_CS_HIGH: printk("HIGH"); break;
	case MMC_CS_LOW: printk("LOW"); break;
	default: printk("Unknown"); break;
	}
	printk("\n");

	printk("Power mode: ");
	switch (ios->power_mode) {
	case MMC_POWER_OFF: printk("OFF"); break;
	case MMC_POWER_UP: printk("UP"); break;
	case MMC_POWER_ON: printk("ON"); break;
	default: printk("Unknown"); break;
	}
	printk("\n");

	printk("Bus width: ");
	switch (ios->bus_width) {
	case MMC_BUS_WIDTH_1: printk("1"); break;
	case MMC_BUS_WIDTH_4: printk("4"); break;
	default: printk("Unknown"); break;
	}
	printk("\n");
#endif
#endif
}

static void sag_mmc_set_ios(struct mmc_host *mmc, struct mmc_ios *ios)
{
	struct sag_mmc_host	*host;

	PR_FUNC;

	host = mmc_priv(mmc);

	if (ios->clock)
		host->spi_dev->baud = qspi_dev_BAUD(ios->clock);

	print_ios(ios);
}

static struct mmc_host_ops sag_mmc_ops = {
	.request	= sag_mmc_request,
	.get_ro		= sag_mmc_get_ro,
	.set_ios	= sag_mmc_set_ios,
};

static int sag_mmc_probe(struct device *dev)
{
	struct mmc_host *mmc;
	struct sag_mmc_host *host = NULL;
	int ret = 0;

	PR_FUNC;

	/*
	 * Alloc a MMC Host
	 */
	mmc = mmc_alloc_host(sizeof(struct sag_mmc_host), dev);
	if (!mmc) {
		ret = -ENOMEM;
		goto out;
	}

	mmc->ops = &sag_mmc_ops;
	mmc->f_min = 125000 * 16; // 2 MHz
#if defined(CONFIG_COLDFIRE)
	#if defined(CONFIG_M532x) || defined(CONFIG_M5249) || defined(CONFIG_M527x)
	mmc->f_max =  MCF_BUSCLK/4;
	#else
	mmc->f_max =  MCF_CLK/4;
	#endif
#else
	mmc->f_max = CONFIG_SAG_MMC_SPI_MAX_HZ;
#endif

	host = mmc_priv(mmc);
	if(!host) {
		ret = -EIO;
		goto out_free_host;
	}

	host->spi_dev = qspi_dev_open(CONFIG_SAG_MMC_SPI_CS_CHAN);
	if (!host->spi_dev) {
		ret = -ENOMEM;
		goto out_free_host;
	}

	host->spi_dev->poll_mod = 0; /* use polling mode */
	host->spi_dev->dohie = 0;
	host->spi_dev->bits = TRANS_BITS;    /* 8 bits transfer */
	host->spi_dev->cpol = 1;
	host->spi_dev->cpha = 1;

	host->mmc = mmc;

	host->flags = HOST_IDLE;

	spin_lock_init( &(host->lock) );

	/*
	 * Host supports only SPI mode
	 */
	mmc->caps = MMC_CAP_SPI;

	/*
	 * ASSUMING 3.2-3.4 V slot power
	 */
	mmc->ocr_avail = MMC_VDD_32_33|MMC_VDD_33_34;

	ret = mmc_add_host(mmc, 0);
	if (ret != 0)
		goto fail_add_host;

	dev_set_drvdata(dev, mmc);

	/*
	 * Init timer
	 */
	init_timer(&host->timer);
	host->timer.function = sag_mmc_poll_event;
	host->timer.data = (unsigned long) host;

	// the first run is directly driven
	host->timer.expires = jiffies + 1;
	add_timer(&host->timer);

	printk(KERN_INFO DRIVER_NAME ": initialization completed\n");

	return 0;

fail_add_host:
	mmc_remove_host(mmc);
	dev_set_drvdata(dev, NULL);

out_free_host:
	mmc_free_host(mmc);

out:
	return ret;
}

static int sag_mmc_remove(struct device *dev)
{
	struct mmc_host* mmc = dev_get_drvdata(dev);
	struct sag_mmc_host* host;

	PR_FUNC;

	if (mmc) {
		host = mmc_priv(mmc);
		/*
		* Unregister host with MMC layer.
		*/
		mmc_remove_host(mmc);

		mmc_free_host(mmc);

		qspi_dev_release(host->spi_dev);

		mmc_dbg("MMC SPI: remove completed\n");
	}
	return 0;
}


#ifdef CONFIG_PM
static int sag_mmc_suspend(struct device *dev, u32 state, u32 level)
{
	struct mmc_host *mmc = dev_get_drvdata(dev);
	int ret = 0;

	PR_FUNC;

	if (mmc && level == SUSPEND_DISABLE)
		ret = mmc_suspend_host(mmc, state);

	return ret;
}

static int sag_mmc_resume(struct device *dev, u32 level)
{
	struct mmc_host *mmc = dev_get_drvdata(dev);
	int ret = 0;

	PR_FUNC;

	if (mmc && level == RESUME_ENABLE)
		ret = mmc_resume_host(mmc);

	return ret;
}
#else
#define sag_mmc_suspend	NULL
#define sag_mmc_resume	NULL
#endif

static void sag_mmc_release(struct device *dev)
{
	PR_FUNC;
}

static struct platform_device sag_mmc_device = {
	.name		= DRIVER_NAME,
	.id			= -1,
	.dev		= {
		.release = sag_mmc_release,
	},
};

static struct device_driver sag_mmc_driver = {
	.name		= DRIVER_NAME,
	.bus		= &platform_bus_type,
	.probe		= sag_mmc_probe,
	.remove		= sag_mmc_remove,
	.suspend	= sag_mmc_suspend,
	.resume		= sag_mmc_resume,
};

static int __init sag_mmc_init(void)
{
	int result;

	MCF_GPIO_PDDR_SSI &= ~(MCF_GPIO_PDDR_SSI_PDDR_SSI3 | MCF_GPIO_PDDR_SSI_PDDR_SSI4); /* set es input */
	MCF_GPIO_PDDR_SSI |= MCF_GPIO_PDDR_SSI_PDDR_SSI2; /* set es output for pm */

	sag_mmc_power_on();

	mmc_dbg(KERN_INFO DRIVER_NAME ": SD/MMC card interface driver, "
		DRIVER_VERSION "\n");
	mmc_dbg(KERN_INFO DRIVER_NAME ": Copyright (C) Sartorius Mechatronics\n");

	result = driver_register(&sag_mmc_driver);
	if (result < 0) {
		printk (KERN_ERR DRIVER_NAME ": error registering driver register: %d\n", result);
		return result;
	}

	result = platform_device_register(&sag_mmc_device);
	if (result < 0) {
		printk (KERN_ERR DRIVER_NAME ": error registering platform device register: %d\n", result);
		return result;
	}

	printk ("Sartorius SD/MMC card interface driver v"DRIVER_VERSION" installed\n");

	return 0;
}

static void __exit sag_mmc_exit(void)
{
	platform_device_unregister(&sag_mmc_device);
	driver_unregister(&sag_mmc_driver);
	sag_mmc_power_off();
}

module_init(sag_mmc_init);
module_exit(sag_mmc_exit);

MODULE_AUTHOR("Jakob Schwabauer <jakob.schwabauer@sartorius.com>");
MODULE_DESCRIPTION("SD/MMC Card Interface Driver over SPI");
MODULE_LICENSE("GPL");
