/*
 * Flash memory access on sartorius m5282lite
 *
 * $Id: sag_flash.c,v 1.14 2006/10/26 07:49:06 dec3ban Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/major.h>
#include <linux/root_dev.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include "sag_flash.h"

#if defined(CONFIG_SAG_PR5800_CORE) ||defined(CONFIG_SAG_PR5410_CORE)	|| defined(CONFIG_SAG_PR5220_CORE)
#define SAG_FLASH_PHYS_ADDR   0xF0000000
#define SAG_FLASH_SIZE        0x800000
#define SAG_FLASH_PAGE_SIZE   0x10000
#endif
#if defined(CONFIG_SAG_COBRA_CORE)
#define SAG_FLASH_PHYS_ADDR   0xC0000000
#define SAG_FLASH_SIZE        0x1000000  /* 16MB */
#define SAG_FLASH_PAGE_SIZE   0x10000
#endif
#if defined(CONFIG_SAG_LABPRO_CORE)
#define SAG_FLASH_PHYS_ADDR   0xC0000000
#define SAG_FLASH_SIZE        0x2000000  /* 32MB */
#define SAG_FLASH_PAGE_SIZE   0x20000
#endif

#if defined(CONFIG_SAG_JFFS2_SUPPORT)
#define SAG_MTD_WRITE_ACCESS
#define SAG_FLASHFS_SIZE    CONFIG_SAG_JFFS2_FLASHFS_SIZE
#define SAG_FLASHFS_BASE    (SAG_FLASH_PHYS_ADDR + SAG_FLASH_SIZE - SAG_FLASHFS_SIZE)
#endif /* CONFIG_JFFS2_SUPPORT */

#if defined(CONFIG_SAG_RAMDISK)
static struct mtd_info *sram_mtd;
#endif
static struct mtd_info *flash_mtd;

#if defined(CONFIG_RAM14MB) && defined(CONFIG_SAG_RAMDISK)
struct map_info combix_sram_map = {
	.name = "SDRAM",
	.size = 0x200000,
	.bankwidth = 4,
	.phys = 0x00E00000,
};
#endif
#if defined(CONFIG_RAM12MB)
struct map_info combix_sram_map = {
	.name = "SDRAM",
	.size = 0x400000,
	.bankwidth = 4,
	.phys = 0x00C00000,
};
#endif

#if defined(CONFIG_SAG_FLASHDISK)
struct map_info combix_flash_map = {
		.name =		"FLASH",
		.size =		SAG_FLASH_SIZE,
		.phys = 	SAG_FLASH_PHYS_ADDR,
		.bankwidth =	2,
};

struct mtd_partition flash_parts[] = {
	{
		.name =		"m0",
		.offset =	0,//FLASH_PARTITION1_ADDR,
		.size =		0,//FLASH_PARTITION1_SIZE
	},
	{
		.name =		"m1",
		.offset =	0,//FLASH_PARTITION1_ADDR,
		.size =		0,//FLASH_PARTITION1_SIZE
	},
	{
		.name =		"m2",
		.offset =	0,//FLASH_PARTITION2_ADDR,
		.size =		0,//FLASH_PARTITION2_SIZE
	},
	{
		.name =		"m3",
		.offset =	0,//FLASH_PARTITION3_ADDR,
		.size =		0,//FLASH_PARTITION3_SIZE
	},
	{
		.name =		"m4",
		.offset =	0,//FLASH_PARTITION3_ADDR,
		.size =		0,//FLASH_PARTITION3_SIZE
	},
	{
		.name =		"m5",
		.offset =	0,//FLASH_PARTITION3_ADDR,
		.size =		0,//FLASH_PARTITION3_SIZE
	}
};
#endif

/****************************************************************************
Flash info for COBRA5329:
-------------------------
Primary Vendor Command Set: 0002 (AMD/Fujitsu Standard)
Primary Algorithm Table at 0040
Alternative Vendor Command Set: 0000 (None)
No Alternate Algorithm Table
Vcc Minimum:  2.7 V
Vcc Maximum:  3.6 V
No Vpp line
Typical byte/word write timeout: 128 s
Maximum byte/word write timeout: 256 s
Typical full buffer write timeout: 128 s
Maximum full buffer write timeout: 4096 s
Typical block erase timeout: 1024 ms
Maximum block erase timeout: 16384 ms
Chip erase not supported
Device size: 0x1000000 bytes (16 MiB)
Flash Device Interface description: 0x0002
  - supports x8 and x16 via BYTE# with asynchronous interface
Max. bytes in buffer write: 0x20
Number of Erase Block Regions: 1
  Erase Region #0: BlockSize 0x10000 bytes, 256 blocks
FLASH: Found 1 x16 devices at 0x0 in 16-bit bank
 Amd/Fujitsu Extended Query Table at 0x0040
FLASH: CFI does not contain boot bank location. Assuming top.
number of CFI chips: 1
cfi_cmdset_0002: Disabling erase-suspend-program due to code brokenness.
****************************************************************************/
#if defined(SAG_MTD_WRITE_ACCESS)
const LONG_REG MCF_ACR0_FLASH_RW =
	( (MCF_ACRX_ADDR_BASE_BITS & SAG_FLASHFS_BASE)
	| (MCF_ACRX_ADDR_MASK_BITS & (SAG_FLASHFS_SIZE >> MCF_ACRX_ADDR_MASK_SHIFT))
	| MCF_ACRX_ENABLE | MCF_ACRX_SUP_MODE_BOTH1 | MCF_ACRX_CACH_PRECISE);		// 0xC000C040

static void remove_flash_write_protection(void)
{
	/* NOTE: flash wont work with enabled cache */
	asm ("movec %0,#0004; nop": :"a"(MCF_ACR0_FLASH_RW));
	/* Remove the writeprotection dedicated to the flash memory */
	MCF_CSM.cs0.mr &= ~BAM_WP;
	MCF_CSM.cs1.mr &= ~BAM_WP;
}
#endif

int flash_point(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char **mtdbuf)
{
	struct map_info *map = (struct map_info *) mtd->priv;
	*mtdbuf = (u_char *) (map->virt + ((int) from));
	*retlen = len;
	return(0);
}

static int  __init sag_add_partition(int partcount,u32 base)
{
#if defined(CONFIG_SAG_FLASHDISK)
	if(memcmp((void*)base,"-rom1fs-",8)==0)
	{
		u32 offs=(u32)(base-SAG_FLASH_PHYS_ADDR);
		u32 size=*(u32*)(base+8);
		// some sanity checks
		if(size>SAG_FLASH_SIZE) return 0;
		if(offs+size>SAG_FLASH_SIZE) return 0;
		size=(size+1023)&~1023;
		flash_parts[partcount].offset = offs;
		flash_parts[partcount].size = size;
		flash_parts[partcount].name =(char*)(base+16);
		// JJvB this is quite confusing, but we have to set the buts, which are NOT to be used
		flash_parts[partcount].mask_flags = MTD_WRITEABLE;	/* This partition is NOT writable */

		printk("mtd%d %08X %08X %s\n",partcount,offs,size,
			flash_parts[partcount].name);
		return size;
	}
#endif
	return 0;
}

static int __init init_disk (void)
{
#if defined(CONFIG_SAG_RAMDISK)
	int err;
#endif
#if defined(CONFIG_SAG_FLASHDISK)
	int partcount=0;
	int bootdev=-1;
	u32 base=SAG_FLASH_PHYS_ADDR;
	//------------------------------
	// add bios
	//------------------------------	
	flash_parts[partcount].name = (char*)(base+0x800);
	flash_parts[partcount].offset = base - SAG_FLASH_PHYS_ADDR;
	flash_parts[partcount].size = CONFIG_SAG_ROMFS_OFFSET;
	flash_parts[partcount].mask_flags = MTD_WRITEABLE;	/* This partition is NOT writable */
	++partcount;
	base+=CONFIG_SAG_ROMFS_OFFSET;
	
	for(;;)
	{
		long size=sag_add_partition(partcount,base);
		if(size==0) {
#if defined(SAG_MTD_WRITE_ACCESS)
			if(partcount < sizeof(flash_parts)/sizeof(*flash_parts) - 1) {

				/*
				** the JFFS2 partitition is at the very end of the flash
				*/

				flash_parts[partcount].name = "JFFS2 data pool";
				flash_parts[partcount].offset = SAG_FLASHFS_BASE - SAG_FLASH_PHYS_ADDR;
				flash_parts[partcount].size = SAG_FLASHFS_SIZE;
				++partcount;

				/*
				** the ALIBI-Partition starts at half the flash and goes to the
				** beginning of theJFFS2 partitition
				*/
				long half_the_flash=SAG_FLASH_SIZE/2;
				flash_parts[partcount].name = "ALIBI-BLOCK";
				flash_parts[partcount].offset = half_the_flash;
				flash_parts[partcount].size = half_the_flash-SAG_FLASHFS_SIZE;
				++partcount;

				/*
				** the Audit-Trail is read only and the last block before ALIBI
				*/
				flash_parts[partcount].name = "AUDIT-TRAIL";
				flash_parts[partcount].offset = half_the_flash-SAG_FLASH_PAGE_SIZE;
				flash_parts[partcount].size = SAG_FLASH_PAGE_SIZE;
				flash_parts[partcount].mask_flags = MTD_WRITEABLE;	/* This partition is NOT writable */
				++partcount;

			}
#endif
			break;
		}
		if(bootdev==-1) bootdev=partcount;
		++partcount;
		// filled partition table
		if(partcount>=sizeof(flash_parts)/sizeof(*flash_parts))
			break;
		size+=16;			// walk behind MD5
		size=(size+SAG_FLASH_PAGE_SIZE-1)&~(SAG_FLASH_PAGE_SIZE-1);	// round to next sector
		base+=size;
	}
	printk("bootdev=%d\n",bootdev);

	//---------------------------------------
	// flash disk init
	//---------------------------------------
	printk(KERN_NOTICE "flash device: %x at %x\n",
			SAG_FLASH_SIZE, SAG_FLASH_PHYS_ADDR);

	combix_flash_map.phys = SAG_FLASH_PHYS_ADDR;
	combix_flash_map.virt = ioremap(SAG_FLASH_PHYS_ADDR,SAG_FLASH_SIZE);

	if (!combix_flash_map.virt) {
		printk("Failed to ioremap\n");
		return -EIO;
	}

	simple_map_init(&combix_flash_map);

	/* use map_rom module for a read only fs in flash */
#if defined(SAG_MTD_WRITE_ACCESS)
	remove_flash_write_protection();
	flash_mtd = do_map_probe("cfi_probe", &combix_flash_map);
#else /* ! SAG_MTD_WRITE_ACCESS*/
	flash_mtd = do_map_probe("map_rom", &combix_flash_map);
#endif /* SAG_MTD_WRITE_ACCESS*/
	if (flash_mtd) {

		flash_mtd->owner = THIS_MODULE;
		flash_mtd->point = flash_point;
		add_mtd_partitions(flash_mtd, flash_parts, partcount);
		printk(KERN_NOTICE "flash device initialized\n");
		printk("uclinux[mtd]: set %s to be root filesystem\n",
	     		flash_parts[bootdev].name);
		ROOT_DEV = MKDEV(MTD_BLOCK_MAJOR, bootdev);
	}
	else
		iounmap((void *)combix_flash_map.virt);
#endif
#if defined(CONFIG_SAG_RAMDISK)
	// cpu ram
	combix_sram_map.virt = ioremap_nocache(combix_sram_map.phys, combix_sram_map.size);
	if (!combix_sram_map.virt) {
		printk("Failed to ioremap RAM space\n");
		err = -EIO;
		goto out;
	}
	simple_map_init(&combix_sram_map);

	sram_mtd = do_map_probe("map_ram", &combix_sram_map);
	if (!sram_mtd) {
		printk("RAM probe failed\n");
		err = -ENXIO;
		goto out_ioremap;
	}

	sram_mtd->owner = THIS_MODULE;
	sram_mtd->erasesize = 16;

	if (add_mtd_device(sram_mtd)) {
		printk("DRAM device addition failed\n");
		err = -ENOMEM;
		goto out_probe;
	}
	printk("DRAM device size with %ldKB registered \n",combix_sram_map.size/1024);
#endif
	return 0;

#if defined(CONFIG_SAG_RAMDISK)
out_probe:
	map_destroy(sram_mtd);
	sram_mtd = 0;

out_ioremap:
	iounmap((void *)combix_sram_map.virt);
out:
	return err;
#endif
}

static void __exit cleanup_disk_maps(void)
{
#if defined(CONFIG_SAG_RAMDISK)
	if (sram_mtd) {
		del_mtd_device(sram_mtd);
		map_destroy(sram_mtd);
		iounmap((void *)combix_sram_map.virt);
	}
#endif
#if defined(CONFIG_SAG_FLASHDISK)
	if (flash_mtd) {
		del_mtd_partitions(flash_mtd);
		map_destroy(flash_mtd);
	}
	if (combix_flash_map.virt) {
		iounmap((void *)combix_flash_map.virt);
		combix_flash_map.virt = 0;
	}
#endif
}

module_init(init_disk);
module_exit(cleanup_disk_maps);

MODULE_AUTHOR("Andreas Horn  <andreas.horn@sartorius.com>");
MODULE_DESCRIPTION("RAM and flash map driver");
MODULE_LICENSE("GPL");
