/* linux/drivers/mtd/nand/sag_nandflash.c
 *
 * Copyright (c) 2005 Sartorius Hamburg GmbH
 * 
 *
 * Combix nandflash driver for 64Mbit Hynix flash
 *
 * $Id: sag_nandflash.c,v 1.3 2005/06/08 11:25:21 dec3hon Exp $
 * $Log: sag_nandflash.c,v $
 * Revision 1.3  2005/06/08 11:25:21  dec3hon
 * *** empty log message ***
 *
 * Revision 1.2  2005/02/17 06:22:12  dec3hon
 * correction for 68mb chips
 *
 * Revision 1.1  2005/02/15 08:18:41  dec3hon
 * Configure.txt
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/err.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>

#include <asm/coldfire.h>
#include <asm/m527xsim.h>

static unsigned long combix_io_base = 0x80020000;
static struct mtd_info *combix_mtd = NULL;

struct s_nandchip
{
 volatile u_char dat;
 volatile u_char cmd;
 volatile u_char adr;
};

/*
 * Define partitions for flash devices
 */

static struct mtd_partition partition_combix64M[] = {
	{ .name =	"Combix partition 1",
	  .offset =	0,
	  .size	=	64 * SZ_1M },
};

static struct mtd_partition partition_combix128M[] = {
	{ 
		.name =	"Combix partition 1",
		.offset =	0,
		.size	=	64 * SZ_1M ,
		.mask_flags = MTD_WRITEABLE  /* force read-only */
	},
	{ 
		.name = "Combix partition 2",
		.offset =  MTDPART_OFS_APPEND,
 		.size =    MTDPART_SIZ_FULL,
	  	.mask_flags = MTD_WRITEABLE  /* force read-only */
	}
};

#define NUM_PARTITIONS64M 1
#define NUM_PARTITIONS128M 2

static void combix_hwcontrol(struct mtd_info *mtd, int cmd)
{

	/*Since the
 	* hardware generates the correct ALE/CLE signaling automatically, we
 	* do not need to use hwcontrol.	*/
	switch(cmd){

		case NAND_CTL_SETCLE: 
		case NAND_CTL_CLRCLE: 

		case NAND_CTL_SETALE: 
		case NAND_CTL_CLRALE: 

		case NAND_CTL_SETNCE: 
		case NAND_CTL_CLRNCE: break;
	}
}

/* return status of busy signal */
static int combix_ioready(struct mtd_info *mtd)
{
	return (*(volatile unsigned char *) (MCF_MBAR+MCF_GPIO_PPDSDR_BUSCTL) &1);
}

u_char combix_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_R;
	return chip->dat;
}

static void combix_write_byte(struct mtd_info *mtd, u_char byte)
{
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_W;
	chip->dat = byte;
}

static void combix_write_buf(struct mtd_info *mtd, const u_char *buf, int len)
{
	int i;
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_W;

	for (i=0; i<len; i++)
		chip->dat = buf[i];
}

static void combix_read_buf(struct mtd_info *mtd, u_char *buf, int len)
{
	int i;
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_R;

	for (i=0; i<len; i++)
		buf[i] = chip->dat;
}

static int combix_verify_buf(struct mtd_info *mtd, const u_char *buf, int len)
{
	int i;
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_R;

	for (i=0; i<len; i++)
		if (buf[i] != chip->dat)
			return -EFAULT;

	return 0;
}

static void combix_nand_command_lp (struct mtd_info *mtd, unsigned command, int column, int page_addr)
{
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_W;

	/* Emulate NAND_CMD_READOOB */
	if (command == NAND_CMD_READOOB) {
		column += mtd->oobblock;
		command = NAND_CMD_READ0;
	}
	
		
	/* Write out the command to the device. */
	chip->cmd = command;

	if (column != -1 || page_addr != -1) {
		this->hwcontrol(mtd, NAND_CTL_SETALE);

		/* Serially input address */
		if (column != -1) {
			/* Adjust columns for 16 bit buswidth */
			if (this->options & NAND_BUSWIDTH_16)
				column >>= 1;
			chip->adr = column & 0xff;
			chip->adr = column >> 8;
		}	
		if (page_addr != -1) {
			chip->adr = (unsigned char) (page_addr & 0xff);
			chip->adr = (unsigned char) ((page_addr >> 8) & 0xff);
			/* One more address cycle for devices > 128MiB */
			if (this->chipsize > (128 << 20))
				chip->adr = (unsigned char) ((page_addr >> 16) & 0xff);
		}
	}
	
	/* 
	 * program and erase have their own busy handlers 
	 * status and sequential in needs no delay
	*/
	switch (command) {
			
	case NAND_CMD_CACHEDPROG:
	case NAND_CMD_PAGEPROG:
	case NAND_CMD_ERASE1:
	case NAND_CMD_ERASE2:
	case NAND_CMD_SEQIN:
	case NAND_CMD_STATUS:
		return;


	case NAND_CMD_RESET:
		if (this->dev_ready)	
			break;
		udelay(this->chip_delay);
		chip->cmd = NAND_CMD_STATUS;
		while ( !(this->read_byte(mtd) & 0x40));
		return;

	case NAND_CMD_READ0:
		/* Write out the start read command */
		chip->cmd = NAND_CMD_READSTART;
		/* Fall through into ready check */
		
	/* This applies to read commands */	
	default:
		/* 
		 * If we don't have access to the busy pin, we apply the given
		 * command delay
		*/
		if (!this->dev_ready) {
			udelay (this->chip_delay);
			return;
		}	
	}
	
	/* Apply this short delay always to ensure that we do wait tWB in
	 * any case on any machine. */
	ndelay (100);
	/* wait until command is processed */
	while (!this->dev_ready(mtd));
}

/* combix_nand_command
 *
 * This function implements sending commands and the relevant address
 * information to the chip, via the hardware controller. Since the
 * combix board generates the correct ALE/CLE signaling automatically, we
 * do not need to use hwcontrol.
*/

static void combix_nand_command (struct mtd_info *mtd, unsigned command,
				  int column, int page_addr)
{
	struct nand_chip *this = mtd->priv;
	struct s_nandchip *chip = this->IO_ADDR_W;

	if (mtd->oobblock > 512)
//		this->cmdfunc = combix_nand_command_lp;
		return combix_nand_command_lp(mtd,command,column,page_addr);
	/*
	 * Write out the command to the device.
	 */
	if (command == NAND_CMD_SEQIN) {
		int readcmd;

		if (column >= mtd->oobblock) {
			/* OOB area */
			column -= mtd->oobblock;
			readcmd = NAND_CMD_READOOB;
		} else if (column < 256) {
			/* First 256 bytes --> READ0 */
			readcmd = NAND_CMD_READ0;
		} else {
			column -= 256;
			readcmd = NAND_CMD_READ1;
		}
		
		chip->cmd = readcmd;
	}
	chip->cmd = command;

	/* Set ALE and clear CLE to start address cycle */

	if (column != -1 || page_addr != -1) {

		/* Serially input address */
		if (column != -1) {
			/* Adjust columns for 16 bit buswidth */
			if (this->options & NAND_BUSWIDTH_16)
				column >>= 1;
			chip->adr = column;
		}
		if (page_addr != -1) {
			 chip->adr = (unsigned char) (page_addr);
			 chip->adr = ((unsigned char) (page_addr >> 8));
			/* One more address cycle for higher density devices */
			if (this->chipsize & 0x0c000000) 
				chip->adr = (unsigned char) ((page_addr >> 16) & 0x0f);
		}
		/* Latch in address */
	}
	
	/* 
	 * program and erase have their own busy handlers 
	 * status and sequential in needs no delay
	*/
	switch (command) {
			
	case NAND_CMD_PAGEPROG:
	case NAND_CMD_ERASE1:
	case NAND_CMD_ERASE2:
	case NAND_CMD_SEQIN:
	case NAND_CMD_STATUS:
		return;

	case NAND_CMD_RESET:
		if (this->dev_ready(mtd))	
			break;

		udelay(this->chip_delay);
		chip->cmd = NAND_CMD_STATUS;

		while ( !(this->read_byte(mtd) & 0x40));
		return;

	/* This applies to read commands */	
	default:
		/* 
		 * If we don't have access to the busy pin, we apply the given
		 * command delay
		*/
		if (!this->dev_ready(mtd)) {
			udelay (this->chip_delay);
			return;
		}	
	}
	
	/* Apply this short delay always to ensure that we do wait tWB in
	 * any case on any machine. */
	ndelay (100);
	/* wait until command is processed */
	while (!this->dev_ready(mtd));
}

/*
 * Main initialization routine
 */
int __init combix_nand_init (void)
{
	struct nand_chip *this;
	int err = 0;

	/* Allocate memory for MTD device structure and private data */
	combix_mtd = kmalloc (sizeof(struct mtd_info) + sizeof (struct nand_chip),
				GFP_KERNEL);
	if (!combix_mtd) {
		printk (KERN_WARNING "Unable to allocate combix NAND MTD device structure.\n");
		err = -ENOMEM;
		goto out;
	}

	/* Get pointer to private data */
	this = (struct nand_chip *) (&combix_mtd[1]);

	/* Initialize structures */
	memset((char *) combix_mtd, 0, sizeof(struct mtd_info));
	memset((char *) this, 0, sizeof(struct nand_chip));

	/* Link the private data with the MTD structure */
	combix_mtd->priv = this;

	/* Set address of NAND IO lines */
	this->IO_ADDR_R = (void*)combix_io_base;
	this->IO_ADDR_W = (void*)combix_io_base;
	this->read_byte = combix_read_byte;
	this->write_byte = combix_write_byte;
	this->write_buf = combix_write_buf;
	this->verify_buf = combix_verify_buf;
	this->read_buf = combix_read_buf;
	this->cmdfunc      = combix_nand_command;
	this->hwcontrol = combix_hwcontrol;
	this->dev_ready = combix_ioready;
	/* 20 us command delay time */
	this->chip_delay = 20;		
	this->eccmode = NAND_ECC_SOFT;

        /* Scan to find existance of the device */
	if (nand_scan (combix_mtd, 1)) {
		err = -ENXIO;
		goto out_mtd;
	}


	/* Register the partitions */
	switch(combix_mtd->size){
		case SZ_64M: add_mtd_partitions(combix_mtd, partition_combix64M, NUM_PARTITIONS64M); break; 
		case SZ_128M: add_mtd_partitions(combix_mtd, partition_combix128M, NUM_PARTITIONS128M); break; 
		default: {
			printk (KERN_WARNING "Unsupported Nand device\n"); 
			err = -ENXIO;
			goto out_buf;
		}
	}

	return err;
    
out_buf:
	kfree (this->data_buf);    
out_mtd:
	kfree (combix_mtd);
out:
	return err;
}


/*
 * Clean up routine
 */
static void __exit combix_nand_cleanup (void)
{
	/* Release resources, unregister device */
	nand_release (combix_mtd);

	/* Free the MTD device structure */
	kfree (combix_mtd);

}
module_exit(combix_nand_cleanup);
module_init(combix_nand_init);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Andreas Horn <andreas.horn@sartorius.com>");
MODULE_DESCRIPTION("Glue layer for NAND flash on combix board");

